﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * :include nw/snd/snd_FxMultiChReverb.h
 *
 * @file snd_FxMultiChReverb.h
 */

#ifndef NW_SND_FX_MULTICH_REVERB_H_
#define NW_SND_FX_MULTICH_REVERB_H_

#include <nw/types.h>

#include <nw/snd/snd_FxParam.h>
#include <nw/snd/snd_FxBase.h>
#include <nw/snd/snd_AxfxImpl.h>

#if defined(NW_SND_CONFIG_ENABLE_MULTICHEFT)

namespace nw {
namespace snd {


//---------------------------------------------------------------------------
//! @brief  指定されたチャンネル数に対するリバーブエフェクトです。
//!
//!         FxMultiChReverb は SDK の AXFXMultiChReverb エフェクトをラッピングしたクラスです。
//!
//! @see FxBase クラス
//!
//! @date 2013/04/05 初版
//---------------------------------------------------------------------------
class FxMultiChReverb : public FxBase
{
    /* ------------------------------------------------------------------------
            constant definition
       ------------------------------------------------------------------------ */
public:
    static const u32 EARLY_MODE_MIN;         //!< MultiChReverbParam::earlyMode の最小値 (1) です。
    static const u32 EARLY_MODE_MAX;         //!< MultiChReverbParam::earlyMode の最大値 (5) です。
    static const u32 LATE_MODE_MIN;          //!< MultiChReverbParam::lateMode の最小値 (1) です。
    static const u32 LATE_MODE_MAX;          //!< MultiChReverbParam::lateMode の最大値 (5) です。
    static const f32 PRE_DELAY_TIME_MIN;     //!< MultiChReverbParam::preDelayTime の最小値 (0.0) です。
    static const f32 PRE_DELAY_TIME_MAX;     //!< MultiChReverbParam::preDelayTime の最大値 (100.0) です。
    static const f32 DECAY_TIME_MIN;         //!< MultiChReverbParam::decayTime の最小値 (0.1) です。
    static const f32 DECAY_TIME_MAX;         //!< MultiChReverbParam::decayTime の最大値 (20.0) です。
    static const f32 HF_DECAY_RATIO_MIN;     //!< MultiChReverbParam::hfDecayRatio の最小値 (0.1) です。
    static const f32 HF_DECAY_RATIO_MAX;     //!< MultiChReverbParam::hfDecayRatio の最大値 (1.0) です。
    static const f32 COLORATION_MIN;         //!< MultiChReverbParam::coloration の最小値 (0.0) です。
    static const f32 COLORATION_MAX;         //!< MultiChReverbParam::coloration の最大値 (1.0) です。
    static const f32 EARLY_GAIN_MIN;         //!< MultiChReverbParam::earlyGain の最小値 (0.0) です。
    static const f32 EARLY_GAIN_MAX;         //!< MultiChReverbParam::earlyGain の最大値 (1.0) です。
    static const f32 LATE_GAIN_MIN;          //!< MultiChReverbParam::lateGain の最小値 (0.0) です。
    static const f32 LATE_GAIN_MAX;          //!< MultiChReverbParam::lateGain の最大値 (1.0) です。
    static const f32 REVERB_GAIN_MIN;        //!< MultiChReverbParam::reverbGain の最小値 (0.0) です。
    static const f32 REVERB_GAIN_MAX;        //!< MultiChReverbParam::reverbGain の最大値 (1.0) です。

    /* ------------------------------------------------------------------------
            typename definition
       ------------------------------------------------------------------------ */
public:
    //---------------------------------------------------------------------------
    //! @brief  チャンネル指定版リバーブエフェクトパラメータの構造体です。
    //!
    //! @see SetParam
    //! @see GetParam
    //!
    //! @date 2013/04/05 初版
    //---------------------------------------------------------------------------
    struct MultiChReverbParam
    {
        //---------------------------------------------------------------------------
        //! @brief    初期反射音のモードです。
        //!           値の範囲は 0～4 です。
        //!           初期値は 0 です。
        //---------------------------------------------------------------------------
        u32 earlyMode;
        //---------------------------------------------------------------------------
        //! @brief    後期残響音のモードです。
        //!           値の範囲は 0～4 です。
        //!           初期値は 0 です。
        //---------------------------------------------------------------------------
        u32 lateMode;
        //---------------------------------------------------------------------------
        //! @brief    反響を開始するまでの時間です。
        //!           値の範囲は 0.0～100.0 (msec 単位) です。
        //!           初期値は 0.02 です。
        //---------------------------------------------------------------------------
        f32 preDelayTime;
        //---------------------------------------------------------------------------
        //! @brief    後期残響の減衰量が 60dB になるまでの時間です。
        //!           値の範囲は 0.1～20.0 (sec 単位) です。
        //!           初期値は 3.0 です。
        //---------------------------------------------------------------------------
        f32 decayTime;
        //---------------------------------------------------------------------------
        //! @brief    低周波数での decayTime に対する、高周波減衰での decayTime の比率を調整します。
        //!           値を小さくすると低周波成分が主流となり、高周波成分で急速な減衰が発生します。
        //!           大きくすると高周波成分が減衰せずに残るようになります。
        //!           値の範囲は 0.1～1.0 です。
        //!           初期値は 0.4 です。
        //---------------------------------------------------------------------------
        f32 hfDecayRatio;
        //---------------------------------------------------------------------------
        //! @brief    アルゴリズムの全パスフィルタの係数を調節し、ルームの壁の音響特性を
        //!           シミュレートします。 値が小さくなるほど残響音の密度が粗くなります。
        //!           値の範囲は 0.0～1.0 です。
        //!           初期値は 0.6 です。
        //---------------------------------------------------------------------------
        f32 coloration;
        //---------------------------------------------------------------------------
        //! @brief    初期反射音の出力ゲインです。
        //!           値の範囲は 0.0～1.0 です。
        //!           初期値は 0.0 です。
        //---------------------------------------------------------------------------
        f32 earlyGain;
        //---------------------------------------------------------------------------
        //! @brief    後期残響音の出力ゲインです。
        //!           値の範囲は 0.0～1.0 です。
        //!           初期値は 1.0 です。
        //---------------------------------------------------------------------------
        f32 lateGain;
        //---------------------------------------------------------------------------
        //! @brief    反射音の全体的な出力ゲインです。
        //!           値の範囲は 0.0～1.0 です。
        //!           初期値は 1.0 です。
        //---------------------------------------------------------------------------
        f32 reverbGain;

        //---------------------------------------------------------------------------
        //! @brief    コンストラクタです。
        //! @date 2013/04/05 初版
        //---------------------------------------------------------------------------
        MultiChReverbParam()
        : earlyMode   ( 0     ),
          lateMode    ( 0     ),
          preDelayTime( 0.02f ),
          decayTime   ( 3.0f  ),
          hfDecayRatio( 0.4f  ),
          coloration  ( 0.6f  ),
          earlyGain   ( 0.0f  ),
          lateGain    ( 1.0f  ),
          reverbGain  ( 1.0f  )
        {}
    };

    /* ------------------------------------------------------------------------
            class member
       ------------------------------------------------------------------------ */
public:
    //! @name コンストラクタ/デストラクタ
    //@{

    //---------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //!
    //!         エフェクトパラメータは、下記のように初期化されます。
    //!
    //!         - preDelayTime = 0.02
    //!         - decayTime = 3.0
    //!         - coloration = 0.6
    //!         - hfDecayRatio = 0.4
    //!         - earlyMode = 1
    //!         - lateMode = 1
    //!         - earlyGain = 0.0
    //!         - reverbGain = 1.0
    //!         - lateGain = 1.0
    //!
    //! @date 2011/08/12 初版
    //---------------------------------------------------------------------------
    FxMultiChReverb();

    //---------------------------------------------------------------------------
    //! @brief    デストラクタです。
    //!
    //! @date 2013/04/05 初版
    //---------------------------------------------------------------------------
    virtual ~FxMultiChReverb() { Finalize(); ReleaseWorkBuffer(); }

    //@}

    //! @name メモリ割り当て
    //@{

    //---------------------------------------------------------------------------
    //! @brief    FxMultiChReverb が必要とするメモリのサイズを取得します。
    //!
    //!           FxMultiChReverb が現在のパラメータ設定でエフェクト処理を行うために
    //!           必要とするメモリのサイズを取得します。
    //!
    //!           この関数を呼び出す前に @ref SetChannelMode を呼び出してチャンネル数を
    //!           設定してください。呼び出さない場合は、初期設定の値が使用されます。
    //!
    //!           エフェクトを使用するためには、取得したメモリサイズのメモリを確保
    //!           した後に @ref AssignWorkBuffer を呼びだし、エフェクトにメモリ領域を
    //!           割り当ててください。
    //!
    //! @return   FxMultiChReverb が必要とするメモリのサイズを返します。
    //!
    //! @see  SetChannelMode
    //! @see  AssignWorkBuffer
    //!
    //! @date 2013/04/05 初版
    //---------------------------------------------------------------------------
    u32 GetRequiredMemSize();

    //---------------------------------------------------------------------------
    //! @brief    エフェクトで使用するバッファ領域を割り当てます。
    //!
    //!           バッファ領域の割り当てはエフェクトを使用する前に行う必要があります。
    //!
    //!           エフェクトが必要とするバッファサイズはパラメータの値によって変化しますので、
    //!           @ref GetRequiredMemSize を呼び出してバッファサイズを取得してください。
    //!
    //! @param[in] buffer   割り当てるバッファの先頭アドレス。
    //! @param[in] size     割り当てるバッファのサイズ。
    //!
    //! @return   バッファ領域の割り当てに成功した場合は true を、失敗した場合は false を返します。
    //!
    //! @see  GetRequiredMemSize
    //!
    //! @date 2013/04/05 初版
    //---------------------------------------------------------------------------
    virtual bool AssignWorkBuffer( void* buffer, u32 size );

    //---------------------------------------------------------------------------
    //! @brief  エフェクトで使用していたメモリを解放します。
    //!
    //!         @ref AssignWorkBuffer で割り当てたバッファ領域を解放します。 この関数で
    //!         解放を行う前に、@ref Finalize を呼び出してエフェクト処理を終了させる
    //!         必要があります。
    //!
    //! @see  AssignWorkBuffer
    //! @see  Finalize
    //!
    //! @date 2013/04/05 初版
    //---------------------------------------------------------------------------
    virtual void ReleaseWorkBuffer();

    //@}

    //! @name エフェクト
    //@{

    //---------------------------------------------------------------------------
    //! @brief    エフェクトの開始処理を行います。
    //!
    //!           @ref SoundSystem クラスを通じてエフェクトを使用する場合、
    //!           @ref SoundSystem クラスからこの関数が呼び出されます。 ユーザがこの関数を
    //!           呼び出す必要はありません。
    //!
    //!           この関数を呼び出す前に、@ref AssignWorkBuffer を呼びだし、エフェクトで
    //!           使用するワークメモリを割り当てる必要があります。
    //!
    //! @return   エフェクトの開始に成功したら true を、失敗したら false を返します。
    //!
    //! @see  SoundSystem クラス
    //! @see  AssignWorkBuffer
    //!
    //! @date 2013/04/05 初版
    //---------------------------------------------------------------------------
    virtual bool Initialize();

    //---------------------------------------------------------------------------
    //! @brief    エフェクトの終了処理を行います。
    //!
    //!           @ref SoundSystem クラスを通じてエフェクトを使用する場合、
    //!           @ref SoundSystem クラスからこの関数が呼び出されます。 ユーザがこの関数を
    //!           呼び出す必要はありません。
    //!
    //! @see  SoundSystem クラス
    //!
    //! @date 2013/04/05 初版
    //---------------------------------------------------------------------------
    virtual void Finalize();

    //---------------------------------------------------------------------------
    //! @brief    チャンネル数を設定します。
    //!
    //! @param[in] mode チャンネル数を指定する列挙型です。
    //!
    //! @return   エフェクト処理を開始する前は true を返しますが、
    //!           エフェクト処理中は false を返し、チャンネル数の設定に失敗します。
    //!
    //!           TV 出力に設定するエフェクトには全ての @ref ChannelMode を使用できますが、
    //!           DRC 出力に設定するエフェクトには 2ch, 4ch のみが使用できます。
    //!
    //!           DRC 出力に設定するエフェクトで 6ch を使用した場合、@ref SoundSystem::AppendEffect でエフェクトを追加する際に
    //!           Debug, Develop ビルドではアサートとなり、Release ビルドでは追加に失敗するためご注意ください。
    //!
    //! @see  GetRequiredMemSize
    //! @see  GetChannelMode
    //!
    //! @date 2013/07/31 初版
    //---------------------------------------------------------------------------
    bool SetChannelMode( ChannelMode mode )
    {
        if ( m_IsActive )
        {
            return false;
        }
        else
        {
            m_ChannelMode = mode;
            return true;
        }
    }

    //---------------------------------------------------------------------------
    //! @brief    チャンネル数を取得します。
    //!
    //! @return   現在のチャンネル数です。
    //!
    //! @see  SetChannelMode
    //!
    //! @date 2013/07/31 初版
    //---------------------------------------------------------------------------
    const ChannelMode GetChannelMode() const { return m_ChannelMode; }

    //---------------------------------------------------------------------------
    //! @brief    サンプリングレートを設定します。
    //!
    //! @param[in] rate サンプリングレートを指定する列挙型です。
    //!
    //! @return   エフェクト処理を開始する前は true を返しますが、
    //!           エフェクト処理中は false を返し、サンプリングレートの設定に失敗します。
    //!
    //! @see  GetRequiredMemSize
    //! @see  GetSampleRate
    //!
    //! @date 2013/12/04 初版
    //---------------------------------------------------------------------------
    bool SetSampleRate( SampleRate rate )
    {
        if ( m_IsActive )
        {
            return false;
        }
        else
        {
            m_SampleRate = rate;
            return true;
        }
    }

    //---------------------------------------------------------------------------
    //! @brief    サンプリングレートを取得します。
    //!
    //! @return   現在のサンプリングレートです。
    //!
    //! @see  SetSampleRate
    //!
    //! @date 2013/12/04 初版
    //---------------------------------------------------------------------------
    const SampleRate GetSampleRate() const { return m_SampleRate; }

    //---------------------------------------------------------------------------
    //! @brief    エフェクトパラメータを設定します。
    //!
    //!           エフェクトパラメータの設定はエフェクト動作中でも行うことができます。
    //!
    //! @param[in] param     エフェクトパラメータ。
    //!
    //! @return   エフェクト動作中は、パラメータ変更に成功したら true を、失敗したら false
    //!           を返します。 エフェクト処理を開始する前は常に true を返します。
    //!
    //! @see  GetParam
    //!
    //! @date 2013/04/05 初版
    //---------------------------------------------------------------------------
    bool SetParam( const MultiChReverbParam& param );

    //---------------------------------------------------------------------------
    //! @brief    エフェクトのパラメータを取得します。
    //!
    //! @return   現在のエフェクトパラメータの値です。
    //!
    //! @see  SetParam
    //!
    //! @date 2013/04/05 初版
    //---------------------------------------------------------------------------
    const MultiChReverbParam& GetParam() const { return m_Param; }

    // エフェクトコールバック
    virtual void UpdateBuffer(
        int numChannels,
        void* buffer[],
        unsigned long bufferSize,
        SampleFormat format,
        f32 sampleRate,
        OutputMode mode
    );

    //@}

private:
    //---------------------------------------------------------------------------
    //! @briefprivate
    //!
    //! @param[in] device チェック対象となるデバイスです。
    //!
    //! @return           チャンネル数が妥当であれば true を、妥当でなければ false を返します。
    //!
    //! @brief            デバイスに対してチャンネル数が妥当かをチェックする仮想関数です。
    //---------------------------------------------------------------------------
    virtual bool IsValidChannelNum( OutputDevice device );

#if defined( NW_PLATFORM_WIN32 ) || defined ( NW_USE_NINTENDO_SDK )
    typedef nw::internal::winext::AXFX_MULTI_CH_REVERB AXFX_MULTI_CH_REVERB;
    typedef nw::internal::winext::AXAUXCBSTRUCT        AXAUXCBSTRUCT;
#elif defined( NW_PLATFORM_ANDROID ) || defined( NW_PLATFORM_IOS )
    typedef nw::internal::winext::AXFX_MULTI_CH_REVERB AXFX_MULTI_CH_REVERB;
    typedef nw::internal::winext::AXAUXCBSTRUCT        AXAUXCBSTRUCT;
#endif // defined( NW_PLATFORM_WIN32 )

    bool m_IsActive;
    internal::AxfxImpl m_Impl;
    MultiChReverbParam m_Param;
    ChannelMode        m_ChannelMode;
    SampleRate         m_SampleRate;

    AXFX_MULTI_CH_REVERB m_AxfxParam;
    AXAUXCBSTRUCT        m_AxfxInfo;
};

} // namespace nw::snd
} // namespace nw

#endif /* NW_SND_CONFIG_ENABLE_MULTICHEFT */

#endif /* NW_SND_FX_MULTICH_REVERB_H_ */
