﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SND_CHANNEL_H_
#define NW_SND_CHANNEL_H_

#include <nw/snd/snd_Global.h>  // WaveInfo
#include <nw/snd/snd_MultiVoice.h>
#include <nw/snd/snd_MoveValue.h>
#include <nw/snd/snd_CurveAdshr.h>
#include <nw/snd/snd_CurveLfo.h>
#include <nw/snd/snd_DisposeCallback.h>
#include <nw/snd/snd_HardwareManager.h>
#include <nw/snd/snd_WaveFileReader.h>

namespace nw {
namespace snd {
namespace internal {
namespace driver {

/* ========================================================================
        class definition
   ======================================================================== */

class Channel
{
    /* ------------------------------------------------------------------------
            typename definition
       ------------------------------------------------------------------------ */
public:
    enum LfoTarget
    {
        LFO_TARGET_PITCH,
        LFO_TARGET_VOLUME,
        LFO_TARGET_PAN
    };

    enum ChannelCallbackStatus
    {
        CALLBACK_STATUS_STOPPED,
        CALLBACK_STATUS_DROP,
        CALLBACK_STATUS_FINISH,
        CALLBACK_STATUS_CANCEL
    };

    typedef void (*ChannelCallback) (
        Channel *channel,
        ChannelCallbackStatus status,
        void* userData );

    /* ------------------------------------------------------------------------
            constant declaration
       ------------------------------------------------------------------------ */
public:
    static const int CHANNEL_NUM            = HardwareManager::MAX_VOICE_COUNT + 1; // チャンネル数
    static const int CHANNEL_MIN            = 0;
    static const int CHANNEL_MAX            = CHANNEL_MIN + CHANNEL_NUM - 1;

    static const int PRIORITY_RELEASE       = 1;

private:
    static const int KEY_INIT               = 60;
    static const int ORIGINAL_KEY_INIT      = 60;
    static const u8  SILENCE_VOLUME_MAX     = 255;
    static const u8  SILENCE_VOLUME_MIN     = 0;

    static const f32 SILENCE_VOLUME_MAX_R;

    /* ------------------------------------------------------------------------
            static members
       ------------------------------------------------------------------------ */
public:
    static Channel* AllocChannel(
        int voiceChannelCount,
        int priority,
        ChannelCallback callback,
        void* callbackData,
        VoiceRendererType mode = VOICE_RENDERER_SDK
    );
    static void FreeChannel( Channel* channel );
    static void DetachChannel( Channel* channel );

private:
    static void VoiceCallbackFunc(
        MultiVoice* voice,
        MultiVoice::VoiceCallbackStatus status,
        void* arg
    );

    static const int WAVE_BUFFER_MAX = 2;

    /* ------------------------------------------------------------------------
            class members
       ------------------------------------------------------------------------ */
public:
    Channel();
    ~Channel();

    void Update( bool doPeriodicProc );
    void CallChannelCallback( ChannelCallbackStatus status );

    void SetOffsetContext(const WaveFileReader::OffsetContextInfo& contextInfo);
    void Start( const WaveInfo& waveParam, int length, u32 startOffsetSamples );
    void Stop();
    void Pause( bool flag ) { m_PauseFlag = flag; m_pVoice->Pause( flag ); }

    void NoteOff();
    void Release();

    bool IsActive() const { return m_ActiveFlag != 0; }
    bool IsPause() const { return m_PauseFlag != 0; }

    //------------------------------------------------------------------
    // 初期パラメータ
    void SetKey( u8 key ) { m_Key = key; }
    void SetKey( u8 key, u8 originalKey ) { m_Key = key; m_OriginalKey = originalKey; }
    void SetInitPan( f32 pan ) { m_InitPan = pan; }
    void SetInitSurroundPan( f32 surroundPan ) { m_InitSurroundPan = surroundPan; }
    void SetTune( f32 tune ) { m_Tune = tune; }

    void SetAttack( int attack ) { m_CurveAdshr.SetAttack( attack ); }
    void SetHold( int hold ) { m_CurveAdshr.SetHold( hold ); }
    void SetDecay( int decay ) { m_CurveAdshr.SetDecay( decay ); }
    void SetSustain( int sustain ) { m_CurveAdshr.SetSustain( sustain ); }
    void SetRelease( int release ) { m_CurveAdshr.SetRelease( release ); }

    void SetSilence( bool silenceFlag, int fadeTimes )
    {
        NW_MINMAX_ASSERT( fadeTimes, 0, USHRT_MAX );
        m_SilenceVolume.SetTarget(
            silenceFlag? SILENCE_VOLUME_MIN: SILENCE_VOLUME_MAX,
            static_cast<u16>( fadeTimes )
        );
    }

    void SetVoiceRendererType(VoiceRendererType mode) { m_RenderingMode = mode; }
    VoiceRendererType GetVoiceRendererType() const { return m_RenderingMode; }

    s32 GetLength() const { return m_Length; }
    void SetLength( s32 length ) { m_Length = length; }
    bool IsRelease() const { return m_CurveAdshr.GetStatus() == CurveAdshr::STATUS_RELEASE; }

    //------------------------------------------------------------------
    // ユーザーパラメータ
    void SetUserVolume( f32 volume ) { m_UserVolume = volume; }
    void SetUserPitch( f32 pitch ) { m_UserPitch = pitch; }
    void SetUserPitchRatio( f32 pitchRatio ) { m_UserPitchRatio = pitchRatio; }

    void SetUserLpfFreq( f32 lpfFreq ) { m_UserLpfFreq = lpfFreq; }
    void SetBiquadFilter( int type, f32 value );
    void SetLfoParam( const CurveLfoParam& param, int i = 0 ) { m_Lfo[i].SetParam( param ); }
    void SetLfoTarget( LfoTarget type, int i = 0 ) { m_LfoTarget[i] = static_cast<u8>(type); }
    void SetPriority( int priority ) { m_pVoice->SetPriority( priority ); }
    void SetReleasePriorityFix( bool fix ) { m_ReleasePriorityFixFlag = fix; }
    void SetIsIgnoreNoteOff( bool flag ) { m_IsIgnoreNoteOff = flag; }
    void SetFrontBypass( bool flag ) { m_pVoice->SetFrontBypass( flag ); }

    void SetSweepParam( f32 sweepPitch, int sweepTime, bool autoUpdate );
    bool IsAutoUpdateSweep() const { return m_AutoSweep != 0 ; }
    void UpdateSweep( int count );

    void SetPanMode( PanMode panMode ) { m_PanMode = panMode; }
    void SetPanCurve( PanCurve panCurve ) { m_PanCurve = panCurve; }
    void SetRemoteFilter( u8 filter ) { m_RemoteFilter = filter; }

    //------------------------------------------------------------------
    // 出力パラメータ
    void SetOutputLine( u32 lineFlag ) { m_OutputLineFlag = lineFlag; }
    u32 GetOutputLine() const { return m_OutputLineFlag; }


    void SetTvParam( const OutputParam& param ) { m_TvParam = param; }
    const OutputParam& GetTvParam() const { return m_TvParam; }

    void SetDrcParam( u32 drcIndex, const OutputParam& param )
    {
        NW_ASSERT_MAXLT( drcIndex, DRC_OUT_COUNT );
        m_DrcParam[drcIndex] = param;
    }
    const OutputParam& GetDrcParam( u32 drcIndex ) const
    {
        NW_ASSERT_MAXLT( drcIndex, DRC_OUT_COUNT );
        return m_DrcParam[drcIndex];
    }

    void SetRemoteParam( u32 remoteIndex, const RemoteOutputParam& param )
    {
        NW_ASSERT_MAXLT( remoteIndex, REMOTE_OUT_COUNT );
        m_RemoteParam[remoteIndex] = param;
    }
    const RemoteOutputParam& GetRemoteParam( u32 remoteIndex ) const
    {
        NW_ASSERT_MAXLT( remoteIndex, REMOTE_OUT_COUNT );
        return m_RemoteParam[remoteIndex];
    }

#ifdef NW_PLATFORM_CAFEWIN
    // SoundMaker 用
    void SetMainSend( f32 send ) { m_TvParam.mainSend = send; }
    void SetUserPan( f32 pan ) { m_TvParam.pan = pan; }
#endif


    //------------------------------------------------------------------
    Channel* GetNextTrackChannel() const { return m_pNextLink; }
    void SetNextTrackChannel( Channel* channel ) { m_pNextLink = channel; }

    u32 GetCurrentPlayingSample(bool isOriginalSamplePosition) const;

    void SetKeyGroupId( u8 id ) { m_KeyGroupId = id; }
    u8 GetKeyGroupId() const { return m_KeyGroupId; }

    void SetInterpolationType( u8 type ) { m_InterpolationType = type; }
    u8 GetInterpolationType() const { return m_InterpolationType; }

    void SetInstrumentVolume( f32 instrumentVolume ) { m_InstrumentVolume = instrumentVolume; }
    void SetVelocity( f32 velocity ) { m_Velocity = velocity; }

    static const int MOD_COUNT = 4;
private:
    class Disposer : public DisposeCallback
    {
      public:
        Disposer() : m_pChannel(NULL) {}
        void Initialize(Channel* channel)
        {
            m_pChannel = channel;
        }
        virtual ~Disposer() {}
        virtual void InvalidateData( const void* start, const void* end );
      private:
        Channel* m_pChannel;
    };
    friend class Disposer;

    f32 GetSweepValue() const;
    void InitParam( ChannelCallback callback, void* callbackData );
    void AppendWaveBuffer( const WaveInfo& waveInfo, u32 startOffsetSamples );

    Disposer m_Disposer;
    CurveAdshr m_CurveAdshr;

    CurveLfo m_Lfo[MOD_COUNT];
    u8 m_LfoTarget[MOD_COUNT]; // enum LfoTarget

    u8 m_PauseFlag;
    u8 m_ActiveFlag;
    u8 m_AllocFlag;
    u8 m_AutoSweep;
    u8 m_ReleasePriorityFixFlag;
    u8 m_IsIgnoreNoteOff;
    u8 m_BiquadType;
    u8 padding;

    f32 m_UserVolume;
    f32 m_UserPitchRatio;
    f32 m_UserLpfFreq;
    f32 m_BiquadValue;
    u32 m_OutputLineFlag;

    OutputParam m_TvParam;
    OutputParam m_DrcParam[DRC_OUT_COUNT];
    RemoteOutputParam m_RemoteParam[REMOTE_OUT_COUNT];

    f32 m_UserPitch;
    f32 m_SweepPitch;
    s32 m_SweepCounter;
    s32 m_SweepLength;

    f32 m_InitPan;
    f32 m_InitSurroundPan;
    f32 m_Tune;
    MoveValue<u8, u16> m_SilenceVolume;

    f32 m_Cent;
    f32 m_CentPitch;

    s32 m_Length;

    PanMode m_PanMode;
    PanCurve m_PanCurve;

    u8 m_Key;
    u8 m_OriginalKey;
    u8 m_KeyGroupId;
    u8 m_InterpolationType;

    f32 m_InstrumentVolume;
    f32 m_Velocity;

    ChannelCallback m_Callback;
    void* m_CallbackData;

    MultiVoice* m_pVoice;

    Channel* m_pNextLink;

    WaveBuffer m_WaveBuffer[WAVE_CHANNEL_MAX][WAVE_BUFFER_MAX];
    AdpcmContext m_AdpcmContext[WAVE_CHANNEL_MAX];
    AdpcmContext m_AdpcmLoopContext[WAVE_CHANNEL_MAX];
    VoiceRendererType m_RenderingMode;
    u32 m_StartOffsetSamples;
    u32 m_LoopStartFrame;
    u32 m_OriginalLoopStartFrame;
    WaveFileReader::OffsetContextInfo m_ContextInfo;
    bool m_UseContextInfo;
    bool m_LoopFlag;
    u8 m_RemoteFilter;
    u8 padding2;

public:
    ut::LinkListNode m_Link;
};

} // namespace nw::snd::internal::driver
} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw


#endif /* NW_SND_CHANNEL_H_ */

