﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SND_FND_STREAM_H_
#define NW_SND_FND_STREAM_H_

#include <nw/ut/ut_Inlines.h>
#include <nw/ut/ut_Preprocessor.h>
#include <nw/snd/fnd/basis/sndfnd_Config.h>
#include <nw/snd/fnd/basis/sndfnd_Result.h>

namespace nw {
namespace snd {
namespace internal {
namespace fnd {

//---------------------------------------------------------------------------
//! @brief  ストリームの基本クラスです。
//---------------------------------------------------------------------------
class Stream
{
    NW_DISALLOW_COPY_AND_ASSIGN(Stream);

public: // 定数の定義
    static const u32 INVALID_POSITION = UINT_MAX;
    static const u32 INVALID_SIZE = UINT_MAX;

public: // 型の定義
    enum SeekOrigin
    {
        SEEK_ORIGIN_BEGIN = 0,
        SEEK_ORIGIN_END,
        SEEK_ORIGIN_CURRENT
    };

public: // デストラクタ
    virtual ~Stream() { }

protected: // コンストラクタ
    Stream() { }

public: // メソッド
    virtual void Close() = 0;

    virtual bool IsOpened() const = 0;

    virtual u32 Read(void* /*buf*/, u32 /*length*/, FndResult* result = NULL)
    {
        NW_WARNING(false, "stream reading is not supported.\n");

        if(result != NULL)
        {
            *result = FndResult(SNDFND_RESULT_NOT_SUPPORTED);
        }

        return 0;
    }

    virtual u32 Write(const void* /*buf*/, u32 /*length*/, FndResult* result = NULL)
    {
        NW_WARNING(false, "stream writing is not supported.\n");

        if(result != NULL)
        {
            *result = FndResult(SNDFND_RESULT_NOT_SUPPORTED);
        }

        return 0;
    }

    virtual FndResult Seek(s32 /*offset*/, SeekOrigin /*origin*/)
    {
        NW_WARNING(false, "stream seeking is not supported.\n");

        return FndResult(SNDFND_RESULT_NOT_SUPPORTED);
    }

    virtual u32 GetCurrentPosition() const
    {
        return INVALID_POSITION;
    }

    virtual u32 GetSize() const
    {
        return INVALID_SIZE;
    }

    virtual bool CanRead() const { return false; }

    virtual bool CanWrite() const { return false; }

    virtual bool CanSeek() const { return false; }
};

//---------------------------------------------------------------------------
//! @brief  指定ストリームからの読み込みをサポートします。
//---------------------------------------------------------------------------
class StreamReader
{
    NW_DISALLOW_COPY_AND_ASSIGN(StreamReader);

public: // コンストラクタ
    explicit StreamReader(Stream* stream) :
      m_Stream(stream)
    {
        NW_ASSERT_NOT_NULL(stream);
    }

    virtual ~StreamReader() { }

public: // メソッド
    virtual u32 Read(void* buf, u32 length, FndResult* result = NULL)
    {
        return m_Stream->Read(buf, length, result);
    }

private: // メンバ変数
    Stream* m_Stream;
};

//---------------------------------------------------------------------------
//! @brief  指定ストリームからの読み込みをサポートします。
//---------------------------------------------------------------------------
class StreamWriter
{
    NW_DISALLOW_COPY_AND_ASSIGN(StreamWriter);

public: // コンストラクタ
    StreamWriter() :
      m_Stream(NULL),
      m_WriteFormatBuffer(NULL),
      m_WriteFormatBufferLength(0)
    {
    }

    explicit StreamWriter(Stream* stream) :
      m_Stream(stream),
      m_WriteFormatBuffer(NULL),
      m_WriteFormatBufferLength(0)
      {
          NW_ASSERT_NOT_NULL(stream);
      }

      virtual ~StreamWriter() { }

public: // メソッド
    FndResult Open(Stream* stream)
    {
        if(stream == NULL)
        {
            NW_ASSERT(false);
            return FndResult(SNDFND_RESULT_FAILED);
        }

        m_Stream = stream;
        return FndResult(SNDFND_RESULT_TRUE);
    }

    void Close()
    {
        m_Stream = NULL;
    }

    bool IsOpened() const { return m_Stream != NULL; }

    u32 Write(const void* buf, u32 length, FndResult* result = NULL)
    {
        NW_ASSERTMSG(IsOpened(), "StreamWriter must be opened.\n");
        NW_ASSERT_NOT_NULL(buf);
        return m_Stream->Write(buf, length, result);
    }

    FndResult Write(const char* text)
    {
        NW_ASSERT_NOT_NULL(text);
        return Write(text, std::strlen(text));
    }

    FndResult Write(const char* text, u32 length)
    {
        NW_ASSERT_NOT_NULL(text);
        NW_ASSERT(length > 0);

        FndResult result(SNDFND_RESULT_TRUE);
        Write(text, length, &result);
        return result;
    }

    FndResult WriteLine(const char* text)
    {
        NW_ASSERT_NOT_NULL(text);
        return WriteLine(text, std::strlen(text));
    }

    FndResult WriteLine(const char* text, u32 length);

    FndResult WriteFormat(const char* format, ...);
    FndResult VWriteFormat(const char* format, va_list vargs);

    void EnableWriteFormat(void* buffer, u32 length);
    void DisableWriteFormat();

private: // メンバ変数
    Stream* m_Stream;
    char*   m_WriteFormatBuffer;
    u32     m_WriteFormatBufferLength;
};

//---------------------------------------------------------------------------
//! @brief  ストリーム入力をアライメント調整します。
//---------------------------------------------------------------------------
class StreamAligner
{
    NW_DISALLOW_COPY_AND_ASSIGN(StreamAligner);

public: // コンストラクタ
    StreamAligner() :
      m_Stream(NULL),
      m_Alignment(0),
      m_BufferForAlignment(NULL),
      m_BufferLengthForAlignment(0)
      { }

    virtual ~StreamAligner() { }

public: // メソッド
    void Open(Stream* stream, u32 alignment, void* buf, u32 length);
    void Close();

    bool IsOpened() const { return m_Stream != NULL && m_Stream->IsOpened(); }

    u32 GetRequiredBufferLength(u32 alignment) const
    {
        return alignment * 2;
    }

    u32 Read(void* buf, u32 length, FndResult* result = NULL);

private: // メンバ変数
    Stream* m_Stream;

    u32   m_Alignment;
    void* m_BufferForAlignment;
    u32   m_BufferLengthForAlignment;
};

} // namespace nw::snd::internal::fnd
} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw

#endif // NW_SND_FND_STREAM_H_
