﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SND_FND_SIMPLERINGBUFFER_H_
#define NW_SND_FND_SIMPLERINGBUFFER_H_

#include <nw/types.h>
#if defined(NW_PLATFORM_CTR)
#include <nn/os/os_CriticalSection.h>
#else
#include <nw/ut/os/ut_Mutex.h>
#endif

namespace nw {
namespace snd {
namespace internal {
namespace fnd {

#if defined(NW_PLATFORM_CTR)
typedef nn::os::CriticalSection LockObject;
#else
typedef nw::ut::Mutex LockObject;
#endif

//---------------------------------------------------------
struct SimpleRingBufferRange
{
    u32                     start;
    u32                     end;

    SimpleRingBufferRange()
    {
        start = 0;
        end   = 0;
    }
};

//---------------------------------------------------------
struct SimpleRingBufferHeader
{
    u32                     bufferSize;
    SimpleRingBufferRange   range;
    LockObject              lockObject;

    SimpleRingBufferHeader()
    {
        bufferSize  = 0;
    }
};

//---------------------------------------------------------------------------
class SimpleRingBuffer
{
public:
    //---------------------------------------------------------------------------
    //! @brief        デフォルトコンストラクタです。
    //---------------------------------------------------------------------------
                            SimpleRingBuffer() {}

    //---------------------------------------------------------------------------
    //! @brief        インスタンスを初期化します。
    //---------------------------------------------------------------------------
    void                    Init(
                                void*   buf,
                                u32     size);

    //---------------------------------------------------------------------------
    //! @brief        現在バッファにあるデータを破棄します。
    //---------------------------------------------------------------------------
    void                    Discard();

    //---------------------------------------------------------------------------
    //! @brief        読み込み可能なデータサイズを取得します。
    //---------------------------------------------------------------------------
    u32                     GetReadableBytes();

    //---------------------------------------------------------------------------
    //! @brief        書き込み可能なデータサイズを取得します。
    //---------------------------------------------------------------------------
    u32                     GetWritableBytes();

    const SimpleRingBufferHeader& GetHeader() { return *m_pHeader; }

    //---------------------------------------------------------------------------
    //! @brief        リングバッファから読み出します。
    //---------------------------------------------------------------------------
    u32                     Read(
                                void*       buf,
                                u32         size);

    //---------------------------------------------------------------------------
    //! @brief        リングバッファへ書き込みます。
    //---------------------------------------------------------------------------
    u32                     Write(
                                const void*     buf,
                                u32             size);

    //---------------------------------------------------------------------------
    //! @brief        リングバッファの内容をスキップします。
    //---------------------------------------------------------------------------
    u32                     Skip(
                                u32             size);

    //---------------------------------------------------------------------------
    //! @brief        リングバッファの内容を読み出します。ポインタは変更しません。
    //---------------------------------------------------------------------------
    u32                     Peek(
                                void*       buf,
                                u32             size);

private:
    void                    BeginRead(SimpleRingBufferRange* pRange);

    u32                     ContinueRead(
                                SimpleRingBufferRange*    pRange,
                                void*                     buf,
                                u32                       size);

    void                    EndRead(const SimpleRingBufferRange& range);

    void                    BeginWrite(SimpleRingBufferRange* pRange);

    u32                     ContinueWrite(
                                SimpleRingBufferRange*    pRange,
                                const void*               buf,
                                u32                       size);

    void                    EndWrite(const SimpleRingBufferRange& range);

    void                    BeginSkip(SimpleRingBufferRange* pRange);

    u32                     ContinueSkip(
                                SimpleRingBufferRange*    pRange,
                                u32                       size);

    void                    EndSkip(const SimpleRingBufferRange& range);

private:
    SimpleRingBufferHeader*     m_pHeader;
};

}   // namespace fnd
}   // namespace internal
}   // namespace snd
}   // namespace nw

/* NW_SND_FND_SIMPLERINGBUFFER_H_ */
#endif
