﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SND_EDIT_QUERY_ITEM_INFO_PACKET_H_
#define NW_SND_EDIT_QUERY_ITEM_INFO_PACKET_H_

#include <nw/snd/snd_Config.h>
#ifdef NW_SND_CONFIG_ENABLE_DEV

#include <nw/snd/edit/hio/sndedit_HioProtocol.h>
#include <nw/snd/edit/res/sndedit_ResItemInfo.h>

namespace nw {
namespace snd {
namespace edit {
namespace internal {

//---------------------------------------------------------------------------
//! @brief  アイテム情報を取得するパケットです。
//---------------------------------------------------------------------------
class QueryItemInfoPacket : public HioPacket
{
public:
    class Body : public HioPacketBody
    {
    public:
        Body(const char* itemName, u32 itemNameLength)
        {
           m_ItemName.len = itemNameLength + 1;

           if(itemName != NULL && itemNameLength > 0)
           {
               SetItemName(itemName, itemNameLength);
           }
        }

    public:
        static u32 GetRequiredSize(u32 maxItemName)
        {
            return ResName::GetRequiredSize(maxItemName);
        }

        u32   GetItemNameLength() const { return m_ItemName.len; }
        const char* GetItemName()       const { return m_ItemName.str; }

    private:
        void SetItemName(const char* value, u32 length)
        {
            if(m_ItemName.len == 0)
            {
                return;
            }

            if(value == NULL || *value == '\0')
            {
                m_ItemName.str[0] = '\0';
            }

            ut::strncpy(m_ItemName.str, m_ItemName.len + 1, value, length);
        }

    private:
        ResNameData m_ItemName;
    };

private:
    QueryItemInfoPacket(const char* itemName, u32 maxItemName) :
       HioPacket(HIO_MESSAGE_QUERY_ITEM_INFO, Body::GetRequiredSize(maxItemName)),
        m_Body(itemName, maxItemName)
    {
    }

public:
    static QueryItemInfoPacket* CreateInstance(void* buffer, const char* itemName)
    {
        NW_ASSERT_NOT_NULL(buffer);

        u32 itemNameLength = itemName == NULL ? 0 : std::strlen(itemName);
        return new(buffer) QueryItemInfoPacket(itemName, itemNameLength);
    }

    static u32 GetRequiredSize(u32 maxItemName)
    {
        // 固定パケットサイズ + 名前の長さ
        return sizeof(HioPacket) + Body::GetRequiredSize(maxItemName);
    }

    Body&       GetBody() { return m_Body; }
    const Body& GetBody() const { return m_Body; }

private:
    Body m_Body;
};

#if defined (NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
#pragma warning(push)
#pragma warning(disable:4200)
#endif
//---------------------------------------------------------------------------
//! @brief  QueryItemInfoPacket の返信パケットです。
//---------------------------------------------------------------------------
class QueryItemInfoReplyPacket : public HioPacket
{
public:
    struct Body : public HioPacketBody
    {
        snd::internal::fnd::BinU32 itemType;
        u8                         itemEditInfo[0];
    };
#if defined (NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
#pragma warning(pop)
#endif

public:
    QueryItemInfoReplyPacket() :
      HioPacket(HIO_MESSAGE_QUERY_ITEM_INFO_REPLY, sizeof(HioPacketHeader))
    {
    }

public:
    static u32 GetRequiredSize(u32 maxItemInfoSize)
    {
        // 固定パケットサイズ + アイテム情報の長さ
        return sizeof(HioPacketHeader) + sizeof(snd::internal::fnd::BinU32) + maxItemInfoSize;
    }

    const void* GetItemInfo() const
    {
        return m_Body.itemEditInfo;
    }

    ResDataType GetItemType() const
    {
        return static_cast<ResDataType>(static_cast<u32>(m_Body.itemType));
    }

    Body&       GetBody() { return m_Body; }
    const Body& GetBody() const { return m_Body; }

private:
    Body m_Body;
};

} // namespace nw::snd::edit::internal
} // namespace nw::snd::edit
} // namespace nw::snd
} // namespace nw

#endif // NW_SND_CONFIG_ENABLE_DEV

#endif // NW_SND_EDIT_QUERY_ITEM_INFO_PACKET_H_
