﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SND_CTRL_SOUND_PARAMETERS_CAFE_H_
#define NW_SND_CTRL_SOUND_PARAMETERS_CAFE_H_

#include <nw/snd/snd_Config.h>
#ifdef NW_SND_CONFIG_ENABLE_DEV

#include <nw/snd/ctrl/sndctrl_Config.h>
#include <nw/snd/ctrl/sndctrl_SequenceVariable.h>

namespace nw {
namespace snd {
namespace ctrl {

//---------------------------------------------------------------------------
//! @brief  Cafe 向けサウンドのパラメータ構造体です。
//---------------------------------------------------------------------------
struct SoundParameters
{
    f32  volume;                                    //!< 音量です。
    f32  pitch;                                     //!< ピッチです。

    u32  startOffset;                               //!< 再生開始位置です。

    f32  pan;                                       //!< パンです。
    f32  mainPan;                                   //!< メイン出力のパンです。
    f32  drcPan[Limits::MAX_DRC_COUNT];             //!< DRC 出力のパンです。

    f32  surroundPan;                               //!< サラウンドパンです。
    f32  mainSurroundPan;                           //!< メイン出力のサラウンドパンです。
    f32  drcSurroundPan[Limits::MAX_DRC_COUNT];     //!< DRC 出力のサラウンドパンです。

    f32  lpfFrequency;                              //!< LPF 周波数です。
    s32  biquadFilterType;                          //!< Biquad フィルタの種類です。
    f32  biquadFilterValue;                         //!< Biquad フィルタのかかり具合です。

    f32  mainSend;                                                          //!< メインバスのセンド量です。
    f32  fxSend[Limits::MAX_EFFECT_BUS_COUNT];                              //!< 各エフェクトバスのセンド量です。
    f32  mainMainSend;                                                      //!< メイン出力 メインバスのセンド量です。
    f32  mainFxSend[Limits::MAX_EFFECT_BUS_COUNT];                          //!< メイン出力 各エフェクトバスのセンド量です。
    f32  drcMainSend[Limits::MAX_DRC_COUNT];                                //!< DRC 出力 メインバスのセンド量です。
    f32  drcFxSend[Limits::MAX_DRC_COUNT][Limits::MAX_EFFECT_BUS_COUNT];    //!< DRC 出力 各エフェクトバスのセンド量です。

    f32  mainOutVolume;                                                     //!< メイン出力の音量です。
    f32  drcOutVolume[Limits::MAX_DRC_COUNT];                               //!< DRC 出力の音量です。
    f32  remoteOutVolume[Limits::MAX_REMOTE_COUNT];                         //!< リモコン 出力の音量です。

    bool isMainOutEnabled;                                  //!< メイン出力の有効化フラグです。
    bool isDrcOutEnabled[Limits::MAX_DRC_COUNT];            //!< DRC 出力の有効化フラグです。
    u8   padding;
    bool isRemoteOutEnabled[Limits::MAX_REMOTE_COUNT];      //!< リモコン出力の有効化フラグです。

    bool tracksMute[Limits::MAX_SEQUENCE_TRACK_COUNT];      //!< シーケンストラックのミュートフラグです。
    f32  tracksVolume[Limits::MAX_SEQUENCE_TRACK_COUNT];    //!< シーケンストラックの音量です。

    // 再生開始時にシーケンス変数を設定するためのコンテナ
    //! @briefprivate
    internal::SequenceVariableContainer localVariables;
    //! @briefprivate
    internal::SequenceVariableContainer trackVariables[ internal::SequenceVariableTraits::TrackCount ];

    SoundParameters() :
      volume(1.f),
      pitch(1.f),
      startOffset(0),
      pan(0.f),
      mainPan(0.f),
      surroundPan(0.f),
      mainSurroundPan(0.f),
      lpfFrequency(0.f),
      biquadFilterType(BIQUAD_FILTER_TYPE_INHERIT),
      biquadFilterValue(0.5f),
      mainSend(0.f),
      mainMainSend(0.f),
      mainOutVolume(1.f),
      isMainOutEnabled(true)
    {
        for (u32 index = 0; index < Limits::MAX_EFFECT_BUS_COUNT; ++index)
        {
            fxSend[index] = 0.f;
            mainFxSend[index] = 0.f;
        }

        for (u32 index = 0; index < Limits::MAX_DRC_COUNT; ++index)
        {
            drcPan[index]= 0.f;
            drcSurroundPan[index] = 0.f;
            isDrcOutEnabled[index] = false;
            drcOutVolume[index] = 1.f;

            drcMainSend[index] = 1.f;
            for (u32 busIndex = 0; busIndex < Limits::MAX_EFFECT_BUS_COUNT; ++busIndex)
            {
                drcFxSend[index][busIndex] = 0.f;
            }
        }

        for (u32 index = 0; index < Limits::MAX_REMOTE_COUNT; ++index)
        {
            isRemoteOutEnabled[index] = false;
            remoteOutVolume[index] = 0.f;
        }

        for (u32 index = 0; index < Limits::MAX_SEQUENCE_TRACK_COUNT; ++index)
        {
            tracksMute[index] = false;
            tracksVolume[index] = 1.0f;
        }
    }
};

} // namespace nw::snd::ctrl
} // namespace nw::snd
} // namespace nw

#endif // NW_SND_CONFIG_ENABLE_DEV

#endif // NW_SND_CTRL_SOUND_PARAMETERS_CAFE_H_
