﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SND_AAC_AACDECODER_H_
#define NW_SND_AAC_AACDECODER_H_

#include <cafe/aacdec.h>
#include <nw/snd/snd_StreamSoundLoader.h> // for StreamDataDecoder
#include <nw/snd/fnd/basis/sndfnd_Memory.h>

namespace nw {
namespace snd {

class SoundArchive;

namespace aac {

//---------------------------------------------------------------------------
//! @brief  AAC ライブラリを初期化します。
//!
//!         @ref GetAacDecBufferSize() で取得したサイズのバッファを設定してください。
//!         必要となるバッファサイズはサウンドアーカイブによって異なるため、
//!         使用するサウンドアーカイブを変更する場合には、再度初期化をする必要があります。
//!
//!         AAC を利用する場合、ストリームサウンドの ファイルリードキャッシュ機能
//!         を使うことが必須となります。ファイルリードキャッシュ機能については、
//!         @ref SoundArchivePlayer::Initialize をご確認ください。
//!
//!
//! @param[in]    buffer      AAC ライブラリ内部で使用するバッファへのポインタです。
//!                           64 バイト境界に配置されている必要があります。
//! @param[in]    size        AAC ライブラリ内部で使用するバッファのサイズです。
//!
//! @see Finalize
//! @see GetAacDecBufferSize
//!
//! @date 2013/07/03 初版
//---------------------------------------------------------------------------
void Initialize( void* buffer, u32 size );

//---------------------------------------------------------------------------
//! @brief  AAC ライブラリの終了処理を行います。
//!
//! @see Initialize
//! @see GetAacDecBufferSize
//!
//! @date 2013/07/03 初版
//---------------------------------------------------------------------------
void Finalize();

//---------------------------------------------------------------------------
//! @brief  AAC ライブラリが内部で使用するバッファのサイズを取得します。
//!
//!         サウンドアーカイブに設定されているストリームサウンドの数が多いほど
//!         必要となるバッファのサイズが大きくなります。
//!
//! @param[in]    arc      対象となるサウンドアーカイブのポインタです。
//!
//! @return 必要となるバッファサイズ(バイト)を返します。
//!
//! @see Initialize
//! @see Finalize
//!
//! @date 2013/07/03 初版
//---------------------------------------------------------------------------
u32 GetAacDecBufferSize( const SoundArchive* arc );

namespace internal {

class AacDecoder : public nw::snd::internal::driver::StreamDataDecoder
{
public:
    AacDecoder(): m_pAacInfo(NULL), m_StreamNum(0), m_pInputDataBuffer(NULL), m_WorkBufferSize(0) {}

    void Initialize( void* buffer, u32 size, u32 streamNum );
    void Finalize();
    u32 GetAacDecBufferSize(HEAACDEC_AAC_INFO* aacInfo);

    virtual void* AllocWorkBuffer();
    virtual void  FreeWorkBuffer(void* buffer);
    virtual bool  ReadDataInfo( ut::FileStream* fileStream, DataInfo* info, void* workBuffer );
    virtual bool  Skip( ut::FileStream* fileStream );
    virtual bool  Decode( ut::FileStream* fileStream, int channelCount, DecodeType decodeType, s16* decodedDataArray[], void* workBuffer );

private:
    void MultiBlockDecode(ut::FileStream* fileStream, u8* inputBitStream, int channelCount, DecodeType decodeType, s16* decodedDataArray[], void* workBuffer);
    void SingleBlockDecode(ut::FileStream* fileStream, u8* inputBitStream, s16* decodedDataArray[], void* workBuffer);
    bool SingleBlockSkip(ut::FileStream* fileStream);

    HEAACDEC_AAC_INFO* m_pAacInfo;
    void* m_pInputDataBuffer;
    u32 m_StreamNum;
    nw::snd::internal::fnd::UnitHeap m_WorkBufferHeap;
    u32 m_WorkBufferSize;
};

} // namespace nw::snd::aac::internal

} // namespace nw::snd::aac
} // namespace nw::snd
} // namespace nw


#endif /* NW_SND_AAC_AACDECODER_H_ */

