﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_MCS_MCS_IO_STREAM_H_
#define NW_MCS_MCS_IO_STREAM_H_

#include <nw/mcs/mcs_Base.h>
#include <nw/ut/ut_IoStream.h>

namespace nw   {
namespace mcs    {

//---------------------------------------------------------------------------
//! @brief Mcs を用いたストリームクラスです。
//!
//! @details
//! :category ストリーム入出力
//!
//! McsIoStream は ut::IoStream の派生クラスです。
//! Mcs サーバとのストリーム入出力操作を実装します。
//!
//! Mcs ストリーム は非同期処理をサポートしないため、
//! 非同期型のメンバー関数は使用できません。
//!
//! @sa ut::IoStream
//---------------------------------------------------------------------------
class McsIoStream : public ut::IoStream
{
#if defined(NW_MCS_ENABLE)

// ------------- public members -------------------
public:
    NW_UT_RUNTIME_TYPEINFO( ut::IoStream )

    //! @brief エラーコードです。
    //!
    //! @sa Read
    //! @sa Write
    enum Result
    {
        //! 成功しました。
        SUCCESS = MCS_ERROR_SUCCESS,

        //! 通信エラーが発生しました。
        ERR_COMERROR           = -MCS_ERROR_COMERROR,
        //! Mcs サーバと接続されていません。
        ERR_NOTCONNECT         = -MCS_ERROR_NOTCONNECT,

        //! 列挙子の最後です。
        ERR_LAST               = ERR_NOTCONNECT
    };

    //! @name コンストラクタ / デストラクタ
    //@{

    //---------------------------------------------------------------------------
    //! @brief      コンストラクタです。
    //!
    //! @details
    //! ストリームをオープンしない状態でオブジェクトが生成されます。
    //---------------------------------------------------------------------------
    /* ctor */              McsIoStream();

    //---------------------------------------------------------------------------
    //! @brief      コンストラクタです。
    //!
    //! @param[in]  channel     チャンネル番号を指定します。
    //! @param[in]  buf         受信用バッファを指定します。
    //! @param[in]  bufSize     受信用バッファのサイズを指定します。
    //!
    //! @details
    //! チャンネル番号を指定してストリームを開きます。
    //---------------------------------------------------------------------------
    /* ctor */ McsIoStream( mcs::ChannelType channel, void* buf, u32 length );

    //---------------------------------------------------------------------------
    //! @brief      デストラクタです。
    //!
    //! @details
    //! ストリームが開かれていた場合にはクローズします。
    //!
    //! @sa McsIoStream::McsIoStream
    //! @sa Open
    //---------------------------------------------------------------------------
    /* dtor */ virtual ~McsIoStream();

    //@}

    //! @name ストリームのオープン / クローズ
    //@{

    //---------------------------------------------------------------------------
    //! @brief      ストリームをオープンします。
    //!
    //! @param[in]  channel     チャンネル番号を指定します。
    //! @param[in]  buf         受信用バッファを指定します。
    //! @param[in]  bufSize     受信用バッファのサイズを指定します。
    //!
    //! @return     ストリームのオープンに成功した場合にはtrue
    //!             失敗した場合にはfalseが返されます。
    //!
    //! @sa Close
    //---------------------------------------------------------------------------
    bool Open( mcs::ChannelType channel, void* buf, u32 length );

    //---------------------------------------------------------------------------
    //! @brief      ストリームをクローズします。
    //!
    //! @details
    //! ストリームが開かれていた場合にはクローズします。
    //!
    //! @sa McsIoStream::McsIoStream
    //! @sa Open
    //---------------------------------------------------------------------------
    virtual void Close( void );

    //@}

    //! @name ストリームの操作
    //@{

    //---------------------------------------------------------------------------
    //! @brief      同期リードします。
    //!
    //! @param[in]  buf     読み込みバッファへのポインタ。\n
    //!                     GetBufferAlign() で取得されるアライメントに合っている必要があります。
    //! @param[in]  length  読み込むバイト数。\n
    //!                     GetSizeAlign() で取得されるアライメントに合っている必要があります。
    //!
    //! @return     正常に読み込めた場合には 読み込んだデータサイズを返します。
    //!             エラーの場合には負の数(McsIoStream::Result)を返します。
    //!
    //! @details
    //! ストリームからバッファへデータを読み出します。
    //! CanRead() の返り値が false になる場合にはアサートで失敗します。
    //!
    //! この関数は読み出しが完了するまでの間、処理をブロックします。
    //!
    //! McsIoStream は非同期処理を行うことができませんので、
    //! 非同期版関数である ReadAsync() を呼び出さないでください。
    //!
    //! この関数が返すエラーの詳細は以下の通りです：\n
    //! (mcs::McsIoStream::Result)
    //!
    //! ''ERR_COMERROR''\n
    //! 通信エラーが発生しました。
    //!
    //! ''ERR_NOTCONNECT''\n
    //! Mcs サーバと接続されていません。
    //!
    //! @sa CanRead
    //! @sa GetBufferAlign
    //! @sa GetSizeAlign
    //! @sa ReadAsync
    //---------------------------------------------------------------------------
    virtual s32 Read( void* buf, u32 length );

    //---------------------------------------------------------------------------
    //! @brief      データを破棄せずに同期リードします。
    //!
    //! @param[in]  buf     読み込みバッファへのポインタ。\n
    //!                     GetBufferAlign() で取得されるアライメントに合っている必要があります。
    //! @param[in]  length  読み込むバイト数。\n
    //!                     GetSizeAlign() で取得されるアライメントに合っている必要があります。
    //!
    //! @return     正常に読み込めた場合には 読み込んだデータサイズを返します。
    //!             エラーの場合には負の数(McsIoStream::Result)を返します。
    //!
    //! @details
    //! ストリームからバッファへデータを読み出します。
    //! CanRead() の返り値が false になる場合にはアサートで失敗します。
    //!
    //! この関数は Read() とは異なり、ストリーム読み込み位置を変更せず、
    //! 次回 Read() / Peek() 呼び出し時には、同じ位置からデータを読み込みます。
    //!
    //! また、読み出しが完了するまでの間、処理をブロックします。
    //!
    //! McsIoStream は非同期処理を行うことができませんので、
    //! 非同期版関数である ReadAsync() を呼び出さないでください。
    //!
    //! この関数が返すエラーの詳細は以下の通りです：\n
    //! (mcs::McsIoStream::Result)
    //!
    //! ''ERR_COMERROR''\n
    //! 通信エラーが発生しました。
    //!
    //! ''ERR_NOTCONNECT''\n
    //! Mcs サーバと接続されていません。
    //!
    //! @sa CanRead
    //! @sa GetBufferAlign
    //! @sa GetSizeAlign
    //! @sa Read
    //---------------------------------------------------------------------------
    virtual s32 Peek( void* buf, u32 length );

    //---------------------------------------------------------------------------
    //! @brief      データをスキップします。
    //!
    //! @param[in]  size  スキップするデータのサイズ。
    //!
    //! @return     関数が成功した場合、McsIoStream::Result::SUCCESSを返します。
    //!             失敗した場合エラーコード(McsIoStream::Result)を返します。
    //!
    //! @details
    //! この関数はスキップが完了するまでの間、処理をブロックします。
    //!
    //! @sa GetReadableBytes
    //! @sa Peek
    //! @sa Read
    //---------------------------------------------------------------------------
    s32 Skip(u32 length);

    //---------------------------------------------------------------------------
    //! @brief      同期ライトします。
    //!
    //! @param[in]  buf     書き込み元バッファへのポインタ。\n
    //!                     GetBufferAlign() で取得されるアライメントに合っている必要があります。
    //! @param[in]  length  書き込みデータのバイト数。\n
    //!                     GetSizeAlign() で取得されるアライメントに合っている必要があります。
    //!
    //! @return     書き込みに成功すれば、実際に書き込まれたバイト数が返ります。
    //!             エラーの場合には負の数(McsIoStream::Result)を返します。
    //!
    //! @details
    //! ストリームにデータを書き込みます。
    //! CanWrite() の返り値が false になる場合にはアサートで失敗します。
    //!
    //! この関数は書き込みが完了するまでの間、処理をブロックします。
    //!
    //! McsIoStream は非同期処理を行うことができませんので、
    //! 非同期版関数である WriteAsync() を呼び出さないでください。
    //!
    //! この関数が返すエラーの詳細は以下の通りです：\n
    //! (mcs::McsIoStream::Result)
    //!
    //! ''ERR_COMERROR''\n
    //! 通信エラーが発生しました。
    //!
    //! ''ERR_NOTCONNECT''\n
    //! Mcs サーバと接続されていません。
    //!
    //! @sa CanWrite
    //! @sa GetBufferAlign
    //! @sa GetSizeAlign
    //! @sa WriteAsync
    //---------------------------------------------------------------------------
    virtual s32 Write( const void* buf, u32 length );

    //@}

    //! @name ストリームの状態取得
    //@{

    //---------------------------------------------------------------------------
    //! @brief      ストリームがオープンされているかどうかを取得します。
    //!
    //! @return     このストリームが現在使用可能な状態であれば true を返します。
    //!             ストリームがオープンされていない、既にクローズされた等の理由で使用不能で
    //!             あった場合には false を返します。
    //---------------------------------------------------------------------------
    virtual bool IsAvailable() const { return m_IsAvailable; }

    //---------------------------------------------------------------------------
    //! @brief      チャンネル番号を取得します。
    //!
    //! @return     チャンネル番号を返します。
    //---------------------------------------------------------------------------
    const mcs::ChannelType GetChannel() const { return m_Channel; }

    //---------------------------------------------------------------------------
    //! @brief      ストリームから読み込み可能なサイズを取得します。
    //!
    //! @return     ストリームから読み込み可能なサイズを返します。
    //---------------------------------------------------------------------------
    u32 GetReadableBytes() const;

    //---------------------------------------------------------------------------
    //! @brief      ストリームに書き込み可能なサイズを取得します。
    //!
    //! @return     ストリームに書き込み可能なサイズを返します。
    //---------------------------------------------------------------------------
    u32 GetWritableBytes() const;

    //@}

    //! @name ストリームの属性情報を取得
    //@{

    //---------------------------------------------------------------------------
    //! @brief      非同期処理の可否を取得します。
    //!
    //! @return     McsIoStream は非同期処理が不可能であるため、必ず false が返ります。
    //---------------------------------------------------------------------------
    virtual bool CanAsync( void )  const { return false; }

    //---------------------------------------------------------------------------
    //! @brief      このストリームがリード可能かどうかを取得します。
    //!
    //! @return     このストリームが読み込み可能である場合には true を、
    //!             読み込みができない場合には false を返します。
    //!
    //! @sa Read
    //---------------------------------------------------------------------------
    virtual bool CanRead ( void ) const { return m_IsAvailable;  }

    //---------------------------------------------------------------------------
    //! @brief      このストリームがライト可能かどうかを取得します。
    //!
    //! @return     このストリームが書き込み可能である場合には true を、
    //!             書き込みできない場合には false を返します。
    //!
    //! @sa Write
    //---------------------------------------------------------------------------
    virtual bool CanWrite( void ) const { return m_IsAvailable; }

    //@}

// ------ private members --------------------------
private:
    bool m_IsAvailable;
    mcs::ChannelType m_Channel;

#else   // #if defined(NW_MCS_ENABLE)

// ------------- public members -------------------
public:
    NW_UT_RUNTIME_TYPEINFO( ut::IoStream )

    /* ctor */ McsIoStream() { }
    /* dtor */ virtual ~McsIoStream() { }

    bool Open( mcs::ChannelType /* channel */, void* /* buf */, u32 /* length */) { return false; }

    virtual void Close( void ) { }

    virtual s32 Read( void* /* buf */, u32 /* length */)        { return 0; }
    virtual s32 Peek( void* /* buf */, u32 /* length */)        { return 0; }
    s32 Skip(u32 /* length */)                                  { return 0; }
    virtual s32 Write( const void* /* buf */, u32 /* length */) { return 0; }
    virtual bool IsAvailable() const                            { return false; }

    const mcs::ChannelType GetChannel() const { return 0; }
    u32 GetReadableBytes() const { return 0; }
    u32 GetWritableBytes() const { return 0; }

    virtual bool CanAsync( void ) const { return false; }
    virtual bool CanRead ( void ) const { return false; }
    virtual bool CanWrite( void ) const { return false; }

#endif  // #if defined(NW_MCS_ENABLE)
};



}   /* namespace mcs    */
}   /* namespace nw   */



/* NW_MCS_MCS_IO_STREAM_H_ */
#endif
