﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_MCS_HIO2RINGBUFFER_H_
#define NW_MCS_HIO2RINGBUFFER_H_

#if (defined(_MSC_VER) && _MSC_VER > 1000) || defined(__CWCC__)
#pragma once
#endif

#include <nw/types.h>

//================================================================================
// Definitions of items such as compile options
// #define ENABLE_MCS_REPORT // Debug Print On
//================================================================================

namespace nw     {
namespace mcs    {
namespace internal {

#if defined(NW_MCS_ENABLE)

//---------------------------------------------------------------------------
//! @brief        Mcs の共有バッファクラスです。
//!
//!  This class uses a ring buffer to manage shared memory for the HIO library
//!  in an HIO Communication Device.
//!
//!  HIO shared memory is manipulated through this ring buffer
//!
//!  The same code is shared by the library for the actual hardware and the PC.
//!  Note that this class is unrelated to SimpleRingBuffer in the class hierarchy.
//---------------------------------------------------------------------------
class Hio2RingBuffer
{
#if defined(MCS_SERVER)

    public:
        static void             SetHIO2FuncAddress(
                                    HIO2ReadType            readFunc,
                                    HIO2WriteType           writeFunc,
                                    HIO2GetLastErrorType    getLastErrorFunc);

    protected:
        static HIO2ReadType         mspfHIO2Read;
        static HIO2WriteType        mspfHIO2Write;
        static HIO2GetLastErrorType mspfHIO2GetLastError;

#endif  // #if defined(WIN32)

public:
    static const int    CopyAlignment   = 32;

    //---------------------------------------------------------------------------
    //! @brief        デフォルトコンストラクタ
    //---------------------------------------------------------------------------
    /* ctor */          Hio2RingBuffer() {}

    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //!
    //! @param[in]    devHandle 通信用の HIO ハンドルです。
    //! @param[in]    bufOffset 共有バッファの開始オフセット。
    //! @param[in]    size      共有バッファのサイズ
    //! @param[out]   tempBuf   HioRingBufferオブジェクトが使用する作業メモリ
    //---------------------------------------------------------------------------                        Hio2RingBuffer(
    /* ctor */          Hio2RingBuffer(
                            HIO2Handle  devHandle,
                            u32         bufOffset,
                            u32         size,
                            void*       tempBuf
                        )
                        {
                            Init(devHandle, bufOffset, size, tempBuf);
                        }

    //---------------------------------------------------------------------------
    //! @brief       HioRingBufferオブジェクトを初期化します。
    //!
    //! @param[in]   devHandle 通信用の HIO ハンドルです。
    //! @param[in]   baseOffset   共有メモリの開始オフセット。
    //! @param[in]   size         共有メモリのサイズ
    //! @param[in]   tempBuf      HioRingBufferオブジェクトが使用する作業メモリ
    //!
    //! @return      なし。
    //---------------------------------------------------------------------------
    void                Init(
                            HIO2Handle  devHandle,
                            u32         baseOffset,
                            u32         size,
                            void*       tempBuf);

    //---------------------------------------------------------------------------
    //! @brief        バッファを初期化します。
    //---------------------------------------------------------------------------
    void                InitBuffer();

    //---------------------------------------------------------------------------
    //! @brief        バッファを初期化します。
    //---------------------------------------------------------------------------
    void                EnablePort()    { m_EnablePort = true;  }
    void                DisablePort()   { m_EnablePort = false; }
    bool                IsPort()        { return m_EnablePort; }

    u32                 GetBufferSize() const   { return m_BufferSize; }

    bool                Read(bool* pbAvailable);

    void*               GetMessage(
                            u32*    pChannel,
                            u32*    pTotalSize);

    bool                GetWritableBytes(
                            u32*    pBytes,
                            bool    withUpdate = true);

    bool                Write(
                            u32         channel,
                            const void* buf,
                            u32         size);

//PC_COMMENT Added functions
    bool                Peek(bool* pbAvailable);
    bool                Skip();
    void*               PeekMessage(
                            u32*    pChannel,
                            u32*    pTotalSize);
protected:
    bool                ReadSection_(
                            u32     offset,
                            void*   data,
                            u32     size);

    bool                WriteSection_(
                            u32         offset,
                            const void* data,
                            u32         size);

    bool                UpdateReadWritePoint_();
    bool                UpdateReadPoint_();
    bool                UpdateWritePoint_();

    HIO2Handle          m_DevHandle;

    u32                 m_BaseOffset;        // Offset for the buffer to use
    u32                 m_BufferSize;        // Size of the buffer that can be used (excluding the header)
    void*               m_TempBuf;           // Temporary read/write buffer

    u32                 m_ReadPoint;         // Current read point in HIO shared memory
    u32                 m_WritePoint;        // Current write point in HIO shared memory

    u32                 m_ProcBytes;         // Number of bytes processed from the data transferred to the temporary buffer
    u32                 m_TransBytes;        // Number of bytes transferred to temporary buffer

    bool                m_EnablePort;        // Can USB2EXI be used without error?
    u8                  padding[3];         // warning handling

};

#endif  // #if defined(NW_MCS_ENABLE)

}   // namespace internal
}   // namespace mcs
}   // namespace nw

#endif /* NW_MCS_HIO2RINGBUFFER_H_ */

