﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_MCS_COMMON_H_
#define NW_MCS_COMMON_H_

#if (defined(_MSC_VER) && _MSC_VER > 1000) || defined(__CWCC__)
#pragma once
#endif

#include <nw/types.h>

namespace nw  {
namespace mcs {
namespace internal {

// #define NW_MCS_LOG_ENABLE

#if defined NW_MCS_LOG_ENABLE
    #define NW_MCS_LOG      NW_LOG
#else
    #define NW_MCS_LOG(...) (void)0
#endif

//----------------------------------------------------------------------------
// An HIO function will sometimes fail if it is called while the OSReport function is running in a low-priority thread.
//
// The MCS library therefore contains processing that only retries for the number of times established by HIOFuncRetryCountMax.
//
//
const int   HIOFuncRetryCountMax   = 30;  //    (retry count)
const int   HIOFuncRetrySleepTime  = 8;   // ms (time to wait during a retry, and the interval between retries)


const u32   MCS_INTERNAL_CHUNKHEADER_SIZE    = 0x20;
const u32   MCS_INTERNAL_CHUNKHEADER_LENGTH    = MCS_INTERNAL_CHUNKHEADER_SIZE / sizeof(u32);
// internal buffer size (actually multiplied by two because it is provided for both sending and receiving)
const u32   MCS_INTERNAL_BUFFER_SIZE    = 0x1000;

// Maximum data chunk size to send from the PC
const u32   MCS_INTERNAL_CHUNK_SIZE_MAX = 0x1000;



template<typename Type>
inline const Type&
GetMin(
    const Type& a,
    const Type& b
)
{
    return a < b ? a: b;
}

//---------------------------------------------------------------------------
//! @brief        切り上げ処理
//!               McsServer のコードでもターゲットと同様のものを使用する為に、
//!               ut を使わずに独自実装
//---------------------------------------------------------------------------
template<typename Type>
inline Type
RoundUp(
    Type    value,
    u32     alignment
)
{
    return Type((u32(value) + alignment -1) & ~(alignment -1));
}

//---------------------------------------------------------------------------
//! @brief        切り捨て処理
//!               McsServer のコードでもターゲットと同様のものを使用する為に、
//!               ut を使わずに独自実装
//---------------------------------------------------------------------------
template<typename Type>
inline Type
RoundDown(
    Type    value,
    u32     alignment
)
{
    return Type(u32(value) & ~(alignment -1));
}

void SleepThread(u32 milSec);

#if NW_PLATFORM_ENDIAN == NW_ENDIAN_BIG
template <typename T> T NetToHost(T value) { return value; }
template <typename T> T HostToNet(T value) { return value; }
#else
template <typename T> T NetToHost(T value) { return nw::ut::ReverseEndian(value); }
template <typename T> T HostToNet(T value) { return nw::ut::ReverseEndian(value); }
#endif

}   // namespace internal
}   // namespace mcs
}   // namespace nw

/* NW_MCS_COMMON_H_ */
#endif
