﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo/HAL Laboratory, Inc. All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

namespace nw {
namespace math {

//---------------------------------------------------------------------------
//        VEC2
//---------------------------------------------------------------------------

namespace internal { namespace standard {

NW_MATH_INLINE VEC2*
VEC2Maximize(VEC2* pOut, const VEC2* p1, const VEC2* p2)
{
    NW_ASSERT_NOT_NULL(pOut);
    NW_ASSERT_NOT_NULL(p1);
    NW_ASSERT_NOT_NULL(p2);

    pOut->x = (p1->x > p2->x) ? p1->x : p2->x;
    pOut->y = (p1->y > p2->y) ? p1->y : p2->y;

    return pOut;
}

NW_MATH_INLINE VEC2*
VEC2Minimize(VEC2* pOut, const VEC2* p1, const VEC2* p2)
{
    NW_ASSERT_NOT_NULL(pOut);
    NW_ASSERT_NOT_NULL(p1);
    NW_ASSERT_NOT_NULL(p2);

    pOut->x = (p1->x < p2->x) ? p1->x : p2->x;
    pOut->y = (p1->y < p2->y) ? p1->y : p2->y;

    return pOut;
}

}} // namespace internal::standard

#if defined(NW_MATH_ENABLE_INTRINSICS)

namespace internal { namespace intrinsics {

NW_MATH_INLINE VEC2*
VEC2Maximize(VEC2* pOut, const VEC2* p1, const VEC2* p2)
{
    NW_ASSERT_NOT_NULL(pOut);
    NW_ASSERT_NOT_NULL(p1);
    NW_ASSERT_NOT_NULL(p2);

    f32x2 f0 = tof32x2(p1->x);
    f32x2 f1 = tof32x2(p2->x);
    tof32x2(pOut->x) = __PS_SEL(__PS_SUB(f0, f1), f0, f1);

    return pOut;
}

NW_MATH_INLINE VEC2*
VEC2Minimize(VEC2* pOut, const VEC2* p1, const VEC2* p2)
{
    NW_ASSERT_NOT_NULL(pOut);
    NW_ASSERT_NOT_NULL(p1);
    NW_ASSERT_NOT_NULL(p2);

    f32x2 f0 = tof32x2(p1->x);
    f32x2 f1 = tof32x2(p2->x);
    tof32x2(pOut->x) = __PS_SEL(__PS_SUB(f1, f0), f0, f1);

    return pOut;
}

}} // namespace internal::intrinsics

#endif // NW_MATH_ENABLE_INTRINSICS

//----------------------------------------
//! @name    ベクトル
//@{

//---------------------------------------------------------------------------
//! @brief        ベクトルがゼロベクトルかどうか判定します。
//!
//! @param[in]    p    判定対象のベクトルへのポインタ。
//!
//! @return       p がゼロベクトルであれば true そうでなければ false を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE bool
VEC2IsZero(const VEC2* p)
{
    return p->x == 0.f && p->y == 0.f;
}

//---------------------------------------------------------------------------
//! @brief        2つのベクトルのそれぞれの成分の最大値から構成されるベクトルを作成します。
//!
//! @param[out]   pOut  計算結果を受け取るバッファへのポインタ。p1, p2 と同じベクトルを指していても構いません。
//! @param[in]    p1    対象のベクトル1へのポインタ。
//! @param[in]    p2    対象のベクトル2へのポインタ。
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE VEC2*
VEC2Maximize(VEC2* pOut, const VEC2* p1, const VEC2* p2)
{
    return NW_MATH_IMPL_NS::VEC2Maximize(pOut, p1, p2);
}


//---------------------------------------------------------------------------
//! @brief        2つのベクトルのそれぞれの成分の最小値から構成されるベクトルを作成します。
//!
//! @param[out]   pOut  計算結果を受け取るバッファへのポインタ。p1, p2 と同じベクトルを指していても構いません。
//! @param[in]    p1    対象のベクトル1へのポインタ。
//! @param[in]    p2    対象のベクトル2へのポインタ。
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE VEC2*
VEC2Minimize(VEC2* pOut, const VEC2* p1, const VEC2* p2)
{
    return NW_MATH_IMPL_NS::VEC2Minimize(pOut, p1, p2);
}


//---------------------------------------------------------------------------
//! @brief        ベクトルを正規化します
//!
//! @param[out]   pOut  計算結果を受け取るバッファへのポインタ。p と同じベクトルを指していても構いません。
//! @param[in]    p     対象のベクトル1へのポインタ。
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE VEC2*
VEC2Normalize(VEC2* pOut, const VEC2* p)
{
    (void)VEC2Scale(pOut, p, FrSqrt(p->x * p->x + p->y * p->y));

    return pOut;
}

//---------------------------------------------------------------------------
//! @brief        ベクトルを正規化します
//!               VEC2Normalize より高速ですが、求められる計算結果の精度に差があります。
//!
//! @param[out]   pOut  計算結果を受け取るバッファへのポインタ。p と同じベクトルを指していても構いません。
//! @param[in]    p     対象のベクトル1へのポインタ。
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE VEC2*
VEC2FastNormalize(VEC2* pOut, const VEC2* p)
{
    (void)VEC2Scale(pOut, p, FrFastSqrt(p->x * p->x + p->y * p->y));

    return pOut;
}


//---------------------------------------------------------------------------
//! @brief        ベクトルを正規化します
//!               正規化に失敗した場合は指定されたベクトルを設定します。
//!
//! @param[out]   pOut  計算結果を受け取るバッファへのポインタ。p と同じベクトルを指していても構いません。
//! @param[in]    p     対象のベクトル1へのポインタ。
//! @param[in]    alt   正規化に失敗した場合に設定するベクトル
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE VEC2*
VEC2SafeNormalize(VEC2* pOut, const VEC2* p, const VEC2& alt)
{
    NW_ASSERT_NOT_NULL(pOut);
    NW_ASSERT_NOT_NULL(p);

    f32 mag = (p->x * p->x) + (p->y * p->y);

    if (mag == 0 || mag == F_INF || isnan(mag))
    {
        *pOut = alt;

        return pOut;
    }

    (void)VEC2Scale(pOut, p, FrSqrt(mag));

    return pOut;
}


//---------------------------------------------------------------------------
//! @brief        ベクトルを正規化します
//!               正規化に失敗した場合は指定されたベクトルを設定します。
//!               VEC2SafeNormalize より高速ですが、求められる計算結果の精度に差があります。
//!
//! @param[out]   pOut  計算結果を受け取るバッファへのポインタ。p と同じベクトルを指していても構いません。
//! @param[in]    p     対象のベクトル1へのポインタ。
//! @param[in]    alt   正規化に失敗した場合に設定するベクトル
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE VEC2*
VEC2FastSafeNormalize(VEC2* pOut, const VEC2* p, const VEC2& alt)
{
    NW_ASSERT_NOT_NULL(pOut);
    NW_ASSERT_NOT_NULL(p);

    f32 mag = (p->x * p->x) + (p->y * p->y);

    if (mag == 0 || mag == F_INF || isnan(mag))
    {
        *pOut = alt;

        return pOut;
    }

    (void)VEC2Scale(pOut, p, FrFastSqrt(mag));

    return pOut;
}


//@}

}  // namespace math
}  // namespace nw
