﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_LYT_WINDOW_FOUNDATION_H_
#define NW_LYT_WINDOW_FOUNDATION_H_

#include <nw/lyt/lyt_Types.h>
#include <nw/lyt/lyt_Common.h>
#include <nw/lyt/lyt_DrawInfo.h>

#include <nw/ut/ut_Color.h>
#include <nw/gfnd/gfnd_ShaderHelper.h>

namespace nw
{
namespace lyt
{

//----------------------------------------

//! @brief 内部用機能のため使用禁止です。
enum WindowFrameMode
{
    AROUND_MODE,
    HORIZONTAL_MODE,
    HORIZONTAL_NOCONTENT_MODE,
    WINDOW_FRAME_MODE_QUANTITY
};
NW_STATIC_ASSERT(WINDOW_FRAME_MODE_QUANTITY <= 256);

//----------------------------------------

//! @brief 内部用機能のため使用禁止です。
struct WindowContent
{
    ut::Color8 vtxColors[VERTEXCOLOR_MAX];
    internal::TexCoordAry texCoordAry;
};

//----------------------------------------
// ウィンドウのフレームサイズをウィンドウクラス内で保持する構造体です。
struct WindowFrameSizeInternal
{
    u16          l; //!< 左辺の枠の幅です。
    u16          r; //!< 右辺の枠の幅です。
    u16          t; //!< 上辺の枠の高さです。
    u16          b; //!< 下辺の枠の高さです。
};

//----------------------------------------

// ウィンドウのコンテントの拡大量とフレームのサイズを持つ構造体です。

//! @brief 内部用機能のため使用禁止です。
struct WindowSize
{
    s16 l; // 左辺の拡大量です。固定少数(WindowInflationFiexdPoint)です。
    s16 r; // 右辺の拡大量です。固定少数(WindowInflationFiexdPoint)です。
    s16 t; // 上辺の拡大量です。固定少数(WindowInflationFiexdPoint)です。
    s16 b; // 下辺の拡大量です。固定少数(WindowInflationFiexdPoint)です。
    WindowFrameSizeInternal frameSize; // 枠のサイズです。
};

//----------------------------------------

//! @brief 内部用機能のため使用禁止です。
struct WindowFrameData
{
    WindowFrameData()
    : textureFlip(0)
    , pMaterial(0)
    {}

    virtual ~WindowFrameData() {}

    TextureFlip GetTextureFlip() const
    {
        return static_cast<TextureFlip>(this->textureFlip);
    }

    u8 textureFlip;
    Material* pMaterial;
};


//! @brief 内部用機能のため使用禁止です。
namespace window_foundation
{

//----------------------------------------
struct Rect
{
    f32 x, y, w, h;

    const nw::math::VEC2& Position() const
    {
        return *(const nw::math::VEC2 *)&this->x;
    }

    const nw::lyt::Size& Size() const
    {
        return *(const nw::lyt::Size *)&this->w;
    }
};

//----------------------------------------
NW_INLINE void
GetLTFrameSize(
    nw::math::VEC2*             pPt,
    Size*                   pSize,
    const nw::math::VEC2&       basePt,
    const Size&             winSize,
    const WindowFrameSize&  frameSize
)
{
    *pPt = basePt;

    pSize->width = winSize.width - frameSize.r;
    pSize->height = frameSize.t;
}

//----------------------------------------
NW_INLINE void
GetRTFrameSize(
    nw::math::VEC2*             pPt,
    Size*                   pSize,
    const nw::math::VEC2&       basePt,
    const Size&             winSize,
    const WindowFrameSize&  frameSize
)
{
    using namespace nw::math;

    *pPt = VEC2(basePt.x + winSize.width - frameSize.r, basePt.y);

    pSize->width  = frameSize.r;
    pSize->height = winSize.height - frameSize.b;
}

//----------------------------------------
NW_INLINE void
GetLBFrameSize(
    nw::math::VEC2*             pPt,
    Size*                   pSize,
    const nw::math::VEC2&       basePt,
    const Size&             winSize,
    const WindowFrameSize&  frameSize
)
{
    *pPt = nw::math::VEC2(basePt.x, basePt.y - frameSize.t);

    pSize->width  = frameSize.l;
    pSize->height = winSize.height - frameSize.t;
}

//----------------------------------------
NW_INLINE void
GetRBFrameSize(
    nw::math::VEC2*             pPt,
    Size*                   pSize,
    const nw::math::VEC2&       basePt,
    const Size&             winSize,
    const WindowFrameSize&  frameSize
)
{
    *pPt = nw::math::VEC2(basePt.x + frameSize.l, basePt.y - winSize.height + frameSize.b);

    pSize->width  = winSize.width - frameSize.l;
    pSize->height = frameSize.b;
}

//----------------------------------------
// @brief フレームのテクスチャ座標変換を設定します。
//
// @param gres グラフィックスリソースです。
// @param frame フレーム番号です。
// @param flip テクスチャフリップ番号です。
NW_INLINE void
SetupFrameTransform(
    const DrawInfo& drawInfo,
    WindowFrame frame,
    TextureFlip flip,
    bool vertexColorEnabled = false)
{
    namespace shhelp = gfnd::shader_helper;

    static const u32 frameFlag[WINDOWFRAME_MAX] =
    {
        internal::FRAMESPECFLAG_FRAME_LT, // WINDOWFRAME_LT
        internal::FRAMESPECFLAG_FRAME_RT, // WINDOWFRAME_RT
        internal::FRAMESPECFLAG_FRAME_LB, // WINDOWFRAME_LG
        internal::FRAMESPECFLAG_FRAME_RB, // WINDOWFRAME_RB
        internal::FRAMESPECFLAG_FRAME_L, // WINDOWFRAME_L
        internal::FRAMESPECFLAG_FRAME_R, // WINDOWFRAME_R
        internal::FRAMESPECFLAG_FRAME_T, // WINDOWFRAME_T
        internal::FRAMESPECFLAG_FRAME_B // WINDOWFRAME_B
    };

    static const u32 flipFlag[TEXTUREFLIP_MAX] =
    {
        internal::FRAMESPECFLAG_NORMAL,       // TEXTUREFLIP_NONE
        internal::FRAMESPECFLAG_FLIP_HFLIP,   // TEXTUREFLIP_FLIPH
        internal::FRAMESPECFLAG_FLIP_VFLIP,   // TEXTUREFLIP_FLIPV
        internal::FRAMESPECFLAG_FLIP_R90,     // TEXTUREFLIP_ROTATE90
        internal::FRAMESPECFLAG_FLIP_R180,    // TEXTUREFLIP_ROTATE180
        internal::FRAMESPECFLAG_FLIP_R270     // TEXTUREFLIP_ROTATE270
    };

    gfnd::ShaderInt flag = frameFlag[frame] | flipFlag[flip];
    gfnd::ShaderUniformId id = drawInfo.GetUniformId(UNIFORM_uFrameSpec);

    if (!vertexColorEnabled) { flag |= internal::VERTEX_EFFECT_CONST_COLOR; }

    shhelp::SetVertexUniformReg(id, flag);
    NW_GL_ASSERT();
}

//----------------------------------------
// @brief フレームのテクスチャ座標変換を設定します。
//
// @param gres グラフィックスリソースです。
// @param frame フレーム番号です。
// @param flip テクスチャフリップ番号です。
NW_INLINE void
SetupFrame4Transform(
    const DrawInfo& drawInfo,
    WindowFrame frame,
    TextureFlip flip,
    bool vertexColorEnabled = false)
{
    namespace shhelp = gfnd::shader_helper;

    static const u32 frameFlag[4] =
    {
        internal::FRAMESPECFLAG_FRAME4_LT, // WINDOWFRAME_LT
        internal::FRAMESPECFLAG_FRAME4_RT, // WINDOWFRAME_RT
        internal::FRAMESPECFLAG_FRAME4_LB, // WINDOWFRAME_LG
        internal::FRAMESPECFLAG_FRAME4_RB, // WINDOWFRAME_RB
    };

    static const u32 flipFlag[TEXTUREFLIP_MAX] =
    {
        internal::FRAMESPECFLAG_NORMAL,       // TEXTUREFLIP_NONE
        internal::FRAMESPECFLAG_FLIP_HFLIP,   // TEXTUREFLIP_FLIPH
        internal::FRAMESPECFLAG_FLIP_VFLIP,   // TEXTUREFLIP_FLIPV
        internal::FRAMESPECFLAG_FLIP_R90,     // TEXTUREFLIP_ROTATE90
        internal::FRAMESPECFLAG_FLIP_R180,    // TEXTUREFLIP_ROTATE180
        internal::FRAMESPECFLAG_FLIP_R270     // TEXTUREFLIP_ROTATE270
    };

    gfnd::ShaderInt flag = frameFlag[frame] | flipFlag[flip];
    gfnd::ShaderUniformId id = drawInfo.GetUniformId(UNIFORM_uFrameSpec);

    if (!vertexColorEnabled) { flag |= internal::VERTEX_EFFECT_CONST_COLOR; }

    shhelp::SetVertexUniformReg(id, flag);
    NW_GL_ASSERT();
}

//----------------------------------------
// @brief 水平ウィンドウのフレームのテクスチャ座標変換を設定します。
//
// @param gres グラフィックスリソースです。
// @param frame フレーム番号です。
// @param flip テクスチャフリップ番号です。
NW_INLINE void
SetupHorizontalFrameTransform(
    const DrawInfo& drawInfo,
    WindowFrame frame,
    TextureFlip flip,
    bool vertexColorEnabled = false)
{
    NW_ASSERTMSG(frame < WINDOWFRAME_LB, "out of bounds: frame[%d] < WINDOWFRAME_LB", frame);
    namespace shhelp = gfnd::shader_helper;

    static const u32 frameFlag[2] =
    {
        internal::FRAMESPECFLAG_FRAME_HORIZONTAL_L,
        internal::FRAMESPECFLAG_FRAME_HORIZONTAL_R,
    };

    static const u32 flipFlag[TEXTUREFLIP_MAX] =
    {
        internal::FRAMESPECFLAG_NORMAL,       // TEXTUREFLIP_NONE
        internal::FRAMESPECFLAG_FLIP_HFLIP,   // TEXTUREFLIP_FLIPH
        internal::FRAMESPECFLAG_FLIP_VFLIP,   // TEXTUREFLIP_FLIPV
        internal::FRAMESPECFLAG_FLIP_R90,     // TEXTUREFLIP_ROTATE90
        internal::FRAMESPECFLAG_FLIP_R180,    // TEXTUREFLIP_ROTATE180
        internal::FRAMESPECFLAG_FLIP_R270     // TEXTUREFLIP_ROTATE270
    };

    gfnd::ShaderInt flag =  frameFlag[frame] | flipFlag[flip];
    if (!vertexColorEnabled) { flag |= internal::VERTEX_EFFECT_CONST_COLOR; }

    gfnd::ShaderUniformId id = drawInfo.GetUniformId(UNIFORM_uFrameSpec);
    shhelp::SetVertexUniformReg(id, flag);
    NW_GL_ASSERT();
}

//----------------------------------------
// @brief フレームのテクスチャ座標変換を設定します。
//
// @param gres グラフィックスリソースです。
// @param frame フレーム番号です。
// @param flip テクスチャフリップ番号です。
NW_INLINE void
SetupHorizontalNocontextFrameTransform(
    const DrawInfo& drawInfo,
    WindowFrame frame,
    TextureFlip flip,
    bool vertexColorEnabled = false)
{
    NW_ASSERTMSG(frame < WINDOWFRAME_LB, "out of bounds: frame[%d] < WINDOWFRAME_LB", frame);
    namespace shhelp = gfnd::shader_helper;

    static const u32 frameFlag[WINDOWFRAME_MAX] =
    {
        internal::FRAMESPECFLAG_FRAME_HORIZONTAL_NOCONTENT_L,
        internal::FRAMESPECFLAG_FRAME_HORIZONTAL_NOCONTENT_R,
    };

    static const u32 flipFlag[TEXTUREFLIP_MAX] =
    {
        internal::FRAMESPECFLAG_NORMAL,       // TEXTUREFLIP_NONE
        internal::FRAMESPECFLAG_FLIP_HFLIP,   // TEXTUREFLIP_FLIPH
        internal::FRAMESPECFLAG_FLIP_VFLIP,   // TEXTUREFLIP_FLIPV
        internal::FRAMESPECFLAG_FLIP_R90,     // TEXTUREFLIP_ROTATE90
        internal::FRAMESPECFLAG_FLIP_R180,    // TEXTUREFLIP_ROTATE180
        internal::FRAMESPECFLAG_FLIP_R270     // TEXTUREFLIP_ROTATE270
    };

    gfnd::ShaderInt flag = frameFlag[frame] | flipFlag[flip];
    if (!vertexColorEnabled) { flag |= internal::VERTEX_EFFECT_CONST_COLOR; }

    gfnd::ShaderUniformId id = drawInfo.GetUniformId(UNIFORM_uFrameSpec);
    shhelp::SetVertexUniformReg(id, flag);
    NW_GL_ASSERT();
}

//----------------------------------------
NW_INLINE void
GetHorizontalLeftFrameSize(
    math::VEC2* pPt,
    Size* pSize,
    const math::VEC2& basePt,
    const Size& winSize,
    const WindowFrameSize& frameSize
)
{
    *pPt = basePt;

    pSize->width = frameSize.l;
    pSize->height = winSize.height;
}

//----------------------------------------
NW_INLINE void
GetHorizontalRightFrameSize(
    math::VEC2* pPt,
    Size* pSize,
    const math::VEC2& basePt,
    const Size& winSize,
    const WindowFrameSize& frameSize
)
{
    *pPt = math::VEC2(basePt.x + winSize.width - frameSize.r, basePt.y);

    pSize->width  = frameSize.r;
    pSize->height = winSize.height;
}

} // window_foundation

} // namespace nw::lyt
} // namespace nw

#endif // NW_LYT_WINDOW_FOUNDATION_H_

