﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_LYT_TEXTURECONTAINER_H_
#define NW_LYT_TEXTURECONTAINER_H_

#include <nw/ut/ut_LinkList.h>

#include <nw/lyt/lyt_Types.h>

namespace nw
{
namespace lyt
{

//---------------------------------------------------------------------------
//  テクスチャを保持するクラスです。
//
// TextureContainer
//
//---------------------------------------------------------------------------

//! @brief 内部用機能のため使用禁止です。
class TextureRefLink
{
    NW_DISALLOW_COPY_AND_ASSIGN(TextureRefLink);

public:

    //----------------------------------------
    //! @name コンストラクタ／デストラクタ
    //@{

    // コンストラクタです。

    //! @brief 内部用機能のため使用禁止です。
    TextureRefLink();


    // デストラクタです。
    // OpenGL のテクスチャオブジェクトを破棄します。

    //! @brief 内部用機能のため使用禁止です。
    ~TextureRefLink();

    // @}

    //----------------------------------------
    //! @name 設定／取得
    //@{

    // テクスチャ名を設定します。

    //! @brief 内部用機能のため使用禁止です。
    //!
    //! @param[in] resourceName テクスチャを特定する名前です。
    //!
    void SetName(
        const char* resourceName);

    // テクスチャのリソース名を取得します。

    //! @brief 内部用機能のため使用禁止です。
    //!
    //! @return テクスチャのリソース名を返します。
    //!
    const char* GetResourceName() const
    {
        return m_Name;
    }

    // テクスチャ情報を取得します。

    //! @brief 内部用機能のため使用禁止です。
    //!
    //! @return テクスチャ情報を返します。
    //!
    TextureInfo& GetTextureInfo()
    {
        return m_TexInfo;
    }

    // テクスチャ情報を取得します。

    //! @brief 内部用機能のため使用禁止です。
    //!
    //! @return テクスチャ情報を返します。
    //!
    const TextureInfo& GetTextureInfo() const
    {
        return m_TexInfo;
    }

    // @}

    // リンク情報です。

    //! @brief 内部用機能のため使用禁止です。
    ut::LinkListNode m_Link;

private:
    char m_Name[TexImageNameMax];
    TextureInfo m_TexInfo;
};

//---------------------------------------------------------------------------
//! @brief テクスチャを保持するクラスです。
//!
//! @details
//! ResourceAccessor の実装で OpenGL のテクスチャオブジェクトを保持するのに
//! 利用されます。
//!
//! @sa ResourceAccessor
//!
//---------------------------------------------------------------------------
class TextureContainer : public ut::LinkList<TextureRefLink, offsetof(TextureRefLink, m_Link)>
{
    NW_DISALLOW_COPY_AND_ASSIGN(TextureContainer);

public:
    //----------------------------------------
    //! @name コンストラクタ／デストラクタ
    //@{

    //! @brief コンストラクタです。
    //!
    TextureContainer() {}

    //! @brief デストラクタです。
    //!
    //! @details
    //! 登録されているすべてのテクスチャを破棄します。
    //!
    ~TextureContainer();

    //@}

    //----------------------------------------
    //! @name 登録／破棄
    //@{

    //! @brief 登録を抹消します。
    //!
    //! @details
    //! 登録されているすべてのテクスチャを破棄します。
    //!
    void Finalize();

    //! @brief テクスチャ情報を登録します。
    //!
    //! @param[in] name テクスチャの名前です。
    //!
    //! @return テクスチャ情報です。メモリ確保できない場合には NULL が返ります。
    //!
    TextureInfo* RegistTexture(const char* name);

    //! @brief テクスチャの登録を抹消します。
    //!
    //! @param[in] info テクスチャ情報です。
    //!
    void UnregistTexture(TextureInfo* info);

    //@}

    //----------------------------------------
    //! @name 検索
    //@{

    //! @brief テクスチャ情報を検索します。
    //!
    //! @param[in] name テクスチャのリソース名です。
    //!
    //! @return
    //! テクスチャ情報を返します。
    //!
    const TextureInfo* FindTextureByName(const char* name) const;

    //@}
};

} // namespace nw::lyt
} // namespace nw

#endif // NW_LYT_TEXTURECONTAINER_H_
