﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_LYT_DRAWINFO_H_
#define NW_LYT_DRAWINFO_H_

#include <nw/lyt/lyt_Types.h>

#include <nw/math/math_Types.h>
#include <nw/font/font_WideTextWriter.h>
#include <nw/ut/ut_Rect.h>
#include <nw/ut/ut_Preprocessor.h>
#include <nw/ut/ut_RuntimeTypeInfo.h>

#include <nw/gfnd/gfnd_ShaderHelper.h>

namespace nw
{
namespace lyt
{

class GraphicsResource;
class Layout;
#if defined(NW_PLATFORM_CAFE)
struct ShaderConnection;
#endif

//---------------------------------------------------------------------------
//! @brief 描画情報クラスです。
//!
//! @details
//! 全体的な描画方法の制御や、描画中の情報の受け渡しに使用されます。
//!
//! 描画の前に SetGraphicsResource() により GraphicsResource を
//! 設定しておく必要があります。
//!
//! @sa Layout
//!
//---------------------------------------------------------------------------
class DrawInfo
{
    NW_DISALLOW_COPY_AND_ASSIGN(DrawInfo);

public:
    //! 実行時型情報です。
    NW_UT_RUNTIME_TYPEINFO_ROOT();

    //----------------------------------------
    //! @name コンストラクタ／デストラクタ
    //@{

    //! @brief コンストラクタです。
    //!
    DrawInfo();

    //! @brief デストラクタです。
    //!
    virtual ~DrawInfo();

    //@}

    //----------------------------------------
    //! @name 設定／取得
    //@{

    //! @brief 現在のビュー行列を取得します。
    //!
    //! @return ビュー行列を返します。
    //!
    //! @sa SetViewMtx
    //!
    const math::MTX34&  GetViewMtx() const
    {
        return m_ViewMtx;
    }

    //! @brief ビュー行列を設定します。
    //!
    //! @param[in] value    ビュー行列です。
    //!
    //! @sa GetViewMtx
    //!
    void SetViewMtx(const math::MTX34& value)
    {
        m_ViewMtx = value;
    }

    //! @brief 位置調整処理を行うかどうかの設定を取得します。
    //!
    //! @return 位置調整処理を行う場合は true を返します。
    //!
    //! @sa SetLocationAdjust
    //!
    bool IsLocationAdjust() const
    {
        return m_Flag.locationAdjust;
    }

    //! @brief 位置調整処理を行うかどうかを設定します。
    //!
    //! @param[in] bEnable  位置調整処理を行う場合には true を指定します。
    //!
    //! @details
    //! 位置調整処理を有効にした場合、Pane::IsLocationAdjust() が true を
    //! を返すペインに位置調整処理のスケールが適用されます。
    //!
    //! 該当ペインの子孫ペインも位置調整処理のスケールの影響を受けます。
    //!
    //! @sa IsLocationAdjust
    //! @sa SetLocationAdjustScale
    //! @sa Pane::IsLocationAdjust
    //!
    void SetLocationAdjust(bool bEnable)
    {
        m_Flag.locationAdjust = bEnable;
    }

    //! @brief 位置調整処理のスケールを取得します。
    //!
    //! @return 位置調整処理のスケールを返します。
    //!
    //! @sa SetLocationAdjustScale
    //!
    const math::VEC2& GetLocationAdjustScale() const
    {
        return m_LocationAdjustScale;
    }

    //! @brief 位置調整処理のスケールを設定します。
    //!
    //! @param[in] scale    スケールです。
    //!
    //! @sa GetLocationAdjustScale
    //!
    void SetLocationAdjustScale(const math::VEC2& scale)
    {
        m_LocationAdjustScale = scale;
    }

    //! @brief 非表示のペインに対して行列計算を行うかどうかを判定します。
    //!
    //! @return 非表示のペインに対して行列計算を行う場合は true を返します。
    //!
    //! @sa SetInvisiblePaneCalculateMtx
    //!
    bool IsInvisiblePaneCalculateMtx() const
    {
        return m_Flag.invisiblePaneCalculateMtx;
    }

    //! @brief 非表示のペインに対して行列計算を行うかどうかを設定します。
    //!
    //! @param[in] bEnable  非表示のペインに対して行列計算を行う場合には true を指定します。
    //!
    //! @details
    //! bEnable に true を指定した場合、非表示のペインに対しても Pane::CalculateMtx() を
    //! 呼び出して行列計算を行います。
    //!
    //! @sa IsInvisiblePaneCalculateMtx
    //! @sa Pane::CalculateMtx
    //! @sa Pane::IsVisible
    //!
    void SetInvisiblePaneCalculateMtx(bool bEnable)
    {
        m_Flag.invisiblePaneCalculateMtx = bEnable;
    }

    //! @brief アルファがゼロのペインに対して行列計算を行うかどうかを判定します。
    //!
    //! @return アルファがゼロのペインに対して行列計算を行う場合は true を返します。
    //!
    //! @sa SetInvisiblePaneCalculateMtx
    //!
    bool IsAlphaZeroPaneCalculateMtx() const
    {
        return m_Flag.alphaZeroPaneCalculateMtx;
    }

    //! @brief アルファがゼロのペインに対して行列計算を行うかどうかを設定します。
    //!
    //! @param[in] bEnable  アルファがゼロのペインに対して行列計算を行う場合には true を指定します。
    //!
    //! @details
    //! bEnable に true を指定した場合、アルファがゼロのペインに対しても Pane::CalculateMtx() を
    //! 呼び出して行列計算を行います。
    //!
    //! @sa IsInvisiblePaneCalculateMtx
    //! @sa Pane::CalculateMtx
    //! @sa Pane::IsVisible
    //!
    void SetAlphaZeroPaneCalculateMtx(bool bEnable)
    {
        m_Flag.alphaZeroPaneCalculateMtx = bEnable;
    }

    //! @brief %GraphicsResource オブジェクトを取得します。
    //!
    //! @return GraphicsResource オブジェクトへのポインタを返します。
    //!
    //! @sa SetGraphicsResource
    //!
    GraphicsResource* GetGraphicsResource() const
    {
        return m_pGraphicsResource;
    }

    //! @brief %GraphicsResource オブジェクトを設定します。
    //!
    //! @param[in] pGraphicsResource    GraphicsResource オブジェクトへのポインタです。
    //!
    //! @sa GetGraphicsResource
    //!
    void SetGraphicsResource(GraphicsResource* pGraphicsResource)
    {
        m_pGraphicsResource = pGraphicsResource;
    }

    //! @brief 射影行列を設定します。
    //!
    //! @param[in] mtx  射影行列です。
    //!
    void SetProjectionMtx(const nw::math::MTX44& mtx);

    //@}

    // 描画中のレイアウトを取得します。
    // Layout::Draw() の実行中に現在描画しているレイアウトを取得します。

    //! @brief 内部用機能のため使用禁止です。
    //!
    //! @return レイアウトへのポインタを返します。
    //!
    Layout* GetLayout() { return m_pLayout; }

    // 描画中のレイアウトを取得します。

    //! @brief 内部用機能のため使用禁止です。
    //!
    //! @return 描画中のレイアウトです。
    //!
    const Layout* GetLayout() const { return m_pLayout; }

    // これから描画するレイアウトを設定します。

    //! @brief 内部用機能のため使用禁止です。
    //!
    //! @param[in] pLayout  これから描画するレイアウトです。
    //!
    void SetLayout(Layout* pLayout)
    {
        m_pLayout = pLayout;
    }

    // 射影行列を取得します。

    //! @brief 内部用機能のため使用禁止です。
    //!
    //! @return 射影行列です。
    //!
    const nw::math::MTX44& GetProjectionMtx() const
    {
        return m_ProjMtx;
    }

    // 射影行列を設定します。

    //! @brief 内部用機能のため使用禁止です。
    void LoadProjectionMtx();

    // モデルビュー行列を取得します。

    //! @brief 内部用機能のため使用禁止です。
    //!
    //! @return モデルビュー行列を返します。
    //!
    const math::MTX34&  GetModelViewMtx() const
    {
        return m_ModelViewMtx;
    }

    // モデルビュー行列を設定します。

    //! @brief 内部用機能のため使用禁止です。
    //!
    //! @param[in] m    モデルビューです。
    //!
    void SetModelViewMtx(const math::MTX34& m)
    {
        m_ModelViewMtx = m;
        m_ModelViewMtxLoaded = false;
    }

    // テクスチャが使用するテクスチャ座標の番号を取得します。

    //! @brief 内部用機能のため使用禁止です。
    //!
    //! @param[in] index    テクスチャのインデックスです。
    //!
    //! @return 使用するテクスチャ座標の番号です。
    //!
    int GetTexCoordSrc(int index) const
    {
        return m_TexCoordSrc[index];
    }

    // テクスチャが使用するテクスチャ座標の番号を設定します。

    //! @brief 内部用機能のため使用禁止です。
    //!
    //! @param[in] index    テクスチャのインデックスです。
    //! @param[in] src      使用するテクスチャ座標の番号です。
    //!
    void SetTexCoordSrc(int index, int src)
    {
        m_TexCoordSrc[index] = static_cast<s8>(src);
    }

    // 前回の描画で使用したTexEnvの段数を得ます。

    //! @brief 内部用機能のため使用禁止です。
    //!
    //! @return TexEnvの段数を返します。
    //!
    u32 GetNumTexEnv() const
    {
        return m_NumTexEnv;
    }

    // 描画中の情報のキャッシュを初期化します。

    //! @brief 内部用機能のため使用禁止です。
    void ResetGlProgramState()
    {
        this->SetNumTexEnv(internal::TexEnvUnitMax);
        this->SetTexEnvAuto(false);
    }

    // 描画中の情報のキャッシュを初期化します。

    //! @brief 内部用機能のため使用禁止です。
    void ResetGlState() { m_FirstDraw = true; }

    // 描画で使用するTexEnvの段数を記録します。

    //! @brief 内部用機能のため使用禁止です。
    //!
    //! @param[in] numTexEnv    TexEnvの段数です。
    //!
    void SetNumTexEnv(u32 numTexEnv)
    {
        m_NumTexEnv = static_cast<u8>(numTexEnv);
    }

    // 前回の描画でのTexEnvの設定が簡易設定によるものか調べます。

    //! @brief 内部用機能のため使用禁止です。
    //!
    //! @return 自動設定であれば true を返します。
    //!
    bool GetTexEnvAuto() const
    {
        return m_TexEnvAuto;
    }

    // 前回の描画で使用されたテクスチャ枚数を取得します。

    //! @brief 内部用機能のため使用禁止です。
    //!
    //! @return テクスチャ枚数を返します。
    //!
    u8 GetNumTexMap() const
    {
        return m_NumTexMap;
    }

    // TexEnvへの設定情報を記録します。

    //! @brief 内部用機能のため使用禁止です。
    //!
    //! @param[in] value        簡易設定であれば true を指定します。
    //! @param[in] numTexMap    使用するテクスチャ枚数を指定します。
    //!
    void SetTexEnvAuto(bool value, u8 numTexMap = 0)
    {
        m_TexEnvAuto = value;
        m_NumTexMap = numTexMap;
    }

    // モデルビュー行列をシェーダーにロードします。

    //! @brief 内部用機能のため使用禁止です。
    void LoadMtxModelView();

    // プログラムオブジェクトを設定します。

    //! @brief 内部用機能のため使用禁止です。
    void SetupProgram();

    // 許容可能なテクスチャ数を設定します。

    //! @brief 内部用機能のため使用禁止です。
    //!
    //! @param[in] quantity 許容可能なテクスチャ数です。
    //!
    void SetAllowableTextureQuantity(u32 quantity)
    {
        m_AllowableTextureQuantity = quantity;
    }

    // 許容可能なテクスチャ数を取得します。

    //! @brief 内部用機能のため使用禁止です。
    //!
    //! @return 許容可能なテクスチャ数です。
    //!
    u32 GetAllowableTextureQuantity() const
    {
        return m_AllowableTextureQuantity;
    }

    // シェーダープログラムを設定します。

    //! @brief 内部用機能のため使用禁止です。
    //!
    //! @param[in] id   シェーダープログラムです。
    //!
    void SetShader(int id = 0);

#if defined(NW_PLATFORM_CAFE)
    // アーカイブシェーダープログラムを設定します。

    //! @brief 内部用機能のため使用禁止です。
    void SetArchiveShader(const ArchiveShaderInfo* pShaderInfo, s32 shaderId);
#endif

    // キャッシュされた Uniform Location を取得します。

    //! @brief 内部用機能のため使用禁止です。
    //!
    //! @param[in] uniformName  Uniform 列挙型の値を指定します。
    //!
    //! @return Uniform Location を返します。
    //!
    gfnd::ShaderUniformId GetUniformId(int uniformName) const;

private:
    math::MTX44 m_ProjMtx;
    math::MTX34 m_ViewMtx;
    math::MTX34 m_ModelViewMtx;
    math::VEC2 m_LocationAdjustScale;
    GraphicsResource* m_pGraphicsResource;
    Layout* m_pLayout;

    s8 m_TexCoordSrc[TexMapMax];
    u8 m_NumTexEnv;
    u8 m_NumTexMap;
    bool m_TexEnvAuto;
    bool m_ModelViewMtxLoaded;
    bool m_FirstDraw;
    u32 m_AllowableTextureQuantity;

#if defined(NW_PLATFORM_CAFE)
    ShaderConnection* m_CurrentShader;
#endif

    struct Flag
    {
        u8 locationAdjust : 1;
        u8 invisiblePaneCalculateMtx : 1;
        u8 alphaZeroPaneCalculateMtx : 1;
    };

    Flag m_Flag;
};

} // namespace nw::lyt
} // namespace nw

#endif // NW_LYT_DRAWINFO_H_

