﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_LYT_COMMON_H_
#define NW_LYT_COMMON_H_

#include <nw/lyt/lyt_Types.h>
#include <nw/ut/ut_Color.h>
#include <nw/lyt/lyt_GraphicsResource.h>
#include <nw/lyt/lyt_DrawInfo.h>
#include <nw/gfnd/gfnd_ShaderHelper.h>

#include <functional>

#if defined(NW_PLATFORM_CTR)
#define NW_LYT_WRITEONLY  __writeonly
#else
#define NW_LYT_WRITEONLY
#endif

// ビルドターゲットにかかわらず出力します。
#define NW_LYT_PRINT NW_LOG

namespace nw
{
namespace lyt
{

class Material;
struct Size;
class DrawInfo;
class GraphicsResource;
class Layout;
class ResourceAccessor;
class TextSearcher;
class ControlCreator;
class Pane;

namespace res
{

struct TextureList;
struct FontList;
struct MaterialList;
struct PartsPaneBasicInfo;
struct ExtUserDataList;
struct Material;

}   // namespace res

//---------------------------------------------------------------------------
//! @brief 構築時に必要なリソースへのポインタを持つ構造体です。
//!
//---------------------------------------------------------------------------
struct BuildResSet
{
    //! テクスチャリソースのリストです。
    const res::TextureList* pTextureList;
    //! フォントリソースのリストです。
    const res::FontList* pFontList;
    //! マテリアルリソースのリストです。
    const res::MaterialList* pMaterialList;
    //! リソースアクセサです。
    ResourceAccessor* pResAccessor;
    //! このリソースを保持するレイアウトクラスです。
    Layout* pLayout;
};

//---------------------------------------------------------------------------
//! @brief 構築時に必要なデータを保持する構造体です。
//!
//---------------------------------------------------------------------------
struct BuildArgSet
{
    //! ペインのサイズに全て共通でかける倍率です。
    math::VEC2 magnify;
    //! 部品としてのサイズです。
    math::VEC2 partsSize;
    //! コントロールを作成するクラスです。
    ControlCreator* pControlCreator;
    //! テキストの検索を行うクラスです。
    TextSearcher* pTextSearcher;
    //! 現在構築を行っているレイアウトクラスです。
    Layout* pLayout;
    //! 現在構築を行っているボディレイアウトクラスです。
    Layout* pBodyLayout;
    //! 現在構築を行っているレイアウトのリソースセットです。
    const BuildResSet* pCurrentBuildResSet;
    //! 上書きを行っているレイアウトのリソースセットです。パーツレイアウトでない場合はNULLになります。
    const BuildResSet* pOverrideBuildResSet;
    //! 部分的な上書きを行う場合のフラグです。
    u16 overrideUsageFlag;
    //! 基本情報の上書きを行う場合のフラグです。
    u16 overrideBasicUsageFlag;
    //! マテリアルの部分上書きを行う場合のフラグです。
    u16 overrideMaterialUsageFlag;
    //! ペイン基本情報を上書きする場合に、データの参照先を指します。
    const res::PartsPaneBasicInfo* pOverridePartsPaneBasicInfo;
    //! 親ペインへのポインタです。
    Pane* pParentPane;
    //! 拡張ユーザーデータリストです。
    const res::ExtUserDataList* pExtUserDataList;
    //! テキストボックスの文字属性バッファをレイアウト内で共有するか否かです。
    bool isShareTextBoxCharAttributeBuffer;
    //! ルートペインを部品ペインにするか否かです。
    bool isRootPaneParts;
#if ! defined(NW_RELEASE)
    //! 親レイアウトの影響で、ビューアで非表示状態になっているか否かです。
    bool isViewerInvisibleByParent;
#endif
};

namespace internal
{

enum FrameSpecFlag
{
    VERTEX_EFFECT_TEXCOORD_ENABLED = 1 << 0,

    VERTEX_EFFECT_FRAME = 1 << 1,
    VERTEX_EFFECT_TEXCOORD_ALIGN_RIGHT = 1 << 2,
    VERTEX_EFFECT_TEXCOORD_ALIGN_BOTTOM = 1 << 3,
    VERTEX_EFFECT_DOTBYDOT_U = 1 << 4,
    VERTEX_EFFECT_DOTBYDOT_V = 1 << 5,

    VERTEX_EFFECT_FRAME_ID_SHIFT = 8,
    VERTEX_EFFECT_FRAME_ID_MASK = 0xf << VERTEX_EFFECT_FRAME_ID_SHIFT, // 9 ～ 12 ビット目
    VERTEX_EFFECT_TEXCOORD_SWAP = 1 << 16,
    VERTEX_EFFECT_TEXCOORD_HFLIP = 1 << 17,
    VERTEX_EFFECT_TEXCOORD_VFLIP = 1 << 18,

    VERTEX_EFFECT_CONST_COLOR = 1 << 30,

    FRAME_ID_LT = WINDOWFRAME_LT << VERTEX_EFFECT_FRAME_ID_SHIFT,
    FRAME_ID_RT = WINDOWFRAME_RT << VERTEX_EFFECT_FRAME_ID_SHIFT,
    FRAME_ID_LB = WINDOWFRAME_LB << VERTEX_EFFECT_FRAME_ID_SHIFT,
    FRAME_ID_RB = WINDOWFRAME_RB << VERTEX_EFFECT_FRAME_ID_SHIFT,
    FRAME_ID_L = WINDOWFRAME_L << VERTEX_EFFECT_FRAME_ID_SHIFT,
    FRAME_ID_R = WINDOWFRAME_R << VERTEX_EFFECT_FRAME_ID_SHIFT,
    FRAME_ID_T = WINDOWFRAME_T << VERTEX_EFFECT_FRAME_ID_SHIFT,
    FRAME_ID_B = WINDOWFRAME_B << VERTEX_EFFECT_FRAME_ID_SHIFT,
    FRAME4_ID_LT = (WINDOWFRAME_B + 1) << VERTEX_EFFECT_FRAME_ID_SHIFT,
    FRAME4_ID_RT = (WINDOWFRAME_B + 2) << VERTEX_EFFECT_FRAME_ID_SHIFT,
    FRAME4_ID_LB = (WINDOWFRAME_B + 3) << VERTEX_EFFECT_FRAME_ID_SHIFT,
    FRAME4_ID_RB = (WINDOWFRAME_B + 4) << VERTEX_EFFECT_FRAME_ID_SHIFT,
    FRAME2_ID_L = (WINDOWFRAME_B + 5) << VERTEX_EFFECT_FRAME_ID_SHIFT,
    FRAME_ID_C = (WINDOWFRAME_B + 6) << VERTEX_EFFECT_FRAME_ID_SHIFT, // Content

    FRAMESPECFLAG_CONTENT = FRAME_ID_C,
    FRAMESPECFLAG_FRAME_HORIZONTAL_L = FRAME_ID_L | VERTEX_EFFECT_TEXCOORD_ENABLED | VERTEX_EFFECT_FRAME,
    FRAMESPECFLAG_FRAME_HORIZONTAL_R = FRAME_ID_R | VERTEX_EFFECT_TEXCOORD_ENABLED | VERTEX_EFFECT_FRAME,
    FRAMESPECFLAG_FRAME_HORIZONTAL_NOCONTENT_L = FRAME2_ID_L | VERTEX_EFFECT_TEXCOORD_ENABLED | VERTEX_EFFECT_FRAME | VERTEX_EFFECT_DOTBYDOT_U,
    FRAMESPECFLAG_FRAME_HORIZONTAL_NOCONTENT_R = FRAME_ID_R | VERTEX_EFFECT_TEXCOORD_ENABLED | VERTEX_EFFECT_FRAME,
    FRAMESPECFLAG_FRAME_LT = FRAME_ID_LT | VERTEX_EFFECT_TEXCOORD_ENABLED | VERTEX_EFFECT_FRAME,
    FRAMESPECFLAG_FRAME_RT = FRAME_ID_RT | VERTEX_EFFECT_TEXCOORD_ENABLED | VERTEX_EFFECT_FRAME,
    FRAMESPECFLAG_FRAME_LB = FRAME_ID_LB | VERTEX_EFFECT_TEXCOORD_ENABLED | VERTEX_EFFECT_FRAME,
    FRAMESPECFLAG_FRAME_RB = FRAME_ID_RB | VERTEX_EFFECT_TEXCOORD_ENABLED | VERTEX_EFFECT_FRAME,
    FRAMESPECFLAG_FRAME_L = FRAME_ID_L | VERTEX_EFFECT_TEXCOORD_ENABLED | VERTEX_EFFECT_FRAME | VERTEX_EFFECT_DOTBYDOT_V | VERTEX_EFFECT_TEXCOORD_ALIGN_BOTTOM,
    FRAMESPECFLAG_FRAME_R = FRAME_ID_R | VERTEX_EFFECT_TEXCOORD_ENABLED | VERTEX_EFFECT_FRAME | VERTEX_EFFECT_DOTBYDOT_V,
    FRAMESPECFLAG_FRAME_T = FRAME_ID_T | VERTEX_EFFECT_TEXCOORD_ENABLED | VERTEX_EFFECT_FRAME | VERTEX_EFFECT_DOTBYDOT_U,
    FRAMESPECFLAG_FRAME_B = FRAME_ID_B | VERTEX_EFFECT_TEXCOORD_ENABLED | VERTEX_EFFECT_FRAME | VERTEX_EFFECT_DOTBYDOT_U | VERTEX_EFFECT_TEXCOORD_ALIGN_RIGHT,
    FRAMESPECFLAG_FRAME4_LT = FRAME4_ID_LT | VERTEX_EFFECT_TEXCOORD_ENABLED | VERTEX_EFFECT_FRAME | VERTEX_EFFECT_DOTBYDOT_U,
    FRAMESPECFLAG_FRAME4_RT = FRAME4_ID_RT | VERTEX_EFFECT_TEXCOORD_ENABLED | VERTEX_EFFECT_FRAME | VERTEX_EFFECT_DOTBYDOT_V,
    FRAMESPECFLAG_FRAME4_LB = FRAME4_ID_LB | VERTEX_EFFECT_TEXCOORD_ENABLED | VERTEX_EFFECT_FRAME | VERTEX_EFFECT_DOTBYDOT_V | VERTEX_EFFECT_TEXCOORD_ALIGN_BOTTOM,
    FRAMESPECFLAG_FRAME4_RB = FRAME4_ID_RB | VERTEX_EFFECT_TEXCOORD_ENABLED | VERTEX_EFFECT_FRAME | VERTEX_EFFECT_DOTBYDOT_U | VERTEX_EFFECT_TEXCOORD_ALIGN_RIGHT,
    FRAMESPECFLAG_FLIP_HFLIP = VERTEX_EFFECT_TEXCOORD_HFLIP,
    FRAMESPECFLAG_FLIP_VFLIP = VERTEX_EFFECT_TEXCOORD_VFLIP,
    FRAMESPECFLAG_FLIP_R90 = VERTEX_EFFECT_TEXCOORD_HFLIP | VERTEX_EFFECT_TEXCOORD_SWAP,
    FRAMESPECFLAG_FLIP_R180 = VERTEX_EFFECT_TEXCOORD_HFLIP | VERTEX_EFFECT_TEXCOORD_VFLIP,
    FRAMESPECFLAG_FLIP_R270 = VERTEX_EFFECT_TEXCOORD_VFLIP | VERTEX_EFFECT_TEXCOORD_SWAP,
    FRAMESPECFLAG_NORMAL = 0
};

template <int StrMax>
bool EqualsName(const char* name1, const char* name2)
{
    for (int i = 0; i < StrMax; i++) {
        if (name1[i] != name2[i]) {
            return false;
        } else if (name1[i] == '\0') {
            return true;
        }
    }
    return true;
}

NW_INLINE bool EqualsResName(const char* name1, const char* name2)
{
    return EqualsName<ResourceNameStrMax>(name1, name2);
}

NW_INLINE bool EqualsMaterialName(const char* name1, const char* name2)
{
    return EqualsName<MaterialNameStrMax>(name1, name2);
}

NW_INLINE bool EqualsFontName(const char* name1, const char* name2)
{
    return EqualsName<FontNameMax>(name1, name2);
}

NW_INLINE bool EqualsTexImageName(const char* name1, const char* name2)
{
    return EqualsName<TexImageNameMax>(name1, name2);
}

NW_INLINE bool EqualsShaderName(const char* name1, const char* name2)
{
    return EqualsName<ShaderNameMax>(name1, name2);
}

NW_INLINE bool EqualsLayoutName(const char* name1, const char* name2)
{
    return EqualsName<LayoutNameStrMax>(name1, name2);
}

NW_INLINE bool EqualsGroupName(const char* name1, const char* name2)
{
    return EqualsName<GroupNameStrMax>(name1, name2);
}

NW_INLINE const char*
GetStrTableStr(const void* pStrTable, int index)
{
    const ut::ResU32* offsets = static_cast<const ut::ResU32*>(pStrTable);
    const char* stringPool = static_cast<const char*>(pStrTable);

    return &stringPool[offsets[index]];
}

// 矩形のテクスチャ座標を複数個保持するクラスです。
class TexCoordAry
{
public:
    // コンストラクタです。
    TexCoordAry();

    // 保持可能なセット数が０か調べます。
    bool IsEmpty() const
    {
        return m_Cap == 0;
    }

    // メンバーを解放し、保持可能なセット数を０に設定します。
    void Free();

    // 指定のセット数を保持するのに十分なメモリを確保します。
    void Reserve(u8 num);

    // 保持しているセット数を取得します。
    u8 GetSize() const
    {
        return m_Num;
    }

    // 保持するセット数を設定し、座標を初期化します。
    void SetSize(u8 num);

    // テクスチャ座標配列の先頭アドレスを取得します。
    const TexCoordQuad* GetArray() const
    {
        return m_pData;
    }

    // 指定したセットのテクスチャ座標配列の先頭アドレスを取得します。
    void GetCoord(
        u32 idx,
        TexCoordQuad coord) const;

    // 指定したセットのテクスチャ座標を設定します。
    void SetCoord(
        u32 idx,
        const TexCoordQuad coord);

    // リソースのテクスチャ座標をコピーします。
    void Copy(
        const void* pResTexCoord,
        u8 texCoordNum);

private:
    u8 m_Cap;
    u8 m_Num;

    math::VEC2 (*m_pData)[VERTEX_MAX];
};

//----------------------------------------
#if 0 // 現在未使用の為。必要なさそうであれば削除する。
template<typename TType>
class FlagList
{
public:
    FlagList() : m_Flags(0) {}

    bool IsEnabled(int index)
    {
        return (m_Flags >> index) & 0x1;
    }

    void SetFlag(int index)
    {
        u32 bits = 0x1 << static_cast<int>(index);
        m_Flags = flag
            ? m_Flags | bits
            : m_Flags & ~bits;
    }

private:
    TType m_Flags;
};

//----------------------------------------
NW_INLINE const ut::Color8 MultipleAlpha(
                    const ut::Color8   col,
                    u8                  alpha);
{
    ut::Color8 ret = col;
    if (alpha != ut::Color8::ALPHA_MAX)
    {
        ret.a = u8(col.a * alpha / ut::Color8::ALPHA_MAX);
    }
    return ret;
}
#endif

//----------------------------------------
const res::Material *const
GetResMaterial(
        const BuildResSet* buildResSet,
        u16 materialIdx);

//----------------------------------------
template<std::size_t TSize>
NW_INLINE bool
TestVertexColorEnabled(const ut::Color8 (&colors)[TSize])
{
    NW_STATIC_ASSERT(4 <= TSize);
    const ut::Color8* end = colors + TSize;
    const ut::Color8* found = std::find_if(
        colors,
        end,
        std::bind2nd(std::not_equal_to<ut::Color8>(), ut::Color8(255, 255, 255, 255)));

    return found != end;
}

//----------------------------------------
template<std::size_t TSize>
NW_INLINE void
SetupVertexColor(const DrawInfo& drawInfo, const ut::Color8 (&colors)[TSize])
{
    NW_STATIC_ASSERT(4 <= TSize);

    gfnd::ShaderUniformId uid = drawInfo.GetUniformId(UNIFORM_uVertexColor);

#if defined(NW_PLATFORM_CAFE)
    if (uid != GX2_UNIFORM_VAR_INVALID_OFFSET)
#endif
    {

#define COLOR_TO_F32_4(x) static_cast<f32>(x.r), static_cast<f32>(x.g), static_cast<f32>(x.b), static_cast<f32>(x.a)
        math::MTX44 colorsMatrix(
            COLOR_TO_F32_4(colors[0]),
            COLOR_TO_F32_4(colors[1]),
            COLOR_TO_F32_4(colors[2]),
            COLOR_TO_F32_4(colors[3])
            );
#undef COLOR_TO_F32_4

#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
        glUniformMatrix4fv(uid, 1, GL_FALSE, colorsMatrix.a);
        NW_GL_ASSERT();
#elif defined(NW_PLATFORM_CAFE)
        GX2SetVertexUniformReg(uid, 4 * 4, colorsMatrix);
#endif

    }
}

//----------------------------------------
NW_INLINE void
SetupFrameSize(const DrawInfo& drawInfo, const Size& size, const WindowFrameSize& frameSize)
{
    namespace shhelp = gfnd::shader_helper;
    shhelp::SetVertexUniformReg(
        drawInfo.GetUniformId(UNIFORM_uPaneSize), size.width, size.height);
    shhelp::SetVertexUniformReg(
        drawInfo.GetUniformId(UNIFORM_uFrameSize),
        static_cast<f32>(frameSize.l),
        static_cast<f32>(frameSize.r),
        static_cast<f32>(frameSize.t),
        static_cast<f32>(frameSize.b));
}

//----------------------------------------
void            DrawQuad(
                    DrawInfo&     drawInfo,
                    const math::VEC2&   basePt,
                    const Size&         size);

//----------------------------------------
void            DrawQuadWithTexCoords(
                    DrawInfo& drawInfo,
                    const math::VEC2& basePt,
                    const Size& size,
                    u8 texCoordNum,
                    const math::VEC2 (*texCoords)[VERTEX_MAX]);

//----------------------------------------
//! @brief 前回と同じ設定で描画を繰り返す
void            DrawQuad_Repeat(
                    const DrawInfo&     drawInfo,
                    const math::VEC2&   basePt,
                    const Size&         size);

//----------------------------------------
void            DrawBox(
                    DrawInfo&           drawInfo,
                    const math::VEC2&   pos,
                    const Size&         size,
                    ut::Color8         color);

//----------------------------------------
NW_INLINE
u8
GetVtxColorElement(
    const ut::Color8 cols[],
    u32             idx
)
{
    NW_ASSERT(idx < ANIMTARGET_VERTEXCOLOR_MAX);

    return reinterpret_cast<const u8*>(&cols[idx / sizeof(ut::Color8)])[idx % sizeof(ut::Color8)];
}

//----------------------------------------
NW_INLINE
void
SetVtxColorElement(
    ut::Color8 cols[],
    u32         idx,
    u8          value
)
{
    NW_ASSERT(idx < ANIMTARGET_VERTEXCOLOR_MAX);

    reinterpret_cast<u8*>(&cols[idx / sizeof(ut::Color8)])[idx % sizeof(ut::Color8)] = value;
}

//----------------------------------------
NW_INLINE
HorizontalPosition
GetHorizontalPosition(u8 var)
{
    return static_cast<HorizontalPosition>(var & 0x3);
}

//----------------------------------------
NW_INLINE
VerticalPosition
GetVerticalPosition(u8 var)
{
    return static_cast<VerticalPosition>((var >> 2) & 0x3);
}

//----------------------------------------
NW_INLINE
void
SetHorizontalPosition(u8* pVar, u8 newVal)
{
    NW_ASSERT(newVal < HORIZONTALPOSITION_MAX);

    *pVar = (*pVar & (~0x3)) | newVal;
}

//----------------------------------------
NW_INLINE
void
SetVerticalPosition(u8* pVar, u8 newVal)
{
    NW_ASSERT(newVal < VERTICALPOSITION_MAX);

    *pVar = (*pVar & (~0xc)) | (newVal << 2);
}

} // namespace nw::lyt::internal
} // namespace nw::lyt
} // namespace nw

#endif // NW_LYT_COMMON_H_
