﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_LYT_ANIMATOR_H_
#define NW_LYT_ANIMATOR_H_

#include <nw/lyt/lyt_Animation.h>
#include <nw/ut/ut_BitFlag.h>

namespace nw
{
namespace lyt
{

class GroupContainer;

//---------------------------------------------------------------------------
//! @brief アニメフレームを制御する機能を持ったAnimTransformです。
//!
//---------------------------------------------------------------------------
class Animator : public AnimTransformBasic
{
public:
    //! 実行時型情報です。
    NW_UT_RUNTIME_TYPEINFO(AnimTransformBasic);

    //---------------------------------------------------------------------------
    //! @brief アニメをどのように再生するかを表します。
    //---------------------------------------------------------------------------
    enum PlayType {
        PLAYTYPE_ONESHOT,   //!< 最後（先頭）までいって停止
        PLAYTYPE_LOOP,      //!< 最後（先頭）までいったら先頭（最後）から
        PLAYTYPE_ROUNDTRIP  //!< 最後（先頭）までいったら往復
    };

    //----------------------------------------
    //! @name コンストラクタ／デストラクタ
    //@{

    //! @brief コンストラクタです。
    //!
    //! @details
    //! アニメは停止状態です。
    //!
    Animator();

    //! @brief デストラクタです。
    //!
    virtual ~Animator();

    //@}

    //----------------------------------------
    //! @name 再生
    //@{
    //! @brief アニメを先頭から再生します。
    //!
    //! @details
    //! speedに負の値を入れた場合は、先頭から再生するのではなく、最後から逆再生します。
    //!
    //! 現在のフレームがどのような値であっても、順再生の場合は0に、逆再生の場合は最大
    //! フレーム値にセットされます。
    //!
    //! また、アニメーションがWaitアニメのときは、再生タイプはPLAYTYPE_ONESHOTのみ指定
    //! できます。speedの値がどのような値でも1フレームで再生が完了し、フレーム位置は0
    //! になります。
    //!
    //! @param[in] type     再生タイプを指定する
    //! @param[in] speed    再生スピードを指定する。単位は一回のUpdateFrameで何フレーム進めるかになる。
    //!                     つまり、1.fを指定するとデザイナが指定したのと同じスピード。
    //!                     負の値を指定すると逆再生になる。
    //!                     なお、スピードの絶対値はGetFrameMax()の数値以上にすることはできません。
    //!                     そのような場合はアサートされます。製品版では再生が行われません。
    //!
    virtual void Play(PlayType type, f32 speed);

    //! @brief アニメを、レイアウトエディタで指定された再生タイプで先頭から再生します。
    //!
    //! @details
    //! レイアウトエディタでループが選ばれたとき(IsLoopData()がtrueを返すとき)はPLAYTYPE_LOOP
    //! になり、それ以外はPLAYTYPE_ONESHOTになります。
    //!
    //! speedに負の値を入れた場合は、先頭から再生するのではなく、最後から逆再生します。
    //!
    //! 現在のフレームがどのような値であっても、順再生の場合は0に、逆再生の場合は最大
    //! フレーム値にセットされます。
    //!
    //! また、アニメーションがWaitアニメのときは、speedの値がどのような値でも1フレームで
    //! 再生が完了し、フレーム位置は0になります。
    //!
    //! @param[in] speed    再生スピードを指定する。単位は一回のUpdateFrameで何フレーム進めるかになる。
    //!                     つまり、1.fを指定するとデザイナが指定したのと同じスピード。
    //!                     負の値を指定すると逆再生になる。
    //!                     なお、スピードの絶対値はGetFrameMax()の数値以上にすることはできません。
    //!                     そのような場合はアサートされます。製品版では再生が行われません。
    //!
    virtual void PlayAuto(f32 speed);

    //! @brief アニメを現在のフレームから再生します。
    //!
    //! @details
    //! speedに負の値を入れた場合は逆再生になります。
    //!
    //! このメソッドでは、playのように現在のフレームが変わることはありません。
    //!
    //! また、アニメーションがWaitアニメのときは、再生タイプはPLAYTYPE_ONESHOTのみ指定
    //! できます。speedの値がどのような値でも1フレームで再生が完了し、フレーム位置は0
    //! になります。
    //!
    //! @param[in] type     再生タイプを指定する
    //! @param[in] speed    再生スピードを指定する。単位は一回のUpdateFrameで何フレーム進めるかになる。
    //!                     つまり、1.fを指定するとデザイナが指定したのと同じスピード。
    //!                     負の値を指定すると逆再生になる。
    //!                     なお、スピードの絶対値はGetFrameMax()の数値以上にすることはできません。
    //!                     そのような場合はアサートされます。製品版では再生が行われません。
    //!
    virtual void PlayFromCurrent(PlayType type, f32 speed);
    //@}

    //----------------------------------------
    //! @name 停止
    //@{
    //! @brief アニメーションの再生を指定したフレームで停止します。
    //!
    //! @details
    //! これはアニメが再生されないわけではなく、「停止した状態が再生されている」
    //! ことにご注意ください。
    //!
    //! @param[in] frame    停止したいフレーム
    //!
    virtual void Stop(f32 frame);

    //! @brief アニメーションの再生を現在のフレームで停止します。
    //!
    virtual void StopCurrent();

    //! @brief アニメーションの再生を先頭のフレームで停止します。
    //!
    virtual void StopAtMin();

    //! @brief アニメーションの再生を最後のフレームで停止します。
    //!
    virtual void StopAtMax();
    //@}

    //----------------------------------------
    //! @name フレーム進行
    //@{
    //! @brief アニメのフレームを進めます。
    //!
    //! @param[in] progress_frame 進めるフレーム数です。
    //!
    virtual void UpdateFrame(f32 progress_frame = 1.0f);
    //@}

    //----------------------------------------
    //! @name メンバへのアクセス
    //@{
    // 現在のフレームへのアクセスは、AnimTransformクラスのGetFrame及びSetFrameを利用します。
    // ただし、アニメの再生中にSetFrameを呼ぶとアニメを再生したままフレームを飛ばすような挙動に
    // なりますので、特定のフレームで止めたい場合はStopメソッドを使用してください。

    //! @brief 現在のスピードを返します。
    //!
    //! @return 現在のスピード。アニメを再生していないときは0.fが返されます。
    //!
    f32 GetSpeed() const { return m_Speed; }

    //! @brief アニメーションが再生中か否かを返します。
    //!
    //! @return アニメーションが再生中のときはtrue。そうでないときはfalse。
    //!
    bool IsPlaying() const { return (m_Speed != 0.f); }

    //! @brief 現在のフレームが先頭か否かを返します。
    //!
    //! @return 現在のフレームが先頭であればtrue。そうでなければfalse。
    //!
    bool IsMinFrame() const { return (GetFrame() == 0.f); }

    //! @brief 現在のフレームが最後か否かを返します。
    //!
    //! @return 現在のフレームが最後であればtrue。そうでなければfalse。
    //!
    bool IsMaxFrame() const { return (GetFrame() == GetFrameSize()); }

    //! @brief 現在のフレームが先頭でも最後でもないときにtrueを返します。
    //!
    //! @details
    //! ( ! IsMinFrame() && ! IsMaxFrame()) のショートカットです。
    //!
    //! @return 現在のフレームが先頭でも最後でもないときにtrue。そうでなければfalse。
    //!
    bool IsFrameMidway() const { return (GetFrame() != 0.f && GetFrame() != GetFrameSize()); }

    //! @brief 再生のタイプを返します。
    //!
    //! @return 再生のタイプ
    //!
    PlayType GetPlayType() const { return m_PlayType; }
    //@}

    //----------------------------------------
    //! @name 再生状態トリガ検出
    //@{
    //! @brief UpdateFrameの後、PlayTypeがPLAYTYPE_LOOPかPLAYTYPE_ROUNDTRIPのときに、最後のフレームを回ったことを示します。
    //!
    //! @details
    //! 次にUpdateFrameしたときにはこのメソッドはfalseを返しますのでご注意ください。
    //!
    //! @return 最後のフレームを回ったか否か
    //!
    bool IsTurnedMaxTrigger() const { return m_Flag.IsMaskOn(MASK_PLAY_STATUS_TURNED_MAX); }

    //! @brief UpdateFrameの後、PlayTypeがPLAYTYPE_LOOPかPLAYTYPE_ROUNDTRIPのときに、先頭のフレームを回ったことを示します。
    //!
    //! @details
    //! 次にUpdateFrameしたときにはこのメソッドはfalseを返しますのでご注意ください。
    //!
    //! @return 先頭のフレームを回ったか否か
    //!
    bool IsTurnedMinTrigger() const { return m_Flag.IsMaskOn(MASK_PLAY_STATUS_TURNED_MIN); }

    //! @brief UpdateFrameの後、PlayTypeがPLAYTYPE_ONESHOTのときに、再生が終了したことを示します。
    //!
    //! @details
    //! IsMaxFrameやIsMinFrameとの違いは、speedに正の値を与えたときも、負の値を与えたときも、
    //! このメソッドで終了したことを検知できることです。
    //!
    //! 明示的にStop等のアニメ停止メソッドを呼んだときには、このフラグは立たないことにご注意ください。
    //! アニメが再生中か否かを知りたい場合は、IsPlayingメソッドを使用してください。
    //!
    //! 次にUpdateFrameしたときにはこのメソッドはfalseを返しますのでご注意ください。
    //!
    //! @return 再生が終了したか否か
    //!
    bool IsEndTrigger() const { return m_Flag.IsMaskOn(MASK_PLAY_STATUS_END); }
    //@}

    //----------------------------------------
    //! @name バインド解除
    //@{
    //! @brief アニメータのペイン等へのバインドを解除します。
    //!
    //! @details
    //! Animatorクラスではバインドの解除はデストラクタでは行いませんので、明示的に
    //! このメソッドを呼び出す必要があります。
    //!
    virtual void Unbind() = 0;
    //@}

protected:
    //---------------------------------------------------------------------------
    //! @brief 再生状態のフラグです。
    //---------------------------------------------------------------------------
    enum FlagMask {
        MASK_PLAY_STATUS_END = 0x1,         //!< PLAYTYPE_ONESHOT再生が終了した
        MASK_PLAY_STATUS_TURNED_MAX = 0x2,  //!< PLAYTYPE_LOOPまたはPLAYTYPE_ROUNDTRIPで、最後のフレームを回った
        MASK_PLAY_STATUS_TURNED_MIN = 0x4   //!< PLAYTYPE_LOOPまたはPLAYTYPE_ROUNDTRIPで、先頭のフレームを回った
    };

    //!
    //! 再生状態のフラグをクリアします。
    //!
    void ClearPlayStatusFlag()
    {
        m_Flag.SetMaskOff(MASK_PLAY_STATUS_END | MASK_PLAY_STATUS_TURNED_MAX | MASK_PLAY_STATUS_TURNED_MIN);
    }

    f32 m_Speed;            //!< 再生スピード
    PlayType m_PlayType;    //!< 再生タイプ
    ut::BitFlag32 m_Flag;   //!< 再生状態フラグ。FlagMaskのマスク値となります。

};

//---------------------------------------------------------------------------
//! @brief ペインをアニメーションさせるAnimatorです。
//!
//---------------------------------------------------------------------------
class PaneAnimator : public Animator
{
public:
    //! 実行時型情報です。
    NW_UT_RUNTIME_TYPEINFO(Animator);

    PaneAnimator()
     : Animator()
     , mPane(NULL)
    {}

    //! @brief 指定したペインにアニメーションをバインドし、内部にペインのポインタを保持します。
    //!
    //! @param[in] pane     バインドしたいペイン
    //! @param[in] enable   開始時にアニメを有効状態にするか
    //!
    void Setup(Pane* pane, bool enable);

    //! @brief Setupで関連付けたアニメーションをバインド解除します。
    //!
    virtual void Unbind();

    //! @brief アニメーションがバインドされているペインを取得します。
    //!
    //! @return バインドされているペイン
    //!
    Pane* GetPane() const { return mPane; }

protected:
    Pane* mPane;        //!< 関連づけられたペイン

};

//---------------------------------------------------------------------------
//! @brief グループをアニメーションさせるAnimatorです。
//!
//---------------------------------------------------------------------------
class GroupAnimator : public Animator
{
public:
    //! 実行時型情報です。
    NW_UT_RUNTIME_TYPEINFO(Animator);

    GroupAnimator()
     : Animator()
     , mGroup(NULL)
    {}

    //! @brief 指定したグループにアニメーションをバインドし、内部にグループのポインタを保持します。
    //!
    //! @param[in] group    バインドしたいグループ
    //! @param[in] enable   開始時にアニメを有効状態にするか
    //!
    void Setup(Group* group, bool enable);

    //! @brief Setupで関連付けたアニメーションをバインド解除します。
    //!
    virtual void Unbind();

    //! @brief 指定したアニメーションリソースに関連づけられているグループのうち、index番目のグループに
    //!        アニメーションをバインドし、内部にグループのポインタを保持します。
    //!
    //! @param[in] animRes          構築するアニメのアニメーションリソース
    //! @param[in] groupContainer   グループのコンテナ。この中から関連づけるグループを捜します。
    //! @param[in] groupIndex       何番目のグループをバインドするか。
    //! @param[in] enable           開始時にアニメを有効状態にするか。
    //!
    void Setup(const AnimResource& animRes, GroupContainer* groupContainer, u32 groupIndex, bool enable);

    //! @brief アニメーションがバインドされているグループを取得します。
    //!
    //! @return バインドされているグループ
    //!
    Group* GetGroup() const { return mGroup; }

protected:
    Group* mGroup;      //!< 関連づけられたグループ

};

//---------------------------------------------------------------------------
//! @brief アニメリソースに関連付けられたすべてのグループをアニメーションさせるAnimatorです。
//!
//---------------------------------------------------------------------------
class GroupArrayAnimator : public Animator
{
public:
    //! 実行時型情報です。
    NW_UT_RUNTIME_TYPEINFO(Animator);

    GroupArrayAnimator()
     : Animator()
     , mGroups(NULL)
     , mGroupNum(0)
    {}

    //! @brief 指定したアニメーションリソースに関連づけられているグループのうち、index番目のグループに
    //!        アニメーションをバインドし、内部にグループのポインタを保持します。
    //!
    //! @param[in] animRes          構築するアニメのアニメーションリソース
    //! @param[in] groupContainer   グループのコンテナ。この中から関連づけるグループを捜します。
    //! @param[in] groupBuffer      関連づけられているグループのポインタを格納するためのバッファ。(animRes.GetGroupNum() * sizeof(Group*))の
    //!                             長さの配列を指定してください。このバッファは、このオブジェクトが破棄されるまで破棄しないでください。
    //! @param[in] enable           開始時にアニメを有効状態にするか。
    //!
    void Setup(const AnimResource& animRes, GroupContainer* groupContainer, Group** groupBuffer, bool enable);

    //! @brief Setupで関連付けたアニメーションをバインド解除します。
    //!
    virtual void Unbind();

    //! @brief アニメーションがバインドされているグループのポインタの配列を取得します。
    //!
    //! 配列の長さはGetGroupNum()で取得できる数となります。
    //!
    //! @return バインドされているグループの配列
    //!
    Group** GetGroups() const { return mGroups; }

    //! @brief アニメーションがバインドされているグループの数を取得します。
    //!
    //! @return バインドされているグループの数
    //!
    u32 GetGroupNum() const { return mGroupNum; }

protected:
    Group** mGroups;    //!< 関連づけられたグループの配列
    u32 mGroupNum;      //!< 関連付けられたグループの数

};

} // namespace nw::lyt
} // namespace nw

#endif // NW_LYT_ANIMATOR_H_

