﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_LYT_ANIMATION_H_
#define NW_LYT_ANIMATION_H_

#include <nw/ut/ut_LinkList.h>
#include <nw/ut/ut_RuntimeTypeInfo.h>

#include <nw/lyt/lyt_Types.h>

namespace nw
{
namespace lyt
{
namespace res
{

struct BinaryFileHeader;
struct AnimationBlock;
struct AnimationTagBlock;
struct AnimationShareBlock;
struct AnimationGroupRef;
struct AnimationShareInfo;
struct AnimationContent;

} // namespace res

namespace internal
{

class AnimPaneTreeLink;

} // namespace internal

class Pane;
class Layout;
class Group;
class Material;
class ResourceAccessor;
class TextureInfo;

//---------------------------------------------------------------------------
//! @brief アニメーションの基底クラスです。
//!
//---------------------------------------------------------------------------
class AnimTransform
{
public:
    //! 実行時型情報です。
    NW_UT_RUNTIME_TYPEINFO_ROOT();

    //---------------------------------------------------------------------------
    //! @brief バインドしたアニメーション内容と適用させるターゲットを保持する構造体です。
    //!
    //! @details
    //! targetは、ペインの場合とマテリアルの場合があります。
    //! animCont->typeで区別してください。
    //!
    //---------------------------------------------------------------------------
    struct BindPair
    {
        void* target;
        const res::AnimationContent* animCont;

        BindPair() : target(NULL), animCont(NULL) {}
    };

    //----------------------------------------
    //! @name コンストラクタ／デストラクタ
    //@{

    //! @brief コンストラクタです。
    //!
    AnimTransform();

    //! @brief デストラクタです。
    //!
    virtual ~AnimTransform();

    //@}

    //----------------------------------------
    //! @name 設定／取得
    //@{

    //! @brief 現在のフレーム位置を取得します。
    //!
    //! @return 現在のフレーム位置を返します。
    //!
    //! @sa SetFrame
    //!
    f32 GetFrame() const
    {
        return m_Frame;
    }

    //! @brief フレーム位置を設定します。
    //!
    //! @param[in] frame  フレーム位置です。
    //!
    //! @sa GetFrame
    //!
    void SetFrame(f32 frame)
    {
        m_Frame = frame;
    }

    //! @brief フレームの長さを取得します。
    //!
    //! @return フレームの長さを返します。
    //!
    u16 GetFrameSize() const;

    //! @brief フレーム位置の最大値を取得します。
    //!
    //! @return フレーム位置の最大値を返します。
    //!
    f32 GetFrameMax() const
    {
        return static_cast<f32>(GetFrameSize());
    }

    //! @brief アニメのフレームを進めます。
    //!
    //! @details
    //! このメソッドは、デフォルトでは何も行いません。
    //! 必要な場合に派生クラスで実装を行います。
    //!
    //! @param[in] progress_frame 進めるフレーム数です。
    //!
    virtual void UpdateFrame(f32 progress_frame = 1.0f);

    //! @brief アニメーションブロックを取得します。
    //!
    //! @return アニメーションブロックへのポインタを返します。
    //!
    //! @sa SetResource
    //!
    const res::AnimationBlock* GetAnimResource() const
    {
        return m_pRes;
    }

    //! @brief 有効か否かを取得します。
    //!
    //! @return 有効か否か
    //!
    bool IsEnable() const
    {
        return m_IsEnable;
    }

    //! @brief 有効か否かを設定します。
    //!
    //! @param[in] bEnable  有効か否か
    //!
    virtual void SetEnable(bool bEnable);
    //@}

    //----------------------------------------
    //! @name アニメーション
    //@{

    //! @brief ループ用のアニメーションデータか判定します。
    //!
    //! @details
    //! アニメーションデータがループ用としてマークされているかどうかを
    //! 判定します。
    //!
    //! @return
    //! アニメーションデータがループ用としてマークされている場合は true を
    //! 返します。
    //!
    bool IsLoopData() const;

    //! @brief Waitのアニメーションデータか判定します。
    //!
    //! @details
    //! アニメーションデータがWaitのアニメーションデータか判定します。
    //! Waitのアニメーションは、フレーム長が0のデータで、レイアウト
    //! エディタのアニメーション区間タグ設定の再生モードで「ウェイト」
    //! が設定されたものです。
    //!
    //! @return
    //! アニメーションがWaitアニメである場合はtrueを返します。
    //!
    bool IsWaitData() const;

    //! @brief 関連付けられているペイン及びマテリアルのアニメーションを全て再生します。
    //!
    //! @sa Bind
    //!
    virtual void Animate() = 0;

    //! @brief アニメーションを再生します。
    //!
    //! @details
    //! pPane で指定したペインのアニメーションを再生します。
    //!
    //! @param[in,out] pPane    ペインです。
    //!
    //! @sa Bind
    //!
    virtual void AnimatePane(Pane* pPane) = 0;

    //! @brief アニメーションを再生します。
    //!
    //! @details
    //! pMaterial で指定したマテリアルのアニメーションを再生します。
    //!
    //! @param[in,out] pMaterial    マテリアルです。
    //!
    //! @sa Bind
    //!
    virtual void AnimateMaterial(Material* pMaterial) = 0;

    //@}

    //----------------------------------------
    //! @name リソース
    //@{

    //! @brief アニメーションが使用するリソースを設定します。
    //!
    //! @param[in] pRes              リソースのアニメーションブロックへのポインタです。
    //! @param[in,out] pResAccessor  リソースアクセサです。
    //!
    //! @details
    //! Bind() でアニメーションに関連付けるアニメーションブロックデータと
    //! アニメーションデータに必要なリソースを設定します。
    //!
    //! また、 Bind() で必要になるBindPairの配列 を
    //! アニメーションブロックに含まれるアニメーション対象の個数分確保します。
    //!
    //! この関数の実装では SetAnimResource() を使用して、アニメーション
    //! ブロックデータを登録しなければなりません。
    //!
    //! @sa Bind
    //! @sa SetAnimResource
    //!
    virtual void SetResource(
        const res::AnimationBlock* pRes,
        ResourceAccessor* pResAccessor) = 0;

    //! @brief アニメーションが使用するリソースを設定します。
    //!
    //! @param[in] pRes             リソースのアニメーションブロックへのポインタです。
    //! @param[in,out] pResAccessor リソースアクセサです。
    //! @param[in] animNum          確保するポインタ配列の要素数です。
    //!
    //! @details
    //! Bind() でアニメーションに関連付けるアニメーションブロックデータと
    //! アニメーションデータに必要なリソースを設定します。
    //!
    //! また、 Bind() で必要になるBindPairの配列 を
    //! animNum で指定された個数、確保します。
    //!
    //! 本関数の実装では SetAnimResource() を使用して、アニメーション
    //! ブロックデータを登録しなければなりません。
    //!
    //! @sa Bind
    //! @sa SetAnimResource
    //!
    virtual void SetResource(
        const res::AnimationBlock* pRes,
        ResourceAccessor* pResAccessor,
        u16 animNum) = 0;

    //! @brief アニメーションの関連付けを行います。
    //!
    //! @details
    //! pPane で指定したペインにアニメーションを関連付けます。
    //!
    //! bRecursive に true を渡して呼び出した場合は、ペインを pPane の子からも
    //! 検索します。
    //!
    //! @param[in] pPane        ペインへのポインタです。
    //! @param[in] bRecursive   再帰的に検索する場合は true を指定します。
    //!
    virtual void  BindPane(
        Pane* pPane,
        bool bRecursive) = 0;

    //! @brief アニメーションの関連付けを行います。
    //!
    //! @details
    //! pGroup で指定したグループに所属するペインにアニメーションを関連付けます。
    //!
    //! @param[in] pGroup   グループへのポインタです。
    //!
    virtual void BindGroup(
        Group* pGroup) = 0;

    //! @brief アニメーションの関連付けを行います。
    //!
    //! @details
    //! pMaterial で指定したマテリアルにアニメーションを関連付けます。
    //!
    //! @param[in] pMaterial   マテリアルへのポインタです。
    //!
    virtual void BindMaterial(Material* pMaterial) = 0;

    //! @brief 強制的に別のペインのアニメーションの関連付けを行います。
    //!
    //! @details
    //! pDstPane で指定したペインに、pSrcPaneで指定したペインの
    //! アニメーションを関連付けます。
    //!
    //! 本来異なるペインに付いているアニメをバインドすることになりますので、
    //! 例えばペインの種類が異なっていたりすると不正な動作になってしまいます。
    //! 十分に注意して使用してください。
    //!
    //! なお、関連付けたアニメはUnbindPaneで解除できます。
    //!
    //! @param[in] pDstPane 強制的に関連付けたい先のペインへのポインタです。
    //! @param[in] pSrcPane 強制的に関連付けたい元のアニメーションを持つペインの名前です。
    //!
    virtual void  ForceBindPane(
        Pane* pDstPane,
        const Pane* pSrcPane) = 0;

    //! @brief アニメーションの関連付けを解除します。
    //!
    //! @details
    //! pPane で指定したペインのアニメーションの関連付けを解除します。
    //!
    //! @param[in] pPane    ペインへのポインタです。
    //!
    virtual void UnbindPane(Pane* pPane) = 0;

    //! @brief アニメーションの関連付けを解除します。
    //!
    //! @details
    //! pGroup で指定したグループのアニメーションの関連付けを解除します。
    //!
    //! @param[in] pGroup   グループへのポインタです。
    //!
    virtual void UnbindGroup(Group* pGroup) = 0;

    //! @brief アニメーションの関連付けを解除します。
    //!
    //! @details
    //! pMaterial で指定したマテリアルのアニメーションの関連付けを解除します。
    //!
    //! @param[in] pMaterial    マテリアルへのポインタです。
    //!
    virtual void UnbindMaterial(Material* pMaterial) = 0;

    //! @brief 全てのアニメーションの関連付けを解除します。
    //!
    virtual void UnbindAll() = 0;

    //@}

    // レイアウトクラスが管理するためのリンク情報です。

    //! @brief 内部用機能のため使用禁止です。
    ut::LinkListNode m_Link;

protected:
    //----------------------------------------
    //! @name 設定／取得
    //@{

    //! @brief アニメーションリソースへのポインタを設定します。
    //!
    //! @details
    //! SetResource() の実装内で使用され、 GetAnimResource() の戻り値を設定します。
    //!
    //! @param[in] pRes アニメーションブロックへのポインタです。
    //!
    //! @sa SetResource
    //! @sa GetAnimResource
    //!
    void SetAnimResource(const res::AnimationBlock* pRes)
    {
        m_pRes = pRes;
    }

    //@}

private:
    // アニメーションリソースです。
    const res::AnimationBlock* m_pRes;

    // カレントフレーム値です。
    f32 m_Frame;

    // 有効か否かです。
    bool m_IsEnable;
};

//---------------------------------------------------------------------------
//! @brief アニメーションの基礎クラスです。
//!
//---------------------------------------------------------------------------
class AnimTransformBasic : public AnimTransform
{
private:
    typedef AnimTransform Base;

public:
    //! 実行時型情報です。
    NW_UT_RUNTIME_TYPEINFO(AnimTransform);

    //----------------------------------------
    //! @name コンストラクタ／デストラクタ
    //@{

    //! @brief コンストラクタです。
    //!
    AnimTransformBasic();

    //! @brief デストラクタです。
    //!
    virtual ~AnimTransformBasic();

    //@}

    //----------------------------------------
    //! @name アニメーション
    //@{

    virtual void Animate();

    virtual void AnimatePane(Pane* pPane);

    virtual void AnimateMaterial(Material* pMaterial);

    //@}

    //----------------------------------------
    //! @name リソース
    //@{

    virtual void SetResource(
        const res::AnimationBlock* pRes,
        ResourceAccessor* pResAccessor);

    virtual void SetResource(
        const res::AnimationBlock* pRes,
        ResourceAccessor* pResAccessor,
        u16 animNum);

    virtual void BindPane(
        Pane* pPane,
        bool bRecursive);

    virtual void BindGroup(
        Group* pGroup);

    virtual void BindMaterial(
        Material* pMaterial);

    virtual void  ForceBindPane(
        Pane* pDstPane,
        const Pane* pSrcPane);

    virtual void UnbindPane(Pane* pPane);

    virtual void UnbindGroup(Group* pGroup);

    virtual void UnbindMaterial(Material* pMaterial);

    virtual void UnbindAll();

    //@}

    // ペインのアニメーションの関連付けを行います。

    //! @brief 内部用機能のため使用禁止です。
    //!
    //! @param[in] pTarget  関連付けを行う対象です。
    //! @param[in] animCont アニメーションです。
    //!
    //! @return 成功したら true を返します。
    //!
    bool BindPaneImpl(
        Pane* pTarget,
        const res::AnimationContent* animCont);

    // マテリアルのアニメーションの関連付けを行います。

    //! @brief 内部用機能のため使用禁止です。
    //!
    //! @param[in] pTarget  関連付けを行う対象です。
    //! @param[in] animCont アニメーションです。
    //!
    //! @return 成功したら true を返します。
    //!
    bool BindMaterialImpl(
        Material* pTarget,
        const res::AnimationContent* animCont);

protected:
    virtual void AnimatePaneImpl(Pane* pPane, const res::AnimationContent* animCont);

    virtual void AnimateMaterialImpl(Material* pMaterial, const res::AnimationContent* animCont);

    bool checkBindAnimationDoubly(const void* target, const res::AnimationContent* animCont) const;

    void eraseBindPair(int pos);

private:
    //! テクスチャの情報です。
    const TextureInfo** m_pTexAry;

    //! BindPairの配列です。
    BindPair* m_BindPairAry;

    //! バインドされているBindPairの数です。
    u16 m_BindPairNum;

    //! BindPairの最大数です。
    u16 m_BindPairNumMax;

};

//---------------------------------------------------------------------------
//! @brief アニメーションリソースクラスです。
//!
//---------------------------------------------------------------------------
class AnimResource
{
public:
    //----------------------------------------
    //! @name コンストラクタ／デストラクタ
    //@{

    //! @brief コンストラクタです。
    //!
    //! @details
    //! リソースの関連付けは行われません。
    //! 別途 Set() を使用して関連付けを行います。
    //!
    //! @sa Set
    //!
    AnimResource();

    //! @brief コンストラクタです。
    //!
    //! @details
    //! anmResBuf で指定したアニメーションリソースが関連付けられます。
    //!
    //! @param[in] anmResBuf    アニメーションリソースへのポインタです。
    //!
    explicit AnimResource(const void* anmResBuf)
    {
        Set(anmResBuf);
    }

    //@}

    //----------------------------------------
    //! @name 設定／取得
    //@{

    //! @brief アニメーションリソースを関連付けます。
    //!
    //! @param[in] anmResBuf    アニメーションリソースへのポインタです。
    //!
    void Set(const void* anmResBuf);

    //! @brief アニメーションリソースを取得します。
    //!
    //! @return ファイルヘッダへのポインタを返します。
    //!
    const ut::BinaryFileHeader* GetFileHeader() const
    {
        return m_pFileHeader;
    }

    //! @brief アニメーションブロックを取得します。
    //!
    //! @return アニメーションブロックへのポインタを返します。
    //!
    const res::AnimationBlock* GetResourceBlock() const
    {
        return m_pResBlock;
    }

    //! @brief アニメーション区間タグブロックを取得します。
    //!
    //! @return アニメーション区間タグブロックへのポインタを返します。
    //!
    const res::AnimationTagBlock* GetTagBlock() const
    {
        return m_pTagBlock;
    }

    //! @brief アニメーション区間タグの通し番号を取得します。
    //!
    //! @return アニメーション区間タグの0から始まる通し番号を返します。
    //!
    u16 GetTagOrder() const;

    //! @brief アニメーション区間タグの名前を取得します。
    //!
    //! @return アニメーション区間タグの名前を返します。
    //!
    const char* GetTagName() const;

    //! @brief アニメーション区間タグのグループ数を取得します。
    //!
    //! @return アニメーション区間タグに設定されているグループ数を返します。
    //!
    u16 GetGroupNum() const;

    //! @brief アニメーション区間タグのグループ配列を取得します。
    //!
    //! @return
    //! アニメーション区間タグに設定されているグループ配列の
    //! 先頭ポインタを返します。
    //!
    const res::AnimationGroupRef* GetGroupArray() const;

    //! @brief
    //! アニメーションをバインドするとき、ペインの子孫となるペインも対象に
    //! するかどうかを判定します。
    //!
    //! @return
    //! アニメーションをバインドするとき、子孫のペインも対象にするときは
    //! true を返します。
    //!
    bool IsDescendingBind() const;

    //! @brief アニメーション共有情報の個数を取得します。
    //!
    //! @return アニメーション共有情報の個数を返します。
    //!
    u16 GetAnimationShareInfoNum() const;

    //! @brief アニメーション共有情報の配列を取得します。
    //!
    //! @return AnimationShareInfo オブジェクト配列の先頭ポインタを返します。
    //!
    const res::AnimationShareInfo* GetAnimationShareInfoArray() const;

    //! @brief バインドされるアニメーションの個数を計算します。
    //!
    //! @details
    //! pPane で指定したペインにバインドされるアニメーションの個数を計算します。
    //!
    //! bRecursive に true を指定した場合は、関連付けるペインを子ペインからも
    //! 検索します。
    //!
    //! @param[in] pPane        ペインへのポインタです。
    //! @param[in] bRecursive   子ペインも検索するか指定します。
    //!
    //! @return バインドされるアニメーションの個数を返します。
    //!
    u16 CalcAnimationNum(Pane* pPane, bool bRecursive) const;

    //! @brief バインドされるアニメーションの個数を計算します。
    //!
    //! @details
    //! pMaterial で指定されたマテリアルにバインドされるアニメーションの
    //! 個数を計算します。
    //!
    //! @param[in] pMaterial    マテリアルへのポインタです。
    //!
    //! @return バインドされるアニメーションの個数を返します。
    //!
    u16 CalcAnimationNum(Material* pMaterial) const;

    //! @brief バインドされるアニメーションの個数を計算します。
    //!
    //! @details
    //! pGroup で指定されたグループに含まれるペインにバインドされる
    //! アニメーションの個数を計算します。
    //!
    //! bRecursive に true を指定した場合は、関連付けるペインを子ペインからも
    //! 検索します。
    //!
    //! @param[in] pGroup       グループへのポインタです。
    //! @param[in] bRecursive   子ペインも検索するか指定します。
    //!
    //! @return バインドされるアニメーションの個数を返します。
    //!
    u16 CalcAnimationNum(Group* pGroup, bool bRecursive) const;

    //@}

private:
    void Init();

    //! @brief アニメーションリソースが設定されているか確認します。
    //!
    //! @return 設定されていれば true を返します。
    //!
    bool CheckResource() const;

    const ut::BinaryFileHeader* m_pFileHeader;
    const res::AnimationBlock* m_pResBlock;
    const res::AnimationTagBlock* m_pTagBlock;
    const res::AnimationShareBlock* m_pShareBlock;
};

namespace internal
{

class AnimPaneTree
{
public:
    AnimPaneTree();

    AnimPaneTree(
        Pane* pTargetPane,
        const AnimResource& animRes);

    void Set(
        Pane* pTargetPane,
        const AnimResource& animRes);

    AnimTransform* Bind(
        Layout* pLayout,
        Pane* pTargetPane,
        ResourceAccessor* pResAccessor) const;

    bool IsEnabled() const
    {
        return m_LinkNum > 0;
    }

    const AnimResource& GetAnimResource() const
    {
        return m_AnimRes;
    }

private:
    static const res::AnimationContent* FindAnimContent(
        const res::AnimationBlock* pAnimBlock,
        const char* animContName,
        u8 animContType);

    void Init();

    static const int MATERIAL_NUM_MAX = 1 + 4 + 4;

    AnimResource m_AnimRes;

    const res::AnimationContent* m_PaneAnimCont;
    const res::AnimationContent* m_MatAnimConts[MATERIAL_NUM_MAX];

    u16 m_LinkNum;
    u8 m_AnimMatCnt;

};

} // namespace nw::lyt::internal
} // namespace nw::lyt
} // namespace nw

#endif // NW_LYT_ANIMATION_H_
