﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_GFND_DISPLAYLIST_H_
#define NW_GFND_DISPLAYLIST_H_

#include <nw/types.h>
#include <nw/ut/ut_Preprocessor.h>

#if defined(NW_PLATFORM_CAFE)
#define NW_DISPLAY_LIST_ENABLED
#endif

namespace nw
{
namespace gfnd
{

//########################################
#if defined(NW_DISPLAY_LIST_ENABLED)
//########################################
//---------------------------------------------------------------------------
//! @brief ディスプレイリストを使用するためのクラスです。
//!
//! @details :private
//---------------------------------------------------------------------------
class DisplayList
{
    NW_DISALLOW_COPY_AND_ASSIGN(DisplayList);

public:
    static const u32 ALIGNMENT = GX2_DISPLAY_LIST_ALIGNMENT;

    //---------------------------------------------------------------------------
    //! @brief コンストラクタです。
    //---------------------------------------------------------------------------
    DisplayList()
    : m_Buffer(NULL)
    , m_Capacity(0)
    , m_Size(0)
    {}

    //---------------------------------------------------------------------------
    //! @brief デストラクタです。
    //---------------------------------------------------------------------------
    ~DisplayList() {}

    //---------------------------------------------------------------------------
    //! @brief        ディスプレイリストにバッファを設定して初期化します。
    //!
    //! @param[in]    buffer ディスプレイリストを保存するバッファです。
    //! @param[in]    capacity バッファのサイズです。
    //!
    //! @sa Shutdown
    //!
    //---------------------------------------------------------------------------
    void Initialize(void* buffer, u32 capacity)
    {
        NW_ASSERT_NOT_NULL(buffer);
        NW_ASSERT((reinterpret_cast<u32>(buffer) % ALIGNMENT) == 0);
        m_Buffer = buffer;
        m_Capacity = capacity;
        GX2Invalidate(GX2_INVALIDATE_CPU, buffer, capacity);
    }

    //---------------------------------------------------------------------------
    //! @brief        ディスプレイリストを後始末します。
    //!
    //! 内部でバッファは解放しないので、返り値のバッファを
    //! 必要に応じて解放してください。
    //!
    //! バッファを使用している描画が終了するまで、
    //! バッファの解放をしないようにしてください。
    //!
    //! @return       バッファを返します。
    //!
    //! @sa Initialize
    //---------------------------------------------------------------------------
    void* Shutdown()
    {
        void* buffer = m_Buffer;
        m_Buffer = NULL;
        m_Capacity = 0;
        m_Size = 0;
        return buffer;
    }

    //---------------------------------------------------------------------------
    //! @brief        ディスプレイリストを現在有効なコマンドバッファに追加します。
    //---------------------------------------------------------------------------
    void Call()
    {
        if (!IsEmpty())
        {
            GX2CallDisplayList(m_Buffer, m_Size);
        }
    }

    //---------------------------------------------------------------------------
    //! @brief        ディスプレイリストを直接グラフィックスリングバッファに追加します。
    //---------------------------------------------------------------------------
    void DirectCall()
    {
        if (!IsEmpty())
        {
            GX2DirectCallDisplayList(m_Buffer, m_Size);
        }
    }

    //---------------------------------------------------------------------------
    //! @brief        ディスプレイリストを後始末します。
    //!
    //! @return       作成済ディスプレイリストを保持している場合は true を返します。
    //---------------------------------------------------------------------------
    bool IsEmpty() { return m_Size == 0;}

    //---------------------------------------------------------------------------
    //! @brief 作成済ディスプレイリストをクリアします。
    //---------------------------------------------------------------------------
    void Clear() { m_Size = 0; }

    //---------------------------------------------------------------------------
    //! @brief 作成済ディスプレイリストのサイズを取得します。
    //---------------------------------------------------------------------------
    u32 GetSize() { return m_Size; }

    //---------------------------------------------------------------------------
    //! @brief ディスプレイリストの作成を開始します。
    //!
    //! @sa End
    //---------------------------------------------------------------------------
    void Begin()
    {
        if (IsValid())
        {
            GX2BeginDisplayList(m_Buffer, m_Capacity);
        }
    }

    //---------------------------------------------------------------------------
    //! @brief ディスプレイリストの作成を終了します。
    //!
    //! @sa Begin
    //---------------------------------------------------------------------------
    void End()
    {
        if (IsValid())
        {
            m_Size = GX2EndDisplayList(m_Buffer);
            NW_WARNING(m_Size <= m_Capacity, "Display list buffer size over!");
        }
    }

    //---------------------------------------------------------------------------
    //! @brief        ディスプレイリストが作成可能か取得します。
    //!
    //! Initialize 前か、Shutdown 後であれば、false を返します。
    //!
    //! @sa Initialize
    //! @sa Shutdown
    //!
    //! @return       ディスプレイリストが作成可能な状態であれば true が返ります。
    //---------------------------------------------------------------------------
    bool IsValid()
    {
        return m_Buffer != NULL;
    }

private:
    void* m_Buffer;
    u32 m_Capacity;
    u32 m_Size;
};

//---------------------------------------------------------------------------
//! @brief ディスプレイリスト作成手続きを簡単にするためのクラスです。
//!
//! @details :private
//---------------------------------------------------------------------------
class DisplayListMaker
{
public:
    explicit DisplayListMaker(DisplayList& displayList)
    : m_DisplayList(displayList) { m_DisplayList.Begin(); }
    ~DisplayListMaker() { m_DisplayList.End(); }

    //! @details :private
    struct SafeBoolHelper { int x; };

    //! @details :private
    typedef int SafeBoolHelper::* SafeBool;

    //! @brief 条件文で有効な場合に true として暗黙的に振る舞います。
    operator SafeBool() const
    {
        return &SafeBoolHelper::x;
    }

private:
    DisplayList& m_DisplayList;
};

//----------------------------------------
#define NW_MAKE_DISPLAY_LIST(VAR, DL) \
    if (!DL.IsValid() || DL.IsEmpty()) \
        if (nw::gfnd::DisplayListMaker VAR = nw::gfnd::DisplayListMaker(DL))

//########################################
#else
//########################################
class DisplayList
{
    NW_DISALLOW_COPY_AND_ASSIGN(DisplayList);

public:
    DisplayList() {}
    ~DisplayList() {}

    void Initialize(void* buffer, u32 capacity)
    {
        NW_UNUSED_VARIABLE(buffer);
        NW_UNUSED_VARIABLE(capacity);
    }
    void* Shutdown() { return 0; }
    void Call() {}
    void DirectCall() {}
    bool IsEmpty() { return true;}
    void Clear() {}
    u32 GetSize() { return 0; }
};

//----------------------------------------
#define NW_MAKE_DISPLAY_LIST(VAR, DL)

#endif // defined(NW_PLATFORM_CAFE)

} // namespace gfnd
} // namespace nw

#endif // NW_GFND_DISPLAYLIST_H_
