﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_G3D_UT_PERF_H_
#define NW_G3D_UT_PERF_H_

#include <nw/g3d/g3d_config.h>

#define NW_G3D_CONCAT_LINE__(str, line) str##line
#define NW_G3D_CONCAT_LINE_(str, line) NW_G3D_CONCAT_LINE__(str, line)
#define NW_G3D_CONCAT_LINE(str) NW_G3D_CONCAT_LINE_(str, __LINE__)

#if NW_G3D_PERF_ENABLE

#define NW_G3D_PERF_INIT() nw::g3d::ut::PerfManager::Init()
#define NW_G3D_PERF_RESET() nw::g3d::ut::PerfManager::Instance().Reset()
#define NW_G3D_PERF_NEXT_FRAME() nw::g3d::ut::PerfManager::Instance().NextFrame()
#define NW_G3D_PERF_PRINT() nw::g3d::ut::PerfManager::Instance().Print()
#define NW_G3D_PERF_PRINT_TABLE() nw::g3d::ut::PerfManager::Instance().PrintTable()

#define NW_G3D_PERF_CPU(name)                                                                      \
    static nw::g3d::ut::PerfNodeCPU NW_G3D_CONCAT_LINE(perfNodeCPU)(name);                         \
    nw::g3d::ut::PerfNodeCPU::StopWatch NW_G3D_CONCAT_LINE(perfNodeStopWatch)(                     \
        NW_G3D_CONCAT_LINE(perfNodeCPU))                                                           \

#if defined( _MSC_VER )
#define NW_G3D_PERF_CPU_FUNC() NW_G3D_PERF_CPU(__FUNCSIG__)
#elif defined( __ghs__ )
#define NW_G3D_PERF_CPU_FUNC() NW_G3D_PERF_CPU(__PRETTY_FUNCTION__)
#else
#define NW_G3D_PERF_CPU_FUNC() NW_G3D_PERF_CPU("")
#endif

#else

#define NW_G3D_PERF_INIT()
#define NW_G3D_PERF_RESET()
#define NW_G3D_PERF_NEXT_FRAME()
#define NW_G3D_PERF_PRINT()
#define NW_G3D_PERF_PRINT_TABLE()
#define NW_G3D_PERF_CPU(name)
#define NW_G3D_PERF_CPU_FUNC()

#endif // NW_G3D_PERF_LEVEL

#if NW_G3D_PERF_LEVEL >= 1
#define NW_G3D_PERF_LEVEL1(name) NW_G3D_PERF_CPU(name)
#define NW_G3D_PERF_LEVEL1_FUNC() NW_G3D_PERF_CPU_FUNC()
#else
#define NW_G3D_PERF_LEVEL1(name)
#define NW_G3D_PERF_LEVEL1_FUNC()
#endif // NW_G3D_PERF_LEVEL >= 1

#if NW_G3D_PERF_LEVEL >= 2
#define NW_G3D_PERF_LEVEL2(name) NW_G3D_PERF_CPU(name)
#define NW_G3D_PERF_LEVEL2_FUNC() NW_G3D_PERF_CPU_FUNC()
#else
#define NW_G3D_PERF_LEVEL2(name)
#define NW_G3D_PERF_LEVEL2_FUNC()
#endif // NW_G3D_PERF_LEVEL >= 2

#if NW_G3D_PERF_LEVEL >= 3
#define NW_G3D_PERF_LEVEL3(name) NW_G3D_PERF_CPU(name)
#define NW_G3D_PERF_LEVEL3_FUNC() NW_G3D_PERF_CPU_FUNC()
#else
#define NW_G3D_PERF_LEVEL3(name)
#define NW_G3D_PERF_LEVEL3_FUNC()
#endif // NW_G3D_PERF_LEVEL >= 3


namespace nw { namespace g3d { namespace ut {

class PerfNodeCPU;

class PerfManager
{
public:
    static void Init();

    static PerfManager& Instance() { return s_Instance; }

    void Add(PerfNodeCPU* pNode);
    PerfNodeCPU* GetNodeCPU() { return m_pFirstNodeCPU; }

    void Reset();
    void NextFrame();

    void Print();
    void PrintTable();

    static s64 GetTicks();
    static s64 GetFreq() { return s_Freq; }
    static float ToSeconds(s64 ticks) { return static_cast<float>(ticks) / s_Freq; }
    static float ToMilliSecs(s64 ticks) { return ToSeconds(ticks) * 1000; }
    static float ToMicroSecs(s64 ticks) { return ToSeconds(ticks) * (1000 * 1000); }
    static float ToPercents(s64 ticks) { return ToSeconds(ticks) * (60 * 100); }

private:
    PerfManager() : m_pFirstNodeCPU(NULL), m_pLastNodeCPU(NULL) {}

private:
    static PerfManager s_Instance;
    static s64 s_Freq;

    PerfNodeCPU* m_pFirstNodeCPU;
    PerfNodeCPU* m_pLastNodeCPU;

    bool m_InitialFrame;

    NW_G3D_DISALLOW_COPY_AND_ASSIGN(PerfManager);
};

class PerfNodeCPU
{
    friend class PerfManager;

public:
    class StopWatch;

    explicit PerfNodeCPU(const char* name) : m_pNext(NULL), m_Name(name)
    {
        Reset();
        PerfManager::Instance().Add(this);
    }

    void Reset();
    void NextFrame();

    void Print();
    void PrintTable();

private:
    PerfNodeCPU() : m_pNext(NULL), m_Name("Empty") { Reset(); }

private:
    PerfNodeCPU* m_pNext;
    const char* m_Name;

    s64 m_CurElapsed[3];
    s64 m_MinElapsed;
    s64 m_MaxElapsed;
    s64 m_TotalElapsed;

    int m_CurCount[3];
    int m_MinCount;
    int m_MaxCount;
    int m_TotalCount; // m_TotalElapsed を割ると1回あたりのコスト
    int m_TotalFrame; // m_TotalElapsed を割ると1フレームあたりのコスト
};

class PerfNodeCPU::StopWatch
{
public:
    explicit StopWatch(PerfNodeCPU& node) : m_Node(node) { m_Start = PerfManager::GetTicks(); }

    ~StopWatch()
    {
#if NW_G3D_IS_HOST_CAFE
        uint core = OSGetCoreId();
#else
        uint core = 0; // マルチコア対応は保留
#endif
        m_Node.m_CurElapsed[core] += PerfManager::GetTicks() - m_Start;
        ++m_Node.m_CurCount[core];
    }

private:
    s64 m_Start;
    PerfNodeCPU& m_Node;

    NW_G3D_DISALLOW_COPY_AND_ASSIGN(StopWatch);
};

}}} // namespace nw::g3d::ut

#endif // NW_G3D_UT_PERF_H_
