﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_G3D_UT_FLOAT_H_
#define NW_G3D_UT_FLOAT_H_

#include <nw/g3d/g3d_config.h>

namespace nw { namespace g3d { namespace ut {

//--------------------------------------------------------------------------
//! @brief        16bit 浮動小数用のユーティリティです。
//
// 16bit 浮動小数のビットフォーマットは次の通りです。
// | sign | exponent | fraction |
// sign     : 符号 1 bit.
// exponent : 指数部 5 bit. bias 15.
// fraction : 仮数部 10 bit.
//---------------------------------------------------------------------------
class Float16
{
public:
    //--------------------------------------------------------------------------
    //! @brief        デフォルトコンストラクタです。
    //--------------------------------------------------------------------------
    Float16() : m_Float32( 0.0f ) {}

    //--------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //!
    //! @param[in]    bits16   16bit 浮動小数のビットを表現した整数値です。
    //--------------------------------------------------------------------------
    /* implicit */ Float16( u16 bits16 )
    {
        m_Float32 = Bits16ToFloat32( bits16 );
    }

    //--------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //!
    //! @param[in]    value   32bit 浮動小数の値から 16bit float を生成します。
    //--------------------------------------------------------------------------
    /* implicit */ Float16( f32 value ) : m_Float32( value ) {}

    f32     GetFloat32Value() const { return m_Float32; }
    u16     GetFloat16Value() const { return Float32ToBits16( m_Float32 ); }

    Float16& operator =(f32 value) { this->m_Float32 = value; return *this; }
    Float16& operator =(u16 bits16) { this->m_Float32 = Bits16ToFloat32( bits16 ); return *this; }

    operator f32() const { return m_Float32; }
    operator u16() const { return GetFloat16Value(); }

    f32 operator +(f32 right) const { return this->m_Float32 + right; }
    f32 operator -(f32 right) const { return this->m_Float32 - right; }
    f32 operator *(f32 right) const { return this->m_Float32 * right; }
    f32 operator /(f32 right) const { return this->m_Float32 / right; }

    Float16& operator +=(f32 rhs) { this->m_Float32 += rhs; return *this; }
    Float16& operator -=(f32 rhs) { this->m_Float32 -= rhs; return *this; }
    Float16& operator *=(f32 rhs) { this->m_Float32 *= rhs; return *this; }
    Float16& operator /=(f32 rhs) { this->m_Float32 /= rhs; return *this; }

    bool operator ==(f32 rhs) const { return (rhs == this->m_Float32); }
    bool operator !=(f32 rhs) const { return !(*this == rhs); }

    //--------------------------------------------------------------------------
    //! @brief        16bit float から 32bit float に変換します。
    //!
    //! @param[in]    bits16   16bit 浮動小数のビットを表現した整数値です。
    //!
    //! @return       f32 に変換した値を返します。
    //--------------------------------------------------------------------------
    static f32 Bits16ToFloat32(u16 bits16)
    {
        u32 sign = bits16 & SIGN16;
        int exp = (int)((bits16 & EXP_MASK16) >> FRACTION_WIDTH16);
        u32 fraction = bits16 & FRACTION_MASK16;

        u32 bits32 = 0;
        bits32 |= (sign != 0) ? SIGN32 : 0;

        if (exp == 0)
        {
            // ビット拡張なので fraction が 0 でない場合は非正規化数を正規化数に変換する必要がある。
            if (fraction != 0)
            {
                exp = 1;
                while ( (fraction & (1 << FRACTION_WIDTH16)) == 0 )
                {
                    --exp;
                    fraction <<= 1;
                }

                fraction &= FRACTION_MASK16;
                exp = exp - EXP_BIAS16 + EXP_BIAS32;
            }
        }
        else
        {
            exp = exp - EXP_BIAS16 + EXP_BIAS32;
        }

        fraction = fraction << (FRACTION_WIDTH32 - FRACTION_WIDTH16);

        // ビット拡張なので、exp がオーバーフローすることは無い。
        bits32 |= fraction & FRACTION_MASK32;
        bits32 |= ((u32)exp & 0xFF) << FRACTION_WIDTH32;

        return *reinterpret_cast<f32*>(&bits32);
    }

    //--------------------------------------------------------------------------
    //! @brief        f32 から 16bit float に変換します。
    //!
    //! @param[in]    value   32bit float の値です。
    //!
    //! @return       16bit 浮動小数のビット表現を返します。
    //--------------------------------------------------------------------------
    static u16 Float32ToBits16(f32 value)
    {
        u32 bits32 = *reinterpret_cast<u32*>(&value);

        u32 sign = bits32 & SIGN32;
        int exp = (int)((bits32 & EXP_MASK32) >> FRACTION_WIDTH32);
        u32 fraction = bits32 & FRACTION_MASK32;

        u32 bits16 = 0;
        bits16 |= (sign != 0) ? SIGN16 : 0;

        if (exp == 0)
        {
            // ビット縮小なので、非正規化数はすべて 0 に丸められる。
            fraction = 0;
        }
        else
        {
            exp = exp - EXP_BIAS32 + EXP_BIAS16;
        }

        fraction = fraction >> (FRACTION_WIDTH32 - FRACTION_WIDTH16);

        if (exp < 0)
        {
            // +0 もしくは -0 なのでそのまま。
        }
        else if (exp > 31)
        {
            // 無限大の処理
            bits16 = (u32)0x1F << FRACTION_WIDTH16;
        }
        else
        {
            bits16 |= fraction & FRACTION_MASK16;
            bits16 |= ((u32)exp & 0x1F) << FRACTION_WIDTH16;
        }

        return static_cast<u16>(bits16);
    }

private:
    f32 m_Float32;

    enum
    {
        SIGN32 = 0x80000000,
        SIGN16 = 0x00008000,

        EXP_BIAS32 = 127,
        EXP_BIAS16 = 15,
        EXP_MASK32 = 0x7F800000,
        EXP_MASK16 = 0x00007C00,

        FRACTION_WIDTH32 = 23,
        FRACTION_MASK32  = 0x007FFFFF,
        FRACTION_WIDTH16 = 10,
        FRACTION_MASK16  = 0x000003FF
    };
};

//--------------------------------------------------------------------------
//! @brief        10bit 浮動小数用のユーティリティです。
//
// 10bit 浮動小数のビットフォーマットは次の通りです。
// | exponent | fraction |
// exponent : 指数部 5 bit. bias 15.
// fraction : 仮数部 5 bit.
//---------------------------------------------------------------------------
class Float10
{
public:
    //--------------------------------------------------------------------------
    //! @brief        デフォルトコンストラクタです。
    //--------------------------------------------------------------------------
    Float10() : m_Float32( 0.0f ) {}

    //--------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //!
    //! @param[in]    bits10   10bit 浮動小数のビットを表現した整数値です。
    //--------------------------------------------------------------------------
    /* implicit */ Float10( u16 bits10 )
    {
        m_Float32 = Bits10ToFloat32( bits10 );
    }

    //--------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //!
    //! @param[in]    value   32bit 浮動小数の値から 10bit float を生成します。
    //--------------------------------------------------------------------------
    /* implicit */ Float10( f32 value ) : m_Float32( value ) {}

    f32     GetFloat32Value() const { return m_Float32; }
    u16     GetFloat10Value() const { return Float32ToBits10( m_Float32 ); }

    Float10& operator =(f32 value) { this->m_Float32 = value; return *this; }
    Float10& operator =(u16 bits10) { this->m_Float32 = Bits10ToFloat32( bits10 ); return *this; }

    operator f32() const { return m_Float32; }

    f32 operator +(f32 right) const { return this->m_Float32 + right; }
    f32 operator -(f32 right) const { return this->m_Float32 - right; }
    f32 operator *(f32 right) const { return this->m_Float32 * right; }
    f32 operator /(f32 right) const { return this->m_Float32 / right; }

    Float10& operator +=(f32 rhs) { this->m_Float32 += rhs; return *this; }
    Float10& operator -=(f32 rhs) { this->m_Float32 -= rhs; return *this; }
    Float10& operator *=(f32 rhs) { this->m_Float32 *= rhs; return *this; }
    Float10& operator /=(f32 rhs) { this->m_Float32 /= rhs; return *this; }

    bool operator ==(f32 rhs) const { return (rhs == this->m_Float32); }
    bool operator !=(f32 rhs) const { return !(*this == rhs); }

    //--------------------------------------------------------------------------
    //! @brief        10bit float から 32bit float に変換します。
    //!
    //! @param[in]    bits10   10bit 浮動小数のビットを表現した整数値です。
    //!
    //! @return       f32 に変換した値を返します。
    //--------------------------------------------------------------------------
    static f32 Bits10ToFloat32(u16 bits10)
    {
        int exp = (int)((bits10 & EXP_MASK10) >> FRACTION_WIDTH10);
        u32 fraction = bits10 & FRACTION_MASK10;

        u32 bits32 = 0;
        if (exp == 0)
        {
            // ビット拡張なので fraction が 0 でない場合は非正規化数を正規化数に変換する必要がある。
            if (fraction != 0)
            {
                exp = 1;
                while ( (fraction & (1 << FRACTION_WIDTH10)) == 0 )
                {
                    --exp;
                    fraction <<= 1;
                }

                fraction &= FRACTION_MASK10;
                exp = exp - EXP_BIAS10 + EXP_BIAS32;
            }
        }
        else
        {
            exp = exp - EXP_BIAS10 + EXP_BIAS32;
        }

        fraction = fraction << (FRACTION_WIDTH32 - FRACTION_WIDTH10);

        // ビット拡張なので、exp がオーバーフローすることは無い。
        bits32 |= fraction & FRACTION_MASK32;
        bits32 |= ((u32)exp & 0xFF) << FRACTION_WIDTH32;

        return *reinterpret_cast<f32*>(&bits32);
    }

    //--------------------------------------------------------------------------
    //! @brief        f32 から 10bit float に変換します。
    //!
    //! @param[in]    value   32bit float の値です。
    //!
    //! @return       10bit 浮動小数のビット表現を返します。
    //--------------------------------------------------------------------------
    static u16 Float32ToBits10(f32 value)
    {
        u32 bits32 = *reinterpret_cast<u32*>(&value);

        int exp = (int)((bits32 & EXP_MASK32) >> FRACTION_WIDTH32);
        u32 fraction = bits32 & FRACTION_MASK32;

        u32 bits10 = 0;

        if (exp == 0)
        {
            // ビット縮小なので、非正規化数はすべて 0 に丸められる。
            fraction = 0;
        }
        else
        {
            exp = exp - EXP_BIAS32 + EXP_BIAS10;
        }

        fraction = fraction >> (FRACTION_WIDTH32 - FRACTION_WIDTH10);

        if (exp < 0)
        {
            // +0 もしくは -0 なのでそのまま。
        }
        else if (exp > 127)
        {
            // 無限大の処理
            // TODO: IEEE float の無限大の表現がGPU上で有効なのかどうかを要確認。
            bits10 = (u32)0x7F << FRACTION_WIDTH10;
        }
        else
        {
            bits10 |= fraction & FRACTION_MASK10;
            bits10 |= ((u32)exp & 0x7F) << FRACTION_WIDTH10;
        }

        return static_cast<u16>(bits10);
    }

private:
    f32 m_Float32;

    enum
    {
        SIGN32 = 0x80000000,

        EXP_BIAS32 = 127,
        EXP_BIAS10 = 15,
        EXP_MASK32 = 0x7F800000,
        EXP_MASK10 = 0x000003E0,

        FRACTION_WIDTH32 = 23,
        FRACTION_MASK32  = 0x007FFFFF,
        FRACTION_WIDTH10 = 5,
        FRACTION_MASK10  = 0x00000001F
    };
};

//--------------------------------------------------------------------------
//! @brief        11bit 浮動小数用のユーティリティです。
//
// 11bit 浮動小数のビットフォーマットは次の通りです。
// | exponent | fraction |
// exponent : 指数部 5 bit. bias 15.
// fraction : 仮数部 6 bit.
//---------------------------------------------------------------------------
class Float11
{
public:
    //--------------------------------------------------------------------------
    //! @brief        デフォルトコンストラクタです。
    //--------------------------------------------------------------------------
    Float11() : m_Float32( 0.0f ) {}

    //--------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //!
    //! @param[in]    bits11   11bit 浮動小数のビットを表現した整数値です。
    //--------------------------------------------------------------------------
    /* implicit */ Float11( u16 bits11 )
    {
        m_Float32 = Bits11ToFloat32( bits11 );
    }

    //--------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //!
    //! @param[in]    value   32bit 浮動小数の値から 11bit float を生成します。
    //--------------------------------------------------------------------------
    /* implicit */ Float11( f32 value ) : m_Float32( value ) {}

    f32     GetFloat32Value() const { return m_Float32; }
    u16     GetFloat11Value() const { return Float32ToBits11( m_Float32 ); }

    Float11& operator =(f32 value) { this->m_Float32 = value; return *this; }
    Float11& operator =(u16 bits11) { this->m_Float32 = Bits11ToFloat32( bits11 ); return *this; }

    operator f32() const { return m_Float32; }

    f32 operator +(f32 right) const { return this->m_Float32 + right; }
    f32 operator -(f32 right) const { return this->m_Float32 - right; }
    f32 operator *(f32 right) const { return this->m_Float32 * right; }
    f32 operator /(f32 right) const { return this->m_Float32 / right; }

    Float11& operator +=(f32 rhs) { this->m_Float32 += rhs; return *this; }
    Float11& operator -=(f32 rhs) { this->m_Float32 -= rhs; return *this; }
    Float11& operator *=(f32 rhs) { this->m_Float32 *= rhs; return *this; }
    Float11& operator /=(f32 rhs) { this->m_Float32 /= rhs; return *this; }

    bool operator ==(f32 rhs) const { return (rhs == this->m_Float32); }
    bool operator !=(f32 rhs) const { return !(*this == rhs); }

    //--------------------------------------------------------------------------
    //! @brief        11bit float から 32bit float に変換します。
    //!
    //! @param[in]    bits11   11bit 浮動小数のビットを表現した整数値です。
    //!
    //! @return       f32 に変換した値を返します。
    //--------------------------------------------------------------------------
    static f32 Bits11ToFloat32(u16 bits11)
    {
        int exp = (int)((bits11 & EXP_MASK11) >> FRACTION_WIDTH11);
        u32 fraction = bits11 & FRACTION_MASK11;

        u32 bits32 = 0;
        if (exp == 0)
        {
            // ビット拡張なので fraction が 0 でない場合は非正規化数を正規化数に変換する必要がある。
            if (fraction != 0)
            {
                exp = 1;
                while ( (fraction & (1 << FRACTION_WIDTH11)) == 0 )
                {
                    --exp;
                    fraction <<= 1;
                }

                fraction &= FRACTION_MASK11;
                exp = exp - EXP_BIAS11 + EXP_BIAS32;
            }
        }
        else
        {
            exp = exp - EXP_BIAS11 + EXP_BIAS32;
        }

        fraction = fraction << (FRACTION_WIDTH32 - FRACTION_WIDTH11);

        // ビット拡張なので、exp がオーバーフローすることは無い。
        bits32 |= fraction & FRACTION_MASK32;
        bits32 |= ((u32)exp & 0xFF) << FRACTION_WIDTH32;

        return *reinterpret_cast<f32*>(&bits32);
    }

    //--------------------------------------------------------------------------
    //! @brief        f32 から 11bit float に変換します。
    //!
    //! @param[in]    value   32bit float の値です。
    //!
    //! @return       11bit 浮動小数のビット表現を返します。
    //--------------------------------------------------------------------------
    static u16 Float32ToBits11(f32 value)
    {
        u32 bits32 = *reinterpret_cast<u32*>(&value);

        int exp = (int)((bits32 & EXP_MASK32) >> FRACTION_WIDTH32);
        u32 fraction = bits32 & FRACTION_MASK32;

        u32 bits11 = 0;

        if (exp == 0)
        {
            // ビット縮小なので、非正規化数はすべて 0 に丸められる。
            fraction = 0;
        }
        else
        {
            exp = exp - EXP_BIAS32 + EXP_BIAS11;
        }

        fraction = fraction >> (FRACTION_WIDTH32 - FRACTION_WIDTH11);

        if (exp < 0)
        {
            // +0 もしくは -0 なのでそのまま。
        }
        else if (exp > 127)
        {
            // 無限大の処理
            // TODO: IEEE float の無限大の表現がGPU上で有効なのかどうかを要確認。
            bits11 = (u32)0x7F << FRACTION_WIDTH11;
        }
        else
        {
            bits11 |= fraction & FRACTION_MASK11;
            bits11 |= ((u32)exp & 0x7F) << FRACTION_WIDTH11;
        }

        return static_cast<u16>(bits11);
    }

private:
    f32 m_Float32;

    enum
    {
        SIGN32 = 0x80000000,

        EXP_BIAS32 = 127,
        EXP_BIAS11 = 15,
        EXP_MASK32 = 0x7F800000,
        EXP_MASK11 = 0x000007C0,

        FRACTION_WIDTH32 = 23,
        FRACTION_MASK32  = 0x007FFFFF,
        FRACTION_WIDTH11 = 6,
        FRACTION_MASK11  = 0x00000003F
    };
};

}}} // namespace nw::g3d::ut

#endif //  NW_G3D_UT_FLOAT_H_
