﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_G3D_RES_RESCOMMON_H_
#define NW_G3D_RES_RESCOMMON_H_

#include <nw/g3d/g3d_config.h>
#include <nw/g3d/ut/g3d_Inlines.h>
#include <nw/g3d/res/g3d_ResDefs.h>
#include <cstring>

NW_G3D_PRAGMA_PUSH_WARNINGS
NW_G3D_DISABLE_WARNING_SHADOW

namespace nw { namespace g3d { namespace res {

#define NW_G3D_VALIDITY_ASSERT \
    NW_G3D_ASSERTMSG(this, "%s::%s: Object not valid.", GetClassName(), __FUNCTION__)

// NW_G3D_RES_COMMON_ALIGN
//
// リソースアクセサクラスを定義するためのマクロ。
// class_nameにはリソースアクセサクラスのクラス名を入れる。
// 4バイトアラインメントでなくてはならない
#define NW_G3D_RES_COMMON_ALIGN(class_name, alignment)                                             \
public:                                                                                            \
    /* ---------------------------------------- */                                                 \
    /*! @name 共通 */                                                                              \
    /*@{ */                                                                                        \
    /*! @briefprivate */                                                                           \
    typedef class_name##Data DataType;                                                             \
    /*! @brief class_name##Data へのポインタを取得します。 */                                      \
    DataType* ptr() { return this; }                                                               \
    /*! @brief class_name##Data へのポインタを取得します。 */                                      \
    const DataType* ptr() const { return this; }                                                   \
    /*! @brief class_name##Data への参照を取得します。 */                                          \
    DataType& ref() { NW_G3D_VALIDITY_ASSERT; return *ptr(); }                                     \
    /*! @brief ref() の別名関数です。*/                                                            \
    const DataType& ToData() const { return ref(); }                                               \
    /*! @brief ref() の別名関数です。*/                                                            \
    DataType& ToData() { return ref(); }                                                           \
    /*! @brief class_name##Data への参照を取得します。 */                                          \
    const DataType& ref() const { NW_G3D_VALIDITY_ASSERT; return *ptr(); }                         \
    /*! @brief class_name##Data を class_name に変換します。 */                                    \
    static class_name* ResCast(DataType* ptr)                                                      \
    { NW_G3D_ASSERT_ADDR_ALIGNMENT(ptr, alignment); return static_cast<class_name*>(ptr); }        \
    /*! @brief class_name##Data を class_name に変換します。 */                                    \
    static const class_name* ResCast(const DataType* ptr)                                          \
    { NW_G3D_ASSERT_ADDR_ALIGNMENT(ptr, alignment); return static_cast<const class_name*>(ptr); }  \
    /*! @brief クラス名を取得します。 */                                                           \
    static const char* GetClassName() { return #class_name; }                                      \
    /*@} */                                                                                        \
private:                                                                                           \
    NW_G3D_DISALLOW_COPY_AND_ASSIGN(class_name)                                                    \


#define NW_G3D_RES_COMMON(class_name) NW_G3D_RES_COMMON_ALIGN(class_name, 4)

//! @brief リソース内での名前を表す構造体です。
struct ResNameData
{
    //! @brief 文字のアライメントサイズです。
    typedef u32 LengthType;
    LengthType len;
    char str[sizeof(LengthType)];   // 実際には長さ len の文字列が入っている。
                                    // 末尾は sizeof(LengthType) バイトアラインメントされていて、
                                    // パディングは0である。
};

//! @brief リソース内での名前を表すクラスです。
class ResName : private ResNameData
{
    NW_G3D_RES_COMMON(ResName);

public:
    using ResNameData::LengthType;

    //! @brief 文字列の長さを返します。
    size_t GetLength() const { return static_cast<size_t>(ref().len); }

    //! @brief 文字列へのポインタを返します。
    const char* GetName() const { return &ref().str[0]; }

    //! @brief 長さ付き文字列と比較します。
    bool Equals(const char* str, size_t len) const
    {
        return (GetLength() == len) && (std::strcmp(GetName(), str) == 0);
    }

    //! @brief ResName と比較します。
    bool Equals(const ResName* rhs) const
    {
        return (ptr() == rhs->ptr()) || Equals(rhs->GetName(), rhs->GetLength());
    }
};

//! @brief リソース内でのデータオフセットを表すクラスです。
class Offset
{
private:
    s32 offset;

public:
    // 共用体に入れるために、コンストラクタを実装しない。

    //! @brief 直接 offset を返す演算子オーバーロードです。
    operator s32() const { return offset; }

    //! @brief 直接 offset に代入する演算子オーバーロードです。
    Offset& operator=(s32 ofs) { offset = ofs; return *this; }

    //! @brief ポインタからオフセットに変換して設定します。
    void  set_ptr(const void* ptr)
    {
        offset = (ptr == NULL) ? 0 :
            static_cast<s32>(reinterpret_cast<intptr_t>(ptr) - reinterpret_cast<intptr_t>(this));
    }

    //! @brief オフセットからポインタに変換して取得します。
    void* to_ptr() { return to_ptr<void>(); }

    //! @brief オフセットからポインタに変換して取得します。
    const void* to_ptr() const { return to_ptr<void>(); }

    //! @brief オフセットからポインタに変換して取得します。
    template<typename T>
    T* to_ptr() { return (offset == 0) ? NULL : AddOffset<T>(this, offset); }

    //! @brief オフセットからポインタに変換して取得します。
    template<typename T>
    const T* to_ptr() const { return (offset == 0) ? NULL : AddOffset<T>(this, offset); }

    //! @brief オフセットテーブルからオフセットを取得します。
    Offset* to_table_ptr() { return static_cast<Offset*>(to_ptr()); }

    //! @brief オフセットテーブルからオフセットを取得します。
    const Offset* to_table_ptr() const { return static_cast<const Offset*>(to_ptr()); }
};

//! @brief 文字列への参照を持つクラスです。
class BinString
{
public:
    //! @brief 文字列へのオフセットです。
    Offset offset;

    //! @brief オフセットから文字列へのポインタに変換します。
    const char* to_ptr() const { return offset.to_ptr<char>(); }

    //! @brief 指定した文字列を参照するように設定します。
    //!
    //! 引数で与えた文字列が破棄された場合、無効なポインタを参照します。
    //!
    void set_ptr(const char* ptr) { offset.set_ptr(ptr); }

    //! @brief リソースでの文字列形式を取得します。
    const ResName* GetResName() const
    {
        return offset == 0 ? NULL : ResName::ResCast(
            AddOffset<ResNameData>(this, offset - sizeof(ResName::LengthType)));
    }
};

//! @brief 実行時に設定されるポインタを持つクラスです。
class BinPtr
{
public:
    union
    {
        u32 addr;
#if NW_G3D_HOST_PTRSIZE == NW_G3D_TARGET_PTRSIZE
        void* ptr;
#endif
    };

    //! @brief 直接 ptr を取得します。
    void* to_ptr() { return to_ptr<void>(); }

    //! @brief 直接 ptr 取得します。
    const void* to_ptr() const { return to_ptr<void>(); }

    //! @brief 直接 ptr を取得します。
    template <typename T>
    T* to_ptr()
    {
#if NW_G3D_HOST_PTRSIZE == NW_G3D_TARGET_PTRSIZE
        return static_cast<T*>(ptr);
#else
        return NULL;
#endif
    }

    //! @brief 直接 ptr 取得します。
    template <typename T>
    const T* to_ptr() const
    {
#if NW_G3D_HOST_PTRSIZE == NW_G3D_TARGET_PTRSIZE
        return static_cast<const T*>(ptr);
#else
        return NULL;
#endif
    }

    //! @brief 直接 ptr に代入する演算子オーバーロードです。
    BinPtr& set_ptr(void* ptr)
    {
#if NW_G3D_HOST_PTRSIZE == NW_G3D_TARGET_PTRSIZE
        this->ptr = ptr;
#else
        NW_G3D_ASSERT(ptr == NULL);
        NW_G3D_UNUSED(ptr);
        this->addr = 0;
#endif
        return *this;
    }
};

//! @brief ファイルヘッダーの構造体です。
struct BinaryFileHeader
{
    enum
    {
        //! @brief エンディアンを判定するための BOM です。
        BYTE_ORDER_MARK = 0xFEFF
    };

    union
    {
        u8 signature[4];
        u32 sigWord;
    };
    union
    {
        u8 version[4];
        u32 verWord;
    };
    u16 byteOrder;
    u16 headerSize;
    u32 fileSize;
};

//! @brief ブロックヘッダーの構造体です。
struct BinaryBlockHeader
{
    union
    {
        u8 signature[4];
        u32 sigWord;
    };
};

//! @brief 外部ファイルの構造体です。
struct ResExternalFileData
{
    Offset offset;
    u32 size;
};

//! @brief 外部ファイルのリソースです。
//!
//! 外部ファイルへの参照を持つクラスです。
//!
class ResExternalFile : private ResExternalFileData
{
    NW_G3D_RES_COMMON(ResExternalFile);
public:
    //----------------------------------------
    //! @name 取得
    //@{

    //! @brief 外部ファイルのサイズを取得します。
    size_t GetSize() const { return ref().size; }

    //! @brief 外部ファイルへのポインタを取得します。
    void* GetData() { return ref().offset.to_ptr(); }

    //! @brief 外部ファイルへのポインタを取得します。
    const void* GetData() const { return ref().offset.to_ptr(); }

    //@}
};

}}} // namespace nw::g3d::res

NW_G3D_PRAGMA_POP_WARNINGS

#endif // NW_G3D_RES_RESCOMMON_H_
