﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_G3D_SHADEROBJ_H_
#define NW_G3D_SHADEROBJ_H_

#include <nw/g3d/g3d_config.h>
#include <nw/g3d/fnd/g3d_GfxObject.h>
#include <nw/g3d/res/g3d_ResShader.h>
#include <nw/g3d/g3d_Sizer.h>

namespace nw { namespace g3d {

//! @brief シェーディングモデルインスタンスです。
class ShadingModelObj
{
public:
    class Builder;
    class InitArg;

    //! @brief インスタンスの構築時に渡すバッファの必要アライメントサイズです。
    enum Alignment
    {
        //! @brief Init() で渡すバッファの必要アライメントサイズです。
        BUFFER_ALIGNMENT        = 4,

        //! @brief SetupBlockBuffer() で渡すバッファの必要アライメントサイズです。
        BLOCK_BUFFER_ALIGNMENT  = GX2_UNIFORM_BLOCK_ALIGNMENT
    };

    //----------------------------------------
    //! @name 構築/破棄
    //@{

    //! @brief コンストラクタです。
    //!
    //! 実際の構築処理は Init() で行います。
    //!
    ShadingModelObj() : m_pRes(NULL), m_pUserPtr(NULL), m_pBufferPtr(NULL), m_pBlockBuffer(NULL) {}

    //! @brief 計算に必要なバッファサイズを計算します。
    static size_t CalcBufferSize(const InitArg& arg);

    //! @brief インスタンスの初期化を行います。
    bool Init(const InitArg& arg, void* pBuffer, size_t bufferSize);

    //! @brief UniformBlock のサイズを計算します。
    size_t CalcBlockBufferSize();

    //! @brief CalcBlockBufferSize() の別名関数です。
    size_t CalculateBlockBufferSize()
    {
        return CalcBlockBufferSize();
    }

    //! @brief UniformBlock を構築します。
    bool SetupBlockBuffer(void* pBuffer, size_t bufferSize);

    //! @brief UniformBlock を破棄します。
    void CleanupBlockBuffer();

    //@}

    //----------------------------------------
    //! @name 更新
    //@{

    //! @brief シェーダプログラムの範囲を更新します。
    bool UpdateShaderRange()
    {
        return m_pRes->FindProgramRange(&m_ShaderRange, m_pShaderKey);
    }

    //! @brief シェーダオプションの描画リソースを計算します。
    //!
    //! GPU から参照されるバッファを書き換えるため、
    //! 前回の描画で GPU が参照し終わった後に呼ぶ必要があります。
    //! フレームバッファのコピーアウト中などに呼ぶことを想定しています。
    //!
    void CalcOptionBlock();

    //! @brief CalcOptionBlock() の別名関数です。
    void CalculateOptionBlock()
    {
        CalcOptionBlock();
    }

    //@}

    //----------------------------------------
    //! @name オプション
    //@{

    //! @brief シェーディングモデル配下のオプションの数を取得します。
    int GetStaticOptionCount() const
    {
        return m_pRes->GetStaticOptionCount();
    }

    //! @brief インデックスからオプションの名前を取得します。
    const char* GetStaticOptionName(int optionIndex) const
    {
        return m_pRes->GetStaticOptionName(optionIndex);
    }

    //! @brief オプションの名前からインデックスを取得します。
    int GetStaticOptionIndex(const char* name) const
    {
        return m_pRes->GetStaticOptionIndex(name);
    }

    //! @brief GetStaticOptionIndex() の別名関数です。
    int FindStaticOptionIndex(const char* name) const
    {
        return GetStaticOptionIndex(name);
    }

    //! @brief インデックス引きでオプションを取得します。
    ResShaderOption* GetStaticOption(int optionIndex)
    {
        return m_pRes->GetStaticOption(optionIndex);
    }

    //! @brief インデックス引きでオプションを取得します。
    const ResShaderOption* GetStaticOption(int optionIndex) const
    {
        return m_pRes->GetStaticOption(optionIndex);
    }

    //! @brief 名前引きでオプションを取得します。
    ResShaderOption* GetStaticOption(const char* name)
    {
        return m_pRes->GetStaticOption(name);
    }

    //! @brief GetStaticOption() の別名関数です。
    ResShaderOption* FindStaticOption(const char* name)
    {
        return GetStaticOption(name);
    }

    //! @brief 名前引きでオプションを取得します。
    const ResShaderOption* GetStaticOption(const char* name) const
    {
        return m_pRes->GetStaticOption(name);
    }

    //! @brief GetStaticOption() の別名関数です。
    const ResShaderOption* FindStaticOption(const char* name) const
    {
        return GetStaticOption(name);
    }

    //! @brief シェーダキーをデフォルトに書き換えます。
    void ClearStaticKey();

    //! @brief 指定した Choice でシェーダキーを書き換えます。
    void WriteStaticKey(int optionIndex, int choiceIndex);

    //! @brief シェーダキーから Choice を読み込みます。
    int ReadStaticKey(int optionIndex) const;

    //! @brief シェーダキーを取得します。
    //!
    //! シェーダプログラムの探索に使用するキーを返します。
    //! ブランチ機能を有効にしたオプションの値は反映されません。
    //!
    const bit32* GetStaticKey() const { return m_pShaderKey; }

    //! @brief シェーダプログラムの検索に使用する static オプションのキーの bit32 換算の長さを取得します。
    int GetStaticKeyLength() const { return m_pRes->GetStaticKeyLength(); }

    //@}

    //----------------------------------------
    //! @name 取得/設定
    //@{

    //! @brief リソースを取得します。
    ResShadingModel* GetResource() { return m_pRes; }

    //! @brief リソースを取得します。
    const ResShadingModel* GetResource() const { return m_pRes; }

    //! @brief 現在のシェーダプログラムの範囲を取得します。
    const ShaderRange& GetShaderRange() const { return m_ShaderRange; }

    //! @brief Init() 時に渡されたバッファを取得します。
    void* GetBufferPtr() { return m_pBufferPtr; }

    //! @brief SetupBlockBuffer() 時に渡されたバッファを取得します。
    void* GetBlockBufferPtr() { return m_pBlockBuffer; }

    //! @brief モデル配下のオブジェクトの UniformBlock が構築済みであるかどうかを取得します。
    bool IsBlockBufferValid() const { return (m_Flag & BLOCK_BUFFER_VALID) != 0; }

    //! @brief シェーディングモデルが保持する UniformBlock を取得します。
    GfxBuffer& GetOptionBlock() { return m_OptionBlock; }

    //! @brief シェーディングモデルが保持する UniformBlock を取得します。
    const GfxBuffer& GetOptionBlock() const { return m_OptionBlock; }

    //! @brief ユーザポインタを設定します。
    void SetUserPtr(void* pUserPtr) { m_pUserPtr = pUserPtr; }

    //! @brief ユーザポインタを取得します。
    void* GetUserPtr() { return m_pUserPtr; }

    //! @brief ユーザポインタを取得します。
    const void* GetUserPtr() const { return m_pUserPtr; }

    //! @brief ユーザポインタを取得します。
    template <typename T>
    T* GetUserPtr() { return static_cast<T*>(m_pUserPtr); }

    //! @brief ユーザポインタを取得します。
    template <typename T>
    const T* GetUserPtr() const { return static_cast<const T*>(m_pUserPtr); }

    //@}

    //----------------------------------------
    //! @name デバッグ用
    //@{

    //! @brief キーを文字列に変換します。
    //!
    //! 戻り値は終端文字を含まない文字列長です。pStr が NULL でない場合は必ず終端されます。
    //! strLength が必要なサイズに満たない場合は戻り値は負になります。
    //! pStr を NULL、strLength を 0 にすることで終端文字を含まない文字列長を返します。
    //!
    int PrintKeyTo(char* pStr, int strLength) const;

    //! @brief バリエーション探索に使用するキーを文字列に変換します。
    //!
    //! 戻り値は終端文字を含まない文字列長です。pStr が NULL でない場合は必ず終端されます。
    //! strLength が必要なサイズに満たない場合は戻り値は負になります。
    //! pStr を NULL、strLength を 0 にすることで終端文字を含まない文字列長を返します。
    //!
    int PrintRawKeyTo(char* pStr, int strLength) const;

    //! @brief キーのオプション表現を文字列に変換します。
    //!
    //! 戻り値は終端文字を含まない文字列長です。pStr が NULL でない場合は必ず終端されます。
    //! strLength が必要なサイズに満たない場合は戻り値は負になります。
    //! pStr を NULL、strLength を 0 にすることで終端文字を含まない文字列長を返します。
    //!
    int PrintOptionTo(char* pStr, int strLength) const;

    //! @brief バリエーション探索に使用するキーのオプション表現を文字列に変換します。
    //!
    //! 戻り値は終端文字を含まない文字列長です。pStr が NULL でない場合は必ず終端されます。
    //! strLength が必要なサイズに満たない場合は戻り値は負になります。
    //! pStr を NULL、strLength を 0 にすることで終端文字を含まない文字列長を返します。
    //!
    int PrintRawOptionTo(char* pStr, int strLength) const;

    //@}

protected:
    class Sizer;

    //! @brief モデルの状態を表すフラグです。
    enum Flag
    {
        //! @briefprivate UniformBlock が構築済みであるかどうかを表すフラグです。
        //!
        //!
        BLOCK_BUFFER_VALID  = 0x1 << 0,

        //! @briefprivate シェーダキーの UniformBlock が更新されたことを表すフラグです。
        //!
        //!
        BRANCH_DIRTY         = 0x1 << 2
    };

    //! @brief シェーダオプションに DirtyFlag を設定します。
    //!
    //! DirtyFlag が設定されたシェーダパラメータは、
    //! CalcMatBlock() を呼び出すことで UniformBlock に反映されます。
    //!
    void SetDirtyFlag(int optionIndex)
    {
        NW_G3D_ASSERT_INDEX_BOUNDS_DETAIL(optionIndex, GetStaticOptionCount(), "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));
        m_Flag |= BRANCH_DIRTY;
        m_pDirtyFlagArray[optionIndex >> 5] |= 0x1 << (optionIndex & 0x1F);
    }

private:

    ResShadingModel* m_pRes;

    bit32 m_Flag;
    bit32* m_pShaderKey;
    bit32* m_pOptionKey; // UniformBlock を使用するオプションの値を記録しておくキー。
    bit32* m_pDirtyFlagArray;
    ShaderRange m_ShaderRange;
    GfxBuffer m_OptionBlock;

    void* m_pUserPtr;
    void* m_pBufferPtr;
    void* m_pBlockBuffer;

    NW_G3D_DISALLOW_COPY_AND_ASSIGN(ShadingModelObj);
};

//--------------------------------------------------------------------------------------------------

//! @briefprivate ShadingModelObj のサイズを計算するためのクラスです。
//!
//!
class ShadingModelObj::Sizer : public nw::g3d::Sizer
{
public:

    //! @brief コンストラクタです。
    Sizer() : nw::g3d::Sizer() {}

    //! @brief ShadingModelObj::InitArg に基づいてサイズを計算します。
    void Calc(const InitArg& arg);

    enum
    {
        SHADER_KEY,
        OPTION_KEY,
        DIRTY_FLAG_ARRAY,
        NUM_CHUNK
    };

    Chunk chunk[NUM_CHUNK];
};

//--------------------------------------------------------------------------------------------------

//! @brief ShadingModelObj::Init() に渡して初期化を行うパラメータです。
class ShadingModelObj::InitArg
{
public:
    //! コンストラクタです。
    explicit InitArg(ResShadingModel* resource)
        : m_pRes(resource)
    {
        NW_G3D_ASSERT_NOT_NULL(resource);
    }

    //! @brief リソースを取得します。
    ResShadingModel* GetResource() const { return m_pRes; }

    //! @brief サイズ計算用のオブジェクトを取得します。
    Sizer& GetSizer() const { return m_Sizer; }

private:
    ResShadingModel* m_pRes;
    mutable Sizer m_Sizer; // キャッシュするために mutable にする
};

//--------------------------------------------------------------------------------------------------

//! @brief ShaderParamAnimObj の構築を行うクラスです。
class ShadingModelObj::Builder : public ShadingModelObj::InitArg
{
public:
    //! @brief コンストラクタです。
    Builder(ResShadingModel* resource) : InitArg(resource), m_Size(0), m_IsCalculated(false)
    {
    }

    //! @brief ShadingModelObj を構築します。
    //!
    //! @param[in] pShadingModelObj ShadingModelObj へのポインタ
    //! @param[in] pBuffer バッファへのポインタ
    //! @param[in] bufferSize バッファのサイズ
    //!
    //! @pre
    //! - CalculateMemorySize() を呼び、メモリサイズが計算済みである
    //! - bufferSize >= GetWorkMemorySize() で返すサイズ
    //!
    bool Build(ShadingModelObj* pShadingModelObj, void* pBuffer, size_t bufferSize) const
    {
        return pShadingModelObj->Init(*this, pBuffer, bufferSize);
    }

    //! @brief ShadingModelObj 構築に必要なメモリサイズを計算します。
    void CalculateMemorySize()
    {
        m_Size = CalcBufferSize(*this);
        m_IsCalculated = true;
    }

    //! @brief ShadingModelObj 構築に必要なメモリサイズを取得します。
    //!
    //! @return  ShadingModelObj 構築に必要なメモリサイズを返します。
    //!
    size_t GetWorkMemorySize() const
    {
        return m_Size;
    }

    //! @brief ShadingModelObj 構築に必要なメモリサイズが計算済みかを取得します。
    //!
    //! @return 計算済みの場合は true、未計算の場合は false を返します。
    //!
    bool IsMemoryCalculated() const
    {
        return m_IsCalculated;
    }

private:
    size_t m_Size;
    bool m_IsCalculated;
};

//--------------------------------------------------------------------------------------------------

//! @brief 動的シェーダバリエーションのセレクタです。
class ShaderSelector
{
public:
    class Builder;
    class InitArg;

    //! @brief インスタンスの構築時に渡すバッファの必要アライメントサイズです。
    enum Alignment
    {
        //! @brief Init() で渡すバッファの必要アライメントサイズです。
        BUFFER_ALIGNMENT        = 4
    };

    //----------------------------------------
    //! @name 構築/破棄
    //@{

    //! @brief コンストラクタです。
    //!
    //! 実際の構築処理は Init() で行います。
    //!
    ShaderSelector() : m_pShadingModel(NULL), m_pBufferPtr(NULL) {}

    //! @brief 計算に必要なバッファサイズを計算します。
    static size_t CalcBufferSize(const InitArg& arg);

    //! @brief インスタンスの初期化を行います。
    bool Init(const InitArg& arg, void* pBuffer, size_t bufferSize);

    //@}

    //----------------------------------------
    //! @name 更新
    //@{

    bool UpdateVariation();

    //@}

    //----------------------------------------
    //! @name オプション
    //@{

    //! @brief シェーディングモデル配下のオプションの数を取得します。
    int GetDynamicOptionCount() const
    {
        return m_pShadingModel->GetResource()->GetDynamicOptionCount();
    }

    //! @brief インデックスからオプションの名前を取得します。
    const char* GetDynamicOptionName(int optionIndex) const
    {
        return m_pShadingModel->GetResource()->GetDynamicOptionName(optionIndex);
    }

    //! @brief オプションの名前からインデックスを取得します。
    int GetDynamicOptionIndex(const char* name) const
    {
        return m_pShadingModel->GetResource()->GetDynamicOptionIndex(name);
    }

    //! @brief GetDynamicOptionIndex() の別名関数です。
    int FindDynamicOptionIndex(const char* name) const
    {
        return GetDynamicOptionIndex(name);
    }

    //! @brief インデックス引きでオプションを取得します。
    ResShaderOption* GetDynamicOption(int optionIndex)
    {
        return m_pShadingModel->GetResource()->GetDynamicOption(optionIndex);
    }

    //! @brief インデックス引きでオプションを取得します。
    const ResShaderOption* GetDynamicOption(int optionIndex) const
    {
        return m_pShadingModel->GetResource()->GetDynamicOption(optionIndex);
    }

    //! @brief 名前引きでオプションを取得します。
    ResShaderOption* GetDynamicOption(const char* name)
    {
        return m_pShadingModel->GetResource()->GetDynamicOption(name);
    }

    //! @brief GetDynamicOption() の別名関数です。
    ResShaderOption* FindDynamicOption(const char* name)
    {
        return GetDynamicOption(name);
    }

    //! @brief 名前引きでオプションを取得します。
    const ResShaderOption* GetDynamicOption(const char* name) const
    {
        return m_pShadingModel->GetResource()->GetDynamicOption(name);
    }

    //! @brief GetDynamicOption() の別名関数です。
    const ResShaderOption* FindDynamicOption(const char* name) const
    {
        return GetDynamicOption(name);
    }

    //! @brief シェーダキーをデフォルトに書き換えます。
    void ClearDynamicKey();

    //! @brief 指定した Choice でシェーダキーを書き換えます。
    void WriteDynamicKey(int optionIndex, int choiceIndex);

    //! @brief シェーダキーから Choice を読み込みます。
    int ReadDynamicKey(int optionIndex) const;

    //! @brief シェーダキーを取得します。
    //!
    //! シェーダプログラムの探索に使用するキーを返します。
    //! ブランチ機能を有効にしたオプションの値は反映されません。
    //!
    const bit32* GetDynamicKey() const { return m_pShaderKey; }

    //! @brief シェーダプログラムの検索に使用する dynamic オプションのキーの bit32 換算の長さを取得します。
    int GetDynamicKeyLength() const
    {
        return m_pShadingModel->GetResource()->GetDynamicKeyLength();
    }

    //@}

    //----------------------------------------
    //! @name 取得/設定
    //@{

    //! @brief シェーディングモデルを取得します。
    ShadingModelObj* GetShadingModel() { return m_pShadingModel; }

    //! @brief シェーディングモデルを取得します。
    const ShadingModelObj* GetShadingModel() const { return m_pShadingModel; }

    //! @brief 現在のシェーダプログラムを取得します。
    ResShaderProgram* GetProgram() { return m_pProgram; }

    //! @brief 現在のシェーダプログラムを取得します。
    const ResShaderProgram* GetProgram() const { return m_pProgram; }

    //! @brief 現在のシェーダプログラムを取得します。
    ResShaderProgram* GetDefaultProgram()
    {
        ResShadingModel* pRes = m_pShadingModel->GetResource();
        int idxProgram = pRes->GetDefaultProgramIndex();
        return idxProgram == SHADER_PROGRAM_NONE ? NULL : pRes->GetShaderProgram(idxProgram);
    }

    //! @brief 現在のシェーダプログラムを取得します。
    const ResShaderProgram* GetDefaultProgram() const
    {
        const ResShadingModel* pRes = m_pShadingModel->GetResource();
        int idxProgram = pRes->GetDefaultProgramIndex();
        return idxProgram == SHADER_PROGRAM_NONE ? NULL : pRes->GetShaderProgram(idxProgram);
    }

    //! @brief Init() 時に渡されたバッファを取得します。
    void* GetBufferPtr() { return m_pBufferPtr; }

    //@}

    //----------------------------------------
    //! @name デバッグ用
    //@{

    //! @brief キーを文字列に変換します。
    //!
    //! 戻り値は終端文字を含まない文字列長です。pStr が NULL でない場合は必ず終端されます。
    //! strLength が必要なサイズに満たない場合は戻り値は負になります。
    //! pStr を NULL、strLength を 0 にすることで終端文字を含まない文字列長を返します。
    //!
    int PrintKeyTo(char* pStr, int strLength) const;

    //! @brief バリエーション探索に使用するキーを文字列に変換します。
    //!
    //! 戻り値は終端文字を含まない文字列長です。pStr が NULL でない場合は必ず終端されます。
    //! strLength が必要なサイズに満たない場合は戻り値は負になります。
    //! pStr を NULL、strLength を 0 にすることで終端文字を含まない文字列長を返します。
    //!
    int PrintRawKeyTo(char* pStr, int strLength) const;

    //! @brief キーのオプション表現を文字列に変換します。
    //!
    //! 戻り値は終端文字を含まない文字列長です。pStr が NULL でない場合は必ず終端されます。
    //! strLength が必要なサイズに満たない場合は戻り値は負になります。
    //! pStr を NULL、strLength を 0 にすることで終端文字を含まない文字列長を返します。
    //!
    int PrintOptionTo(char* pStr, int strLength) const;

    //! @brief バリエーション探索に使用するキーのオプション表現を文字列に変換します。
    //!
    //! 戻り値は終端文字を含まない文字列長です。pStr が NULL でない場合は必ず終端されます。
    //! strLength が必要なサイズに満たない場合は戻り値は負になります。
    //! pStr を NULL、strLength を 0 にすることで終端文字を含まない文字列長を返します。
    //!
    int PrintRawOptionTo(char* pStr, int strLength) const;

    //@}

protected:
    class Sizer;

private:

    ShadingModelObj* m_pShadingModel;

    bit32* m_pShaderKey;
    bit32* m_pOptionKey;
    bit32* m_pLastShaderKey;

    ResShaderProgram* m_pProgram;

    void* m_pBufferPtr;

    NW_G3D_DISALLOW_COPY_AND_ASSIGN(ShaderSelector);
};

//--------------------------------------------------------------------------------------------------

//! @briefprivate ShaderSelector のサイズを計算するためのクラスです。
//!
class ShaderSelector::Sizer : public nw::g3d::Sizer
{
public:

    //! @brief コンストラクタです。
    Sizer() : nw::g3d::Sizer() {}

    //! @brief ShadingModelObj::InitArg に基づいてサイズを計算します。
    void Calc(const InitArg& arg);

    enum
    {
        SHADER_KEY,
        OPTION_KEY,
        LAST_SHADER_KEY,
        NUM_CHUNK
    };

    Chunk chunk[NUM_CHUNK];
};

//--------------------------------------------------------------------------------------------------

//! @brief ShaderSelector::Init() に渡して初期化を行うパラメータです。
class ShaderSelector::InitArg
{
public:
    //! コンストラクタです。
    explicit InitArg(ShadingModelObj* pShadingModel)
        : m_pShadingModel(pShadingModel)
    {
        NW_G3D_ASSERT_NOT_NULL(pShadingModel);
    }

    //! @brief シェーディングモデルを取得します。
    ShadingModelObj* GetShadingModel() const { return m_pShadingModel; }

    //! @brief サイズ計算用のオブジェクトを取得します。
    Sizer& GetSizer() const { return m_Sizer; }

private:
    ShadingModelObj* m_pShadingModel;
    mutable Sizer m_Sizer; // キャッシュするために mutable にする
};

//! @brief ShaderSelector の構築を行うクラスです。
class ShaderSelector::Builder : public ShaderSelector::InitArg
{
public:
    //! @brief コンストラクタです。
    Builder(ShadingModelObj* pShadingModel) : InitArg(pShadingModel), m_Size(0), m_IsCalculated(false)
    {
    }

    //! @brief ShaderSelector を構築します。
    //!
    //! @param[in] pShaderSelector ShaderSelector へのポインタ
    //! @param[in] pBuffer バッファへのポインタ
    //! @param[in] bufferSize バッファのサイズ
    //!
    //! @pre
    //! - CalculateMemorySize() を呼び、メモリサイズが計算済みである
    //! - bufferSize >= GetWorkMemorySize() で返すサイズ
    //!
    bool Build(ShaderSelector* pShaderSelector, void* pBuffer, size_t bufferSize) const
    {
        return pShaderSelector->Init(*this, pBuffer, bufferSize);
    }

    //! @brief ShaderSelector 構築に必要なメモリサイズを計算します。
    void CalculateMemorySize()
    {
        m_Size = CalcBufferSize(*this);
        m_IsCalculated = true;
    }

    //! @brief ShaderSelector 構築に必要なメモリサイズを取得します。
    //!
    //! @return  ShaderSelector 構築に必要なメモリサイズを返します。
    //!
    size_t GetWorkMemorySize() const
    {
        return m_Size;
    }

    //! @brief ShaderSelector 構築に必要なメモリサイズが計算済みかを取得します。
    //!
    //! @return 計算済みの場合は true、未計算の場合は false を返します。
    //!
    bool IsMemoryCalculated() const
    {
        return m_IsCalculated;
    }

private:
    size_t m_Size;
    bool m_IsCalculated;
};

}} // namespace nw::g3d

#endif // NW_G3D_SHADEROBJ_H_
