﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_G3D_SCENEANIMOBJ_H_
#define NW_G3D_SCENEANIMOBJ_H_

#include <nw/g3d/g3d_config.h>
#include <nw/g3d/g3d_AnimObj.h>
#include <nw/g3d/res/g3d_ResSceneAnim.h>

namespace nw { namespace g3d {

//! @brief シーンアニメーションインスタンスの基底クラスです。
class SceneAnimObj : public AnimObj
{
public:
    //----------------------------------------
    //! @name 取得/設定
    //@{

    //! @brief ユーザポインタを設定します。
    void SetUserPtr(void* pUserPtr) { m_pUserPtr = pUserPtr; }

    //! @brief ユーザポインタを取得します。
    void* GetUserPtr() { return m_pUserPtr; }

    //! @brief ユーザポインタを取得します。
    const void* GetUserPtr() const { return m_pUserPtr; }

    //! @brief ユーザポインタを取得します。
    template <typename T>
    T* GetUserPtr() { return static_cast<T*>(m_pUserPtr); }

    //! @brief ユーザポインタを取得します。
    template <typename T>
    const T* GetUserPtr() const { return static_cast<const T*>(m_pUserPtr); }

    //@}
protected:
    //! @briefprivate
    SceneAnimObj() : AnimObj(), m_pUserPtr(NULL) {}

private:
    void* m_pUserPtr;

    NW_G3D_DISALLOW_COPY_AND_ASSIGN(SceneAnimObj);
};

//--------------------------------------------------------------------------------------------------

//! @brief カメラアニメーションインスタンスです。
class CameraAnimObj : public SceneAnimObj
{
public:
    class Builder;
    class InitArg;

    //! @brief インスタンスの構築時に渡すバッファの必要アライメントサイズです。
    enum Alignment
    {
        //! @brief Init() で渡すバッファの必要アライメントサイズです。
        BUFFER_ALIGNMENT        = 4,
    };

    //----------------------------------------
    //! @name 構築/破棄
    //@{

    //! @brief コンストラクタです。
    //!
    //! 実際の構築処理は Init() で行います。
    //!
    CameraAnimObj() : SceneAnimObj(), m_pRes(NULL) {}

    //! @brief インスタンスの初期化を行います。
    bool Init(const InitArg& arg, void* pBuffer, size_t bufferSize);

    //! @brief 計算に必要なバッファサイズを計算します。
    static size_t CalcBufferSize(const InitArg& arg);

    //@}

    //----------------------------------------
    //! @name 更新
    //@{

    virtual void ClearResult();
    virtual void Calc();
    //! @brief Calc() の別名関数です。
    void Calculate()
    {
        Calc();
    }

    //@}

    //----------------------------------------
    //! @name 取得/設定
    //@{

    //! @brief リソースを取得します。
    const ResCameraAnim* GetResource() const { return m_pRes; };

    //! @brief リソースを設定します。
    //!
    //! 計算するカメラアニメーションを差し替えます。
    //!
    void SetResource(ResCameraAnim* pRes);

    //! @brief リソースを取り外します。
    void ResetResource() { m_pRes = NULL; }

    //! @brief アニメーションの計算結果を取得します。
    CameraAnimResult* GetResult()
    {
        return static_cast<CameraAnimResult*>(GetResultBuffer());
    }

    //! @brief アニメーションの計算結果を取得します。
    const CameraAnimResult* GetResult() const
    {
        return static_cast<const CameraAnimResult*>(GetResultBuffer());
    }

    //! @brief 回転の計算方式を取得します。
    bit32 GetRotateMode() const { return m_pRes->GetRotateMode(); }

    //! @brief プロジェクションの計算方式を取得します。
    bit32 GetProjectionMode() const { return m_pRes->GetProjectionMode(); }

    //@}

private:
    class Sizer;
    ResCameraAnim* m_pRes;

    NW_G3D_DISALLOW_COPY_AND_ASSIGN(CameraAnimObj);
};

//--------------------------------------------------------------------------------------------------

//! @briefprivate CameraAnimObj のサイズを計算するためのクラスです。
//!
//!
//!
class CameraAnimObj::Sizer : public nw::g3d::Sizer
{
public:
    //! @brief コンストラクタです。
    Sizer() : nw::g3d::Sizer() {}

    //! @brief CameraAnimObj::InitArg に基づいてサイズを計算します。
    void Calc(const InitArg& arg);

    enum
    {
        RESULT_BUFFER,
        FRAMECACHE_ARRAY,
        NUM_CHUNK
    };

    Chunk chunk[NUM_CHUNK];
};

//--------------------------------------------------------------------------------------------------

//! @brief CameraAnimObj::Init() に渡して初期化を行うパラメータです。
class CameraAnimObj::InitArg
{
public:
    //! @brief コンストラクタです。
    //!
    //! インスタンスで使用するすべての ResCameraAnim に対して Reserve() するか、
    //! SetMaxCurveCount() でバッファサイズの計算に必要なパラメータを指定する必要があります。
    //!
    InitArg() { Clear(); }

    //! @brief パラメータを初期化します。
    void Clear()
    {
        m_NumCurve = -1;
        m_ContextEnabled = true;
        m_ContextAvailable = false;
    }

    //! @brief 対象となる ResCameraAnim を使用するのに必要なパラメータを設定します。
    void Reserve(const ResCameraAnim* pResAnim)
    {
        NW_G3D_ASSERT_NOT_NULL(pResAnim);
        m_NumCurve = std::max(m_NumCurve, pResAnim->GetCurveCount());
        m_ContextAvailable |= !pResAnim->IsCurveBaked();
        m_Sizer.Invalidate();
    }

    //! @brief アニメーション高速化のためのコンテクストを有効にします。
    void EnableContext() { m_ContextEnabled = true; m_Sizer.Invalidate(); }

    //! @brief アニメーション高速化のためのコンテクストを無効にします。
    void DisableContext() { m_ContextEnabled = false; m_Sizer.Invalidate(); }

    //! @brief アニメーション高速化のためのコンテクストが有効かどうかを取得します。
    bool IsContextEnabled() const { return m_ContextAvailable && m_ContextEnabled; }

    //! @brief 使用できる最大のカーブ数を指定します。
    void SetMaxCurveCount(int curveCount)
    {
        NW_G3D_ASSERT(curveCount >= 0);
        m_NumCurve = curveCount;
        m_ContextAvailable = true;
        m_Sizer.Invalidate();
    }

    //! @brief 使用できる最大のカーブ数を取得します。
    int GetMaxCurveCount() const { return m_NumCurve; }

    //! @brief 必要なパラメータが設定されているかどうかを取得します。
    bool IsValid() const { return m_NumCurve >= 0; }

    //! @brief サイズ計算用のオブジェクトを取得します。
    Sizer& GetSizer() const { return m_Sizer; }

private:
    int m_NumCurve;
    bool m_ContextEnabled;
    bool m_ContextAvailable; // リソースがベイクされていないか、カーブ数を手動指定した場合に true
    mutable Sizer m_Sizer; // キャッシュするために mutable にする
};

//--------------------------------------------------------------------------------------------------

//! @brief CameraAnimObj の構築を行うクラスです。
class CameraAnimObj::Builder : public CameraAnimObj::InitArg
{
public:
    //! @brief コンストラクタです。
    //!
    //! @details
    //! インスタンスで使用するすべての ResCameraAnim に対して Reserve() するか、
    //! SetMaxCurveCount() でバッファサイズの計算に必要なパラメータを指定する必要があります。
    //!
    Builder() : m_Size(0), m_IsCalculated(false)
    {
    }

    //! @brief CameraAnimObj を構築します。
    //!
    //! @param[in] pCameraAnimObj CameraAnimObj へのポインタ
    //! @param[in] pBuffer バッファへのポインタ
    //! @param[in] bufferSize バッファのサイズ
    //!
    //! @pre
    //! - CalculateMemorySize() を呼び、メモリサイズが計算済みである
    //! - bufferSize >= GetWorkMemorySize() で返すサイズ
    //!
    bool Build(CameraAnimObj* pCameraAnimObj, void* pBuffer, size_t bufferSize) const
    {
        return pCameraAnimObj->Init(*this, pBuffer, bufferSize);
    }

    //! @brief CameraAnimObj 構築に必要なメモリサイズを計算します。
    void CalculateMemorySize()
    {
        m_Size = CalcBufferSize(*this);
        m_IsCalculated = true;
    }

    //! @brief CameraAnimObj 構築に必要なメモリサイズを取得します。
    //!
    //! @return  CameraAnimObj 構築に必要なメモリサイズを返します。
    //!
    size_t GetWorkMemorySize() const
    {
        return m_Size;
    }

    //! @brief CameraAnimObj 構築に必要なメモリサイズが計算済みかを取得します。
    //!
    //! @return 計算済みの場合は true、未計算の場合は false を返します。
    //!
    bool IsMemoryCalculated() const
    {
        return m_IsCalculated;
    }

private:
    size_t m_Size;
    bool m_IsCalculated;
};

//--------------------------------------------------------------------------------------------------

//! @brief ライトアニメーションインスタンスです。
class LightAnimObj : public SceneAnimObj
{
public:
    class Builder;
    class InitArg;

    //! @brief インスタンスの構築時に渡すバッファの必要アライメントサイズです。
    enum Alignment
    {
        //! @brief Init() で渡すバッファの必要アライメントサイズです。
        BUFFER_ALIGNMENT        = 4,
    };

    //----------------------------------------
    //! @name 構築/破棄
    //@{

    //! @brief コンストラクタです。
    //!
    //! 実際の構築処理は Init() で行います。
    //!
    LightAnimObj() : SceneAnimObj(), m_pRes(NULL) {}

    //! @brief インスタンスの初期化を行います。
    bool Init(const InitArg& arg, void* pBuffer, size_t bufferSize);

    //! @brief 計算に必要なバッファサイズを計算します。
    static size_t CalcBufferSize(const InitArg& arg);

    //@}

    //----------------------------------------
    //! @name 更新
    //@{

    virtual void ClearResult();
    virtual void Calc();
    //! @brief Calc() の別名関数です。
    void Calculate()
    {
        Calc();
    }

    //@}

    //----------------------------------------
    //! @name 取得/設定
    //@{

    //! @brief リソースを取得します。
    const ResLightAnim* GetResource() const { return m_pRes; };

    //! @brief リソースを設定します。
    //!
    //! 計算するライトアニメーションを差し替えます。
    //!
    void SetResource(ResLightAnim* pRes);

    //! @brief リソースを取り外します。
    void ResetResource() { m_pRes = NULL; }

    //! @brief アニメーションの計算結果を取得します。
    LightAnimResult* GetResult()
    {
        return static_cast<LightAnimResult*>(GetResultBuffer());
    }

    //! @brief アニメーションの計算結果を取得します。
    const LightAnimResult* GetResult() const
    {
        return static_cast<const LightAnimResult*>(GetResultBuffer());
    }

    //! @brief リソースに関連付けたライトタイプのインデックスを取得します。
    int GetLightTypeIndex() const { return m_pRes->GetLightTypeIndex(); }

    //! @brief リソースに関連付けた距離減衰関数のインデックスを取得します。
    int GetDistAttnFuncIndex() const { return m_pRes->GetDistAttnFuncIndex(); }

    //! @brief リソースに関連付けた角度減衰関数のインデックスを取得します。
    int GetAngleAttnFuncIndex() const { return m_pRes->GetAngleAttnFuncIndex(); }

    //@}

private:
    class Sizer;
    ResLightAnim* m_pRes;

    NW_G3D_DISALLOW_COPY_AND_ASSIGN(LightAnimObj);
};

//--------------------------------------------------------------------------------------------------

//! @briefprivate LightAnimObj のサイズを計算するためのクラスです。
//!
//!
//!
class LightAnimObj::Sizer : public nw::g3d::Sizer
{
public:
    //! @brief コンストラクタです。
    Sizer() : nw::g3d::Sizer() {}

    //! @brief LightAnimObj::InitArg に基づいてサイズを計算します。
    void Calc(const InitArg& arg);

    enum
    {
        RESULT_BUFFER,
        FRAMECACHE_ARRAY,
        NUM_CHUNK
    };

    Chunk chunk[NUM_CHUNK];
};

//--------------------------------------------------------------------------------------------------

//! @brief LightAnimObj::Init() に渡して初期化を行うパラメータです。
class LightAnimObj::InitArg
{
public:
    //! @brief コンストラクタです。
    //!
    //! インスタンスで使用するすべての ResLightAnim に対して Reserve() するか、
    //! SetMaxCurveCount() でバッファサイズの計算に必要なパラメータを指定する必要があります。
    //!
    InitArg() { Clear(); }

    //! @brief パラメータを初期化します。
    void Clear()
    {
        m_NumCurve = -1;
        m_ContextEnabled = true;
        m_ContextAvailable = false;
    }

    //! @brief 対象となる ResLightAnim を使用するのに必要なパラメータを設定します。
    void Reserve(const ResLightAnim* pResAnim)
    {
        NW_G3D_ASSERT_NOT_NULL(pResAnim);
        m_NumCurve = std::max(m_NumCurve, pResAnim->GetCurveCount());
        m_ContextAvailable |= !pResAnim->IsCurveBaked();
        m_Sizer.Invalidate();
    }

    //! @brief アニメーション高速化のためのコンテクストを有効にします。
    void EnableContext() { m_ContextEnabled = true; m_Sizer.Invalidate(); }

    //! @brief アニメーション高速化のためのコンテクストを無効にします。
    void DisableContext() { m_ContextEnabled = false; m_Sizer.Invalidate(); }

    //! @brief アニメーション高速化のためのコンテクストが有効かどうかを取得します。
    bool IsContextEnabled() const { return m_ContextAvailable && m_ContextEnabled; }

    //! @brief 使用できる最大のカーブ数を指定します。
    void SetMaxCurveCount(int curveCount)
    {
        NW_G3D_ASSERT(curveCount >= 0);
        m_NumCurve = curveCount;
        m_ContextAvailable = true;
        m_Sizer.Invalidate();
    }

    //! @brief 使用できる最大のカーブ数を取得します。
    int GetMaxCurveCount() const { return m_NumCurve; }

    //! @brief 必要なパラメータが設定されているかどうかを取得します。
    bool IsValid() const { return m_NumCurve >= 0; }

    //! @brief サイズ計算用のオブジェクトを取得します。
    Sizer& GetSizer() const { return m_Sizer; }

private:
    int m_NumCurve;
    bool m_ContextEnabled;
    bool m_ContextAvailable; // リソースがベイクされていないか、カーブ数を手動指定した場合に true
    mutable Sizer m_Sizer; // キャッシュするために mutable にする
};

//--------------------------------------------------------------------------------------------------

//! @brief LightAnimObj の構築を行うクラスです。
class LightAnimObj::Builder : public LightAnimObj::InitArg
{
public:
    //! @brief コンストラクタです。
    //!
    //! @details
    //! インスタンスで使用するすべての ResLightAnim に対して Reserve() するか、
    //! SetMaxCurveCount() でバッファサイズの計算に必要なパラメータを指定する必要があります。
    //!
    Builder() : m_Size(0), m_IsCalculated(false)
    {
    }

    //! @brief LightAnimObj を構築します。
    //!
    //! @param[in] pLightAnimObj LightAnimObj へのポインタ
    //! @param[in] pBuffer バッファへのポインタ
    //! @param[in] bufferSize バッファのサイズ
    //!
    //! @pre
    //! - CalculateMemorySize() を呼び、メモリサイズが計算済みである
    //! - bufferSize >= GetWorkMemorySize() で返すサイズ
    //!
    bool Build(LightAnimObj* pLightAnimObj, void* pBuffer, size_t bufferSize) const
    {
        return pLightAnimObj->Init(*this, pBuffer, bufferSize);
    }

    //! @brief LightAnimObj 構築に必要なメモリサイズを計算します。
    void CalculateMemorySize()
    {
        m_Size = CalcBufferSize(*this);
        m_IsCalculated = true;
    }

    //! @brief LightAnimObj 構築に必要なメモリサイズを取得します。
    //!
    //! @return  LightAnimObj 構築に必要なメモリサイズを返します。
    //!
    size_t GetWorkMemorySize() const
    {
        return m_Size;
    }

    //! @brief LightAnimObj 構築に必要なメモリサイズが計算済みかを取得します。
    //!
    //! @return 計算済みの場合は true、未計算の場合は false を返します。
    //!
    bool IsMemoryCalculated() const
    {
        return m_IsCalculated;
    }

private:
    size_t m_Size;
    bool m_IsCalculated;
};

//--------------------------------------------------------------------------------------------------

//! @brief フォグアニメーションインスタンスです。
class FogAnimObj : public SceneAnimObj
{
public:
    class Builder;
    class InitArg;

    //! @brief インスタンスの構築時に渡すバッファの必要アライメントサイズです。
    enum Alignment
    {
        //! @brief Init() で渡すバッファの必要アライメントサイズです。
        BUFFER_ALIGNMENT        = 4,
    };

    //----------------------------------------
    //! @name 構築/破棄
    //@{

    //! @brief コンストラクタです。
    //!
    //! 実際の構築処理は Init() で行います。
    //!
    FogAnimObj() : SceneAnimObj(), m_pRes(NULL) {}

    //! @brief インスタンスの初期化を行います。
    bool Init(const InitArg& arg, void* pBuffer, size_t bufferSize);

    //! @brief 計算に必要なバッファサイズを計算します。
    static size_t CalcBufferSize(const InitArg& arg);

    //@}

    //----------------------------------------
    //! @name 更新
    //@{

    virtual void ClearResult();
    virtual void Calc();
    //! @brief Calc() の別名関数です。
    void Calculate()
    {
        Calc();
    }

    //@}

    //----------------------------------------
    //! @name 取得/設定
    //@{

    //! @brief リソースを取得します。
    const ResFogAnim* GetResource() const { return m_pRes; };

    //! @brief リソースを設定します。
    //!
    //! 計算するフォグアニメーションを差し替えます。
    //!
    void SetResource(ResFogAnim* pRes);

    //! @brief リソースを取り外します。
    void ResetResource() { m_pRes = NULL; }

    //! @brief アニメーションの計算結果を取得します。
    FogAnimResult* GetResult()
    {
        return static_cast<FogAnimResult*>(GetResultBuffer());
    }

    //! @brief アニメーションの計算結果を取得します。
    const FogAnimResult* GetResult() const
    {
        return static_cast<const FogAnimResult*>(GetResultBuffer());
    }

    //! @brief リソースに関連付けた距離減衰関数のインデックスを取得します。
    int GetDistAttnFuncIndex() const { return m_pRes->GetDistAttnFuncIndex(); }

    //@}

private:
    class Sizer;
    ResFogAnim* m_pRes;

    NW_G3D_DISALLOW_COPY_AND_ASSIGN(FogAnimObj);
};

//--------------------------------------------------------------------------------------------------

//! @briefprivate FogAnimObj のサイズを計算するためのクラスです。
//!
class FogAnimObj::Sizer : public nw::g3d::Sizer
{
public:
    //! @brief コンストラクタです。
    Sizer() : nw::g3d::Sizer() {}

    //! @brief FogAnimObj::InitArg に基づいてサイズを計算します。
    void Calc(const InitArg& arg);

    enum
    {
        RESULT_BUFFER,
        FRAMECACHE_ARRAY,
        NUM_CHUNK
    };

    Chunk chunk[NUM_CHUNK];
};

//--------------------------------------------------------------------------------------------------

//! @brief FogAnimObj::Init() に渡して初期化を行うパラメータです。
class FogAnimObj::InitArg
{
public:
    //! @brief コンストラクタです。
    //!
    //! インスタンスで使用するすべての ResFogAnim に対して Reserve() するか、
    //! SetMaxCurveCount() でバッファサイズの計算に必要なパラメータを指定する必要があります。
    //!
    InitArg() { Clear(); }

    //! @brief パラメータを初期化します。
    void Clear()
    {
        m_NumCurve = -1;
        m_ContextEnabled = true;
        m_ContextAvailable = false;
    }

    //! @brief 対象となる ResFogAnim を使用するのに必要なパラメータを設定します。
    void Reserve(const ResFogAnim* pResAnim)
    {
        NW_G3D_ASSERT_NOT_NULL(pResAnim);
        m_NumCurve = std::max(m_NumCurve, pResAnim->GetCurveCount());
        m_ContextAvailable |= !pResAnim->IsCurveBaked();
        m_Sizer.Invalidate();
    }

    //! @brief アニメーション高速化のためのコンテクストを有効にします。
    void EnableContext() { m_ContextEnabled = true; m_Sizer.Invalidate(); }

    //! @brief アニメーション高速化のためのコンテクストを無効にします。
    void DisableContext() { m_ContextEnabled = false; m_Sizer.Invalidate(); }

    //! @brief アニメーション高速化のためのコンテクストが有効かどうかを取得します。
    bool IsContextEnabled() const { return m_ContextAvailable && m_ContextEnabled; }

    //! @brief 使用できる最大のカーブ数を指定します。
    void SetMaxCurveCount(int curveCount)
    {
        NW_G3D_ASSERT(curveCount >= 0);
        m_NumCurve = curveCount;
        m_ContextAvailable = true;
        m_Sizer.Invalidate();
    }

    //! @brief 使用できる最大のカーブ数を取得します。
    int GetMaxCurveCount() const { return m_NumCurve; }

    //! @brief 必要なパラメータが設定されているかどうかを取得します。
    bool IsValid() const { return m_NumCurve >= 0; }

    //! @brief サイズ計算用のオブジェクトを取得します。
    Sizer& GetSizer() const { return m_Sizer; }

private:
    int m_NumCurve;
    bool m_ContextEnabled;
    bool m_ContextAvailable; // リソースがベイクされていないか、カーブ数を手動指定した場合に true
    mutable Sizer m_Sizer; // キャッシュするために mutable にする
};

//--------------------------------------------------------------------------------------------------

//! @brief FogAnimObj の構築を行うクラスです。
class FogAnimObj::Builder : public FogAnimObj::InitArg
{
public:
    //! @brief コンストラクタです。
    //!
    //! @details
    //! インスタンスで使用するすべての ResFogAnim に対して Reserve() するか、
    //! SetMaxCurveCount() でバッファサイズの計算に必要なパラメータを指定する必要があります。
    //!
    Builder() : m_Size(0), m_IsCalculated(false)
    {
    }

    //! @brief FogAnimObj を構築します。
    //!
    //! @param[in] pFogAnimObj FogAnimObj へのポインタ
    //! @param[in] pBuffer バッファへのポインタ
    //! @param[in] bufferSize バッファのサイズ
    //!
    //! @pre
    //! - CalculateMemorySize() を呼び、メモリサイズが計算済みである
    //! - bufferSize >= GetWorkMemorySize() で返すサイズ
    //!
    bool Build(FogAnimObj* pFogAnimObj, void* pBuffer, size_t bufferSize) const
    {
        return pFogAnimObj->Init(*this, pBuffer, bufferSize);
    }

    //! @brief FogAnimObj 構築に必要なメモリサイズを計算します。
    void CalculateMemorySize()
    {
        m_Size = CalcBufferSize(*this);
        m_IsCalculated = true;
    }

    //! @brief FogAnimObj 構築に必要なメモリサイズを取得します。
    //!
    //! @return  FogAnimObj 構築に必要なメモリサイズを返します。
    //!
    size_t GetWorkMemorySize() const
    {
        return m_Size;
    }

    //! @brief FogAnimObj 構築に必要なメモリサイズが計算済みかを取得します。
    //!
    //! @return 計算済みの場合は true、未計算の場合は false を返します。
    //!
    bool IsMemoryCalculated() const
    {
        return m_IsCalculated;
    }

private:
    size_t m_Size;
    bool m_IsCalculated;
};

//--------------------------------------------------------------------------------------------------

namespace math {

class Mtx34;
class Mtx44;

} // namespace math

//! @brief シーンアニメーションの計算補助用関数群です。
class SceneAnimHelper
{
public:

    //! @brief AimCamera のビュー行列を計算します。
    static void CalcAimCameraMtx(Mtx34* pMtx, const CameraAnimResult* pResult);

    //! @brief CalcAimCameraMtx() の別名関数です。
    static void CalculateAimCameraMtx(Mtx34* pMtx, const CameraAnimResult* pResult)
    {
        CalcAimCameraMtx(pMtx, pResult);
    }

    //! @brief RotateCamera のビュー行列を計算します。
    static void CalcRotateCameraMtx(Mtx34* pMtx, const CameraAnimResult* pResult);

    //! @brief CalcRotateCameraMtx() の別名関数です。
    static void CalculateRotateCameraMtx(Mtx34* pMtx, const CameraAnimResult* pResult)
    {
        CalcRotateCameraMtx(pMtx, pResult);
    }

    //! @brief 平行投影のプロジェクション行列を計算します。
    static void CalcOrthoProjMtx(Mtx44* pMtx, const CameraAnimResult* pResult);

    //! @brief CalcOrthoProjMtx() の別名関数です。
    static void CalculateOrthoProjMtx(Mtx44* pMtx, const CameraAnimResult* pResult)
    {
        CalcOrthoProjMtx(pMtx, pResult);
    }

    //! @brief 透視投影のプロジェクション行列を計算します。
    static void CalcPerspProjMtx(Mtx44* pMtx, const CameraAnimResult* pResult);

    //! @brief CalcPerspProjMtx() の別名関数です。
    static void CalculatePerspProjMtx(Mtx44* pMtx, const CameraAnimResult* pResult)
    {
        CalcPerspProjMtx(pMtx, pResult);
    }

    //! @brief 平行投影のプロジェクションテクスチャ行列を計算します。
    static void CalcOrthoProjTexMtx(Mtx34* pMtx, const CameraAnimResult* pResult);

    //! @brief CalcOrthoProjTexMtx() の別名関数です。
    static void CalculateOrthoProjTexMtx(Mtx34* pMtx, const CameraAnimResult* pResult)
    {
        CalcOrthoProjTexMtx(pMtx, pResult);
    }

    //! @brief 透視投影のプロジェクションテクスチャ行列を計算します。
    static void CalcPerspProjTexMtx(Mtx34* pMtx, const CameraAnimResult* pResult);

    //! @brief CalcPerspProjTexMtx() の別名関数です。
    static void CalculatePerspProjTexMtx(Mtx34* pMtx, const CameraAnimResult* pResult)
    {
        CalcPerspProjTexMtx(pMtx, pResult);
    }
};

}} // namespace nw::g3d

#endif // NW_G3D_SCENEANIMOBJ_H_
