﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_G3D_MATERIALOBJ_H_
#define NW_G3D_MATERIALOBJ_H_

#include <nw/g3d/g3d_config.h>
#include <nw/g3d/res/g3d_ResMaterial.h>
#include <nw/g3d/g3d_Sizer.h>

namespace nw { namespace g3d {

//! @brief マテリアルインスタンスです。
class MaterialObj
{
public:
    enum
    {
        MAX_TEXTURE = 16
    };

    class Builder;
    class InitArg;

    //! @brief インスタンスの構築時に渡すバッファの必要アライメントサイズです。
    enum Alignment
    {
        //! @brief Init() で渡すバッファの必要アライメントサイズです。
        BUFFER_ALIGNMENT        = 4,

        //! @brief SetupBlockBuffer() で渡すバッファの必要アライメントサイズです。
        BLOCK_BUFFER_ALIGNMENT  = GX2_UNIFORM_BLOCK_ALIGNMENT
    };

    //! @briefprivate テクスチャ差し替え時に呼ばれるコールバックの型
    typedef void (*TextureChangeCallback)(MaterialObj* pMaterialObj, int idxTexture);

    //----------------------------------------
    //! @name 構築/破棄
    //@{

    //! @brief コンストラクタです。
    //!
    //! 実際の構築処理は Init() で行います。
    //!
    MaterialObj() : m_pRes(NULL), m_pUserPtr(NULL), m_pBufferPtr(NULL), m_pBlockBuffer(NULL), m_pTextureChangeCallback(NULL) {}

    //! @brief 計算に必要なバッファサイズを計算します。
    static size_t CalcBufferSize(const InitArg& arg);

    //! @brief インスタンスの初期化を行います。
    bool Init(const InitArg& arg, void* pBuffer, size_t bufferSize);

    //! @brief UniformBlock のサイズを計算します。
    size_t CalcBlockBufferSize() const;

    //! @brief CalcBlockBufferSize() の別名関数です。
    size_t CalculateBlockBufferSize() const
    {
        return CalcBlockBufferSize();
    }

    //! @brief UniformBlock を構築します。
    bool SetupBlockBuffer(void* pBuffer, size_t bufferSize);

    //! @brief UniformBlock を破棄します。
    void CleanupBlockBuffer();

    //@}

    //----------------------------------------
    //! @name 更新
    //@{

    //! @brief マテリアルに関する描画リソースを計算します。
    //!
    //! GPU から参照されるバッファを書き換えるため、
    //! 前回の描画で GPU が参照し終わった後に呼ぶ必要があります。
    //! 構築時のオプションでダブルバッファ化するか、
    //! フレームバッファのコピーアウト中などに呼ぶことを想定しています。
    //!
    void CalcMatBlock(int bufferIndex = 0);

    //! @brief CalcMatBlock() の別名関数です。
    void CalculateMaterial(int bufferIndex = 0)
    {
        CalcMatBlock(bufferIndex);
    }

    //@}

    //----------------------------------------
    //! @name 取得/設定
    //@{

    //! @brief リソースを取得します。
    ResMaterial* GetResource() { return m_pRes; }

    //! @brief リソースを取得します。
    const ResMaterial* GetResource() const { return m_pRes; }

    //! @brief Init() 時に渡されたバッファを取得します。
    void* GetBufferPtr() { return m_pBufferPtr; }

    //! @brief SetupBlockBuffer() 時に渡されたバッファを取得します。
    void* GetBlockBufferPtr() { return m_pBlockBuffer; }

    //! @brief UniformBlock が構築済みであるかどうかを取得します。
    bool IsBlockBufferValid() const { return (m_Flag & BLOCK_BUFFER_VALID) != 0; }

    //! @brief マテリアルが保持する UniformBlock を取得します。
    GfxBuffer& GetMatBlock() { return m_MatBlock; }

    //! @brief マテリアルが保持する UniformBlock を取得します。
    const GfxBuffer& GetMatBlock() const { return m_MatBlock; }

    //! @brief UniformBlock をバッファリングする数を設定します。
    int GetBufferingCount() const { return m_NumBuffering; }

    //! @brief UniformBlock のエンディアンスワップを行うように設定します。
    void EnableBlockSwap() { m_Flag |= BLOCK_BUFFER_SWAP; }

    //! @brief UniformBlock のエンディアンスワップを行わないように設定します。
    void DisableBlockSwap() { m_Flag &= ~BLOCK_BUFFER_SWAP; }

    //! @brief UniformBlock のエンディアンスワップを行うかどうかを取得します。
    bool IsBlockSwapEnabled() const { return (m_Flag & BLOCK_BUFFER_SWAP) != 0; }

    //! @brief ユーザポインタを設定します。
    void SetUserPtr(void* pUserPtr) { m_pUserPtr = pUserPtr; }

    //! @brief ユーザポインタを取得します。
    void* GetUserPtr() { return m_pUserPtr; }

    //! @brief ユーザポインタを取得します。
    const void* GetUserPtr() const { return m_pUserPtr; }

    //! @brief ユーザポインタを取得します。
    template <typename T>
    T* GetUserPtr() { return static_cast<T*>(m_pUserPtr); }

    //! @brief ユーザポインタを取得します。
    template <typename T>
    const T* GetUserPtr() const { return static_cast<const T*>(m_pUserPtr); }

    //@}

    //----------------------------------------
    //! @name シェーダパラメータ
    //@{

    //! @brief マテリアル配下のシェーダパラメータの数を取得します。
    int GetShaderParamCount() const { return m_pRes->GetShaderParamCount(); }

    //! @brief インデクスからシェーダパラメータの名前を取得します。
    //!
    //! インデクスはマテリアルが持つシェーダパラメータ数の範囲内とする必要があります。
    //!
    const char* GetShaderParamName(int paramIndex) const
    {
        return m_pRes->GetShaderParamName(paramIndex);
    }

    //! @brief シェーダパラメータの名前からインデクスを取得します。
    //!
    //! 指定した名前のシェーダパラメータが存在しない場合は -1 を返します。
    //!
    int GetShaderParamIndex(const char* name) const { return m_pRes->GetShaderParamIndex(name); }

    //! @brief GetShaderParamIndexの別名関数です。
    int FindShaderParamIndex(const char* name) const { return m_pRes->FindShaderParamIndex(name); }

    //! @brief インデクス引きでシェーダパラメータを取得します。
    //!
    //! インデクスはマテリアルが持つシェーダパラメータ数の範囲内とする必要があります。
    //!
    ResShaderParam* GetResShaderParam(int paramIndex)
    {
        return m_pRes->GetShaderParam(paramIndex);
    }

    //! @brief インデクス引きでシェーダパラメータを取得します。
    //!
    //! インデクスはマテリアルが持つシェーダパラメータ数の範囲内とする必要があります。
    //!
    const ResShaderParam* GetResShaderParam(int paramIndex) const
    {
        return m_pRes->GetShaderParam(paramIndex);
    }

    //! @brief 名前引きでシェーダパラメータを取得します。
    //!
    //! 指定した名前のシェーダパラメータが存在しない場合は NULL を返します。
    //!
    ResShaderParam* GetResShaderParam(const char* name)
    {
        return m_pRes->GetShaderParam(name);
    }

    //! @brief 名前引きでシェーダパラメータを取得します。
    //!
    //! 指定した名前のシェーダパラメータが存在しない場合は NULL を返します。
    //!
    const ResShaderParam* GetResShaderParam(const char* name) const
    {
        return m_pRes->GetShaderParam(name);
    }

    //! @brief シェーダパラメータを編集するために取得します。
    //!
    //! この関数で取得したパラメータは DirtyFlag で設定され、
    //! CalcMatBlock() を呼び出すことで UniformBlock に反映されます。
    //!
    //! インデクスはマテリアルが持つシェーダパラメータ数の範囲内とする必要があります。
    //!
    void* EditShaderParam(int paramIndex)
    {
        ResShaderParam* pParam = GetResShaderParam(paramIndex);
        if (pParam->GetOffset() >= 0)
        {
            SetDirtyFlag(paramIndex); // 書き込み先が存在する場合のみフラグを立てる。
        }
        int dependedIndex = pParam->GetDependedIndex();
        ResShaderParam *pDependedParam = GetResShaderParam(dependedIndex);
        if (pDependedParam->GetOffset() >= 0)
        {
            SetDirtyFlag(dependedIndex); // 書き込み先が存在する場合のみフラグを立てる。
        }
        return AddOffset(m_pSrcParam, pParam->GetSrcOffset());
    }

    //! @brief シェーダパラメータを取得します。
    //!
    //! 取得したメモリを書き換える場合は EditShaderParam() を呼び出してください。
    //!
    //! インデクスはマテリアルが持つシェーダパラメータ数の範囲内とする必要があります。
    //!
    const void* GetShaderParam(int paramIndex) const
    {
        return AddOffset(m_pSrcParam, GetResShaderParam(paramIndex)->GetSrcOffset());
    }

    //! @brief シェーダパラメータを編集するために取得します。
    //!
    //! この関数で取得したパラメータは DirtyFlag で設定され、
    //! CalcMatBlock() を呼び出すことで UniformBlock に反映されます。
    //! テンプレート引数を指定することで型チェックを行います。
    //! 真偽値は4バイトで格納されているため、BOOL、int、uint、s32、u32 などの
    //! 4バイトの型で取得する必要があります。
    //!
    //! インデクスはマテリアルが持つシェーダパラメータ数の範囲内とする必要があります。
    //!
    template <typename T>
    T* EditShaderParam(int paramIndex)
    {
        ResShaderParam* pParam = GetResShaderParam(paramIndex);
        NW_G3D_ASSERTMSG(sizeof(T) <= pParam->GetSrcSize(), "Oversized T. %u <= %u\n%s\n",
            sizeof(T), pParam->GetSrcSize(),
            NW_G3D_RES_GET_NAME(m_pRes, GetName()));
        if (pParam->GetOffset() >= 0)
        {
            SetDirtyFlag(paramIndex); // 書き込み先が存在する場合のみフラグを立てる。
        }
        int dependedIndex = pParam->GetDependedIndex();
        ResShaderParam *pDependedParam = GetResShaderParam(dependedIndex);
        if (pDependedParam->GetOffset() >= 0)
        {
            SetDirtyFlag(dependedIndex); // 書き込み先が存在する場合のみフラグを立てる。
        }
        return AddOffset<T>(m_pSrcParam, pParam->GetSrcOffset());
    }

    //! @brief シェーダパラメータを取得します。
    //!
    //! 取得したメモリを書き換える場合は EditShaderParam() を呼び出してください。
    //! テンプレート引数を指定することで型チェックを行います。
    //! 真偽値は4バイトで格納されているため、BOOL、int、uint、s32、u32 などの
    //! 4バイトの型で取得する必要があります。
    //!
    //! インデクスはマテリアルが持つシェーダパラメータ数の範囲内とする必要があります。
    //!
    template <typename T>
    const T* GetShaderParam(int paramIndex) const
    {
        const ResShaderParam* pParam = GetResShaderParam(paramIndex);
        NW_G3D_ASSERTMSG(sizeof(T) <= pParam->GetSrcSize(), "Oversized T. %u <= %u\n%s\n",
            sizeof(T), pParam->GetSrcSize(),
            NW_G3D_RES_GET_NAME(m_pRes, GetName()));
        return AddOffset<T>(m_pSrcParam, pParam->GetSrcOffset());
    }

    //! @brief シェーダパラメータをリソースの状態に戻します。
    void ClearShaderParam()
    {
        ResetDirtyFlags();
        memcpy(m_pSrcParam, m_pRes->ref().ofsSrcParam.to_ptr(), m_pRes->GetSrcParamSize());
        InitDependPointer();
    }

    //! @brief UniformBlock に書き込まれる前の論理的なシェーダパラメータのバッファを取得します。
    void* GetSrcParam() { return m_pSrcParam; }

    //! @brief UniformBlock に書き込まれる前の論理的なシェーダパラメータのバッファを取得します。
    const void* GetSrcParam() const { return m_pSrcParam; }

    //! @brief すべてのシェーダパラメータの DirtyFlag を Dirty 状態に戻します。
    void ResetDirtyFlags();

    //! @brief CalcMatBlock() による更新の必要性有無を取得します。
    bool IsMatBlockDirty() const
    {
        return (m_Flag & PARAM_DIRTY) || (m_pRes->ref().numShaderParamVolatile > 0);
    }

    //! @brief IsMatBlockDirty() の別名関数です。
    bool IsMaterialBlockDirty() const
    {
        return IsMatBlockDirty();
    }

    //@}

    //----------------------------------------
    //! @name テクスチャ
    //@{

    //! @brief マテリアル配下のテクスチャの数を取得します。
    int GetTextureCount() const { return m_pRes->GetTextureCount(); }

    //! @brief テクスチャを取得します。
    //!
    //! インデクスはマテリアルが持つテクスチャ数の範囲内とする必要があります。
    //!
    ResTexture* GetResTexture(int texIndex)
    {
        NW_G3D_ASSERT_INDEX_BOUNDS_DETAIL(texIndex, GetTextureCount(), "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));
        return m_ppTextureArray[texIndex];
    }

    //! @brief テクスチャを取得します。
    //!
    //! インデクスはマテリアルが持つテクスチャ数の範囲内とする必要があります。
    //!
    const ResTexture* GetResTexture(int texIndex) const
    {
        NW_G3D_ASSERT_INDEX_BOUNDS_DETAIL(texIndex, GetTextureCount(), "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));
        return m_ppTextureArray[texIndex];
    }

    //! @brief テクスチャを設定します。
    //!
    //! マテリアルが参照するテクスチャを差し替えることができます。
    //!
    //! インデクスはマテリアルが持つテクスチャ数の範囲内とする必要があります。
    //!
    void SetResTexture(int texIndex, ResTexture* texture)
    {
        NW_G3D_ASSERT_INDEX_BOUNDS_DETAIL(texIndex, GetTextureCount(), "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));
        ResTexture* oldTexture = m_ppTextureArray[texIndex];
        m_ppTextureArray[texIndex] = texture;
        if (m_pTextureChangeCallback && oldTexture != texture)
        {
            m_pTextureChangeCallback(this, texIndex);
        }
    }

    //! @brief テクスチャをリソースの状態に戻します。
    void ClearTexture()
    {
        for (int idxTex = 0, numTex = GetTextureCount(); idxTex < numTex; ++idxTex)
        {
            ResTexture* texture = m_pRes->GetTextureRef(idxTex)->Get();
            SetResTexture(idxTex, texture);
        }
    }

    //! @briefprivate テクスチャ差し替え時に呼ばれるコールバックを設定します。
    void SetTextureChangeCallback(TextureChangeCallback pCallback)
    {
        m_pTextureChangeCallback = pCallback;
    }

    //! @briefprivate テクスチャ差し替え時に呼ばれるコールバックを取得します。
    TextureChangeCallback GetTextureChangeCallback() const
    {
        return m_pTextureChangeCallback;
    }

    //@}

    //----------------------------------------
    //! @name 描画ステート
    //@{

    //! @brief 描画ステートを取得します。
    ResRenderState* GetResRenderState() { return m_pRes->GetRenderState(); }

    //! @brief 描画ステートを取得します。
    const ResRenderState* GetResRenderState() const { return m_pRes->GetRenderState(); }

    //@}

    //----------------------------------------
    //! @name サンプラー
    //@{

    //! @brief マテリアル配下のサンプラーの数を取得します。
    int GetSamplerCount() const { return m_pRes->GetSamplerCount(); }

    //! @brief インデクスからサンプラーの名前を取得します。
    //!
    //! インデクスはマテリアルが持つサンプラー数の範囲内とする必要があります。
    //!
    const char* GetSamplerName(int samplerIndex) const
    {
        return m_pRes->GetSamplerName(samplerIndex);
    }

    //! @brief サンプラーの名前からインデクスを取得します。
    //!
    //! 指定した名前のサンプラーが存在しない場合は -1 を返します。
    //!
    int GetSamplerIndex(const char* name) const { return m_pRes->GetSamplerIndex(name); }

    //! @brief GetSamplerIndexの別名関数です。
    int FindSamplerIndex(const char* name) const { return m_pRes->FindSamplerIndex(name); }

    //! @brief インデクス引きでサンプラーを取得します。
    //!
    //! インデクスはマテリアルが持つサンプラー数の範囲内とする必要があります。
    //!
    ResSampler* GetResSampler(int samplerIndex)
    {
        return m_pRes->GetSampler(samplerIndex);
    }

    //! @brief インデクス引きでサンプラーを取得します。
    //!
    //! インデクスはマテリアルが持つサンプラー数の範囲内とする必要があります。
    //!
    const ResSampler* GetResSampler(int samplerIndex) const
    {
        return m_pRes->GetSampler(samplerIndex);
    }

    //! @brief 名前引きでサンプラーを取得します。
    //!
    //! 指定した名前のサンプラーが存在しない場合は NULL を返します。
    //!
    ResSampler* GetResSampler(const char* name) { return m_pRes->GetSampler(name); }

    //! @brief 名前引きでサンプラーを取得します。
    //!
    //! 指定した名前のサンプラーが存在しない場合は NULL を返します。
    //!
    const ResSampler* GetResSampler(const char* name) const { return m_pRes->GetSampler(name); }

    //@}

protected:
    //! @brief マテリアルの状態を表すフラグです。
    enum Flag
    {
        //! @briefprivate UniformBlock が構築済みであるかどうかを表すフラグです。
        //!
        BLOCK_BUFFER_VALID  = 0x1 << 0,

        //! @briefprivate UniformBlock のエンディアンスワップを行うかどうかを表すフラグです。
        //!
        BLOCK_BUFFER_SWAP   = 0x1 << 1,

        //! @brief シェーダパラメータが更新されたことを表すフラグです。
        PARAM_DIRTY         = 0x1 << 2
    };

    //! @brief シェーダパラメータに DirtyFlag を設定します。
    //!
    //! DirtyFlag が設定されたシェーダパラメータは、
    //! CalcMatBlock() を呼び出すことで UniformBlock に反映されます。
    //!
    void SetDirtyFlag(int paramIndex)
    {
        NW_G3D_ASSERT_INDEX_BOUNDS_DETAIL(paramIndex, GetShaderParamCount(), "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));
        m_Flag |= PARAM_DIRTY;
        m_pDirtyFlagArray[paramIndex >> 5] |= 0x1 << (paramIndex & 0x1F);
    }

    //! @brief SrcParam を変換して UniformBlock に反映します。
    //!
    //! DirtyFlag を考慮して UniformBlock に反映する場合は ConvertDirtyParams を呼び出してください。
    //!
    template <bool swap>
    void ConvertParams(void* pBuffer);

    //! @brief SrcParam を変換して UniformBlock に反映します。
    //!
    //! DirtyFlag を考慮して変換処理を行います。
    //! CalcMatBlock() が内部で呼び出す関数です。
    //!
    template <bool swap>
    void ConvertDirtyParams(void* pBuffer, bit32* pDirtyFlagArray);

    //! @brief SrcParam を変換して UniformBlock に反映します。
    //!
    //! CalcMatBlock() が内部で呼び出す関数です。
    //!
    template <bool swap>
    void ConvertVolatileParams(void* pBuffer);

    //! @brief 依存パラメータのポインタを設定します。
    //!
    //! Init() と ClearShaderParam() が内部で呼び出す関数です。
    //!
    void InitDependPointer();

private:
    class Sizer;
    ResMaterial* m_pRes;

    bit16 m_Flag;
    bit8 m_DirtyFlags;
    u8 m_NumBuffering;
    u16 m_NumFlag32;
    u16 reserved;
    bit32* m_pDirtyFlagArray;
    bit32* m_pBufferFlagArray;
    GfxBuffer m_MatBlock;
    void* m_pSrcParam; // m_MatBlock の元データ
    ResTexture** m_ppTextureArray;

    void* m_pUserPtr;
    void* m_pBufferPtr;
    void* m_pBlockBuffer;

    TextureChangeCallback m_pTextureChangeCallback;
    NW_G3D_DISALLOW_COPY_AND_ASSIGN(MaterialObj);
};

// 宣言のみで定義は行わない。
// bool は4バイトで格納されているため、BOOL、int、uint、s32、u32 等で取得しなければならない。
template <>
bool* MaterialObj::EditShaderParam<bool>(int);
template <>
const bool* MaterialObj::GetShaderParam<bool>(int) const;

//--------------------------------------------------------------------------------------------------

//! @briefprivate MaterialObj のサイズを計算するためのクラスです。
//!
class MaterialObj::Sizer : public nw::g3d::Sizer
{
public:
    //! @brief コンストラクタです。
    Sizer() : nw::g3d::Sizer() {}

    //! @brief MaterialObj::InitArg に基づいてサイズを計算します。
    void Calc(const InitArg& arg);

    enum
    {
        DIRTY_FLAG_ARRAY,
        BUFFER_FLAG_ARRAY,
        SRC_PARAM_ARRAY,
        TEXTURE_ARRAY,
        NUM_CHUNK
    };

    Chunk chunk[NUM_CHUNK];
};

//--------------------------------------------------------------------------------------------------

//! @brief MaterialObj::Init() に渡して初期化を行うパラメータです。
class MaterialObj::InitArg
{
public:
    //! @brief コンストラクタです。
    explicit InitArg(ResMaterial* resource)
        : m_pRes(resource)
        , m_NumBuffering(1)
    {
        NW_G3D_ASSERT_NOT_NULL(resource);
    }

    //! @brief リソースを取得します。
    ResMaterial* GetResource() const { return m_pRes; }

    //! @brief UniformBlock をバッファリングする数を設定します。
    void BufferingCount(int count) { m_NumBuffering = count; m_Sizer.Invalidate(); }

    //! @brief UniformBlock をバッファリングする数を取得します。
    int GetBufferingCount() const { return m_NumBuffering; }

    //! @brief サイズ計算用のオブジェクトを取得します。
    Sizer& GetSizer() const { return m_Sizer; }

private:
    ResMaterial* m_pRes;
    int m_NumBuffering;
    mutable Sizer m_Sizer; // キャッシュするために mutable にする
};

//--------------------------------------------------------------------------------------------------

//! @brief MaterialObj の構築を行うクラスです。
class MaterialObj::Builder : public MaterialObj::InitArg
{
public:
    //! @brief コンストラクタです。
    Builder(ResMaterial* resource) : InitArg(resource), m_Size(0), m_IsCalculated(false)
    {
    }

    //! @brief MaterialObj を構築します。
    //!
    //! @param[in] pMaterialObj MaterialObj へのポインタ
    //! @param[in] pBuffer バッファへのポインタ
    //! @param[in] bufferSize バッファのサイズ
    //!
    //! @pre
    //! - CalculateMemorySize() を呼び、メモリサイズが計算済みである
    //! - bufferSize >= GetWorkMemorySize() で返すサイズ
    //!
    bool Build(MaterialObj* pMaterialObj, void* pBuffer, size_t bufferSize) const
    {
        return pMaterialObj->Init(*this, pBuffer, bufferSize);
    }

    //! @brief MaterialObj 構築に必要なメモリサイズを計算します。
    void CalculateMemorySize()
    {
        m_Size = CalcBufferSize(*this);
        m_IsCalculated = true;
    }

    //! @brief MaterialObj 構築に必要なメモリサイズを取得します。
    //!
    //! @return  MaterialObj 構築に必要なメモリサイズを返します。
    //!
    size_t GetWorkMemorySize() const
    {
        return m_Size;
    }

    //! @brief MaterialObj 構築に必要なメモリサイズが計算済みかを取得します。
    //!
    //! @return 計算済みの場合は true、未計算の場合は false を返します。
    //!
    bool IsMemoryCalculated() const
    {
        return m_IsCalculated;
    }

private:
    size_t m_Size;
    bool m_IsCalculated;
};

}} // namespace nw::g3d

#endif // NW_G3D_MATERIALOBJ_H_
