﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_G3D_BILLBOARD_H_
#define NW_G3D_BILLBOARD_H_

#include <nw/g3d/g3d_config.h>
#include <nw/g3d/math/g3d_Matrix34.h>
#include <nw/g3d/g3d_SkeletonObj.h>

namespace nw { namespace g3d {

//! @brief ビルボード計算を行うクラスです。
class Billboard
{
public:
    // 各種ビルボード計算
    // スケールは保存されないので抽出して別途適用する必要がある。
    // Viewpoint 系のビルボードは pWorldView から平行移動成分を取得して使用する。

    //! @brief World 方式でビルボード計算を行います。
    //!
    //! Z 軸が視点軸と平行となるように回転させます。
    //!
    static void CalcWorld(Mtx34* pWorldView, const Mtx34& view, const Mtx34& world);

    //! @brief CalcWorld() の別名関数です。
    static void CalculateWorld(Mtx34* pWorldView, const Mtx34& view, const Mtx34& world)
    {
        CalcWorld(pWorldView, view, world);
    }

    //! @brief WorldViewpoint 方式でビルボード計算を行います。
    //!
    //! Z 軸が視点方向に向くように回転させます。
    //!
    static void CalcWorldViewpoint(Mtx34* pWorldView, const Mtx34& view, const Mtx34& world);

    //! @brief CalcWorldViewpoint() の別名関数です。
    static void CalculateWorldViewpoint(Mtx34* pWorldView, const Mtx34& view, const Mtx34& world)
    {
        CalcWorldViewpoint(pWorldView, view, world);
    }

    //! @brief Screen 方式でビルボード計算を行います。
    //!
    //! Z 軸が視点軸と平行で、Y 軸が画面上方向になるように回転させます。
    //!
    static void CalcScreen(Mtx34* pWorldView, const Mtx34& local);

    //! @brief CalcScreen() の別名関数です。
    static void CalculateScreen(Mtx34* pWorldView, const Mtx34& local)
    {
        CalcScreen(pWorldView, local);
    }

    //! @brief ScreenViewpoint 方式でビルボード計算を行います。
    //!
    //! Z 軸が視点方向に向き、Y 軸が画面上方向になるように回転させます。
    //!
    static void CalcScreenViewpoint(Mtx34* pWorldView, const Mtx34& local);

    //! @brief CalcScreenViewpoint() の別名関数です。
    static void CalculateScreenViewpoint(Mtx34* pWorldView, const Mtx34& local)
    {
        CalcScreenViewpoint(pWorldView, local);
    }

    //! @brief YAxis 方式でビルボード計算を行います。
    //!
    //! Z 軸が視点軸と平行となるように Y 軸だけ回転させます。
    //!
    static void CalcYAxis(Mtx34* pWorldView, const Mtx34& view, const Mtx34& world);

    //! @brief CalcScreenViewpoint() の別名関数です。
    static void CalculateYAxis(Mtx34* pWorldView, const Mtx34& view, const Mtx34& world)
    {
        CalcYAxis(pWorldView, view, world);
    }

    //! @brief YAxisViewpoint 方式でビルボード計算を行います。
    //!
    //! Z 軸が視点方向に向くように Y 軸だけ回転させます。
    //!
    static void CalcYAxisViewpoint(Mtx34* pWorldView, const Mtx34& view, const Mtx34& world);

    //! @brief CalcYAxisViewpoint() の別名関数です。
    static void CalculateYAxisViewpoint(Mtx34* pWorldView, const Mtx34& view, const Mtx34& world)
    {
        CalcYAxisViewpoint(pWorldView, view, world);
    }

    //! @brief ビルボード全種類を同じ引数で呼ぶためのラッパーです。
    static void CalcWorld(Mtx34* pWorldView,
        const Mtx34& view, const Mtx34& world, const Mtx34& /*local*/)
    {
        CalcWorld(pWorldView, view, world);
    }

    //! @brief CalcWorld() の別名関数です。
    static void CalculateWorld(Mtx34* pWorldView,
        const Mtx34& view, const Mtx34& world, const Mtx34& /*local*/)
    {
        CalcWorld(pWorldView, view, world);
    }

    //! @brief ビルボード全種類を同じ引数で呼ぶためのラッパーです。
    static void CalcWorldViewpoint(Mtx34* pWorldView,
        const Mtx34& view, const Mtx34& world, const Mtx34& /*local*/)
    {
        CalcWorldViewpoint(pWorldView, view, world);
    }

    //! @brief CalcWorldViewpoint() の別名関数です。
    static void CalculateWorldViewpoint(Mtx34* pWorldView,
        const Mtx34& view, const Mtx34& world, const Mtx34& /*local*/)
    {
        CalcWorldViewpoint(pWorldView, view, world);
    }

    //! @brief ビルボード全種類を同じ引数で呼ぶためのラッパーです。
    static void CalcScreen(Mtx34* pWorldView,
        const Mtx34& /*view*/, const Mtx34& /*world*/, const Mtx34& local)
    {
        CalcScreen(pWorldView, local);
    }

    //! @brief CalcScreen() の別名関数です。
    static void CalculateScreen(Mtx34* pWorldView,
        const Mtx34& /*view*/, const Mtx34& /*world*/, const Mtx34& local)
    {
        CalcScreen(pWorldView, local);
    }

    //! @brief ビルボード全種類を同じ引数で呼ぶためのラッパーです。
    static void CalcScreenViewpoint(Mtx34* pWorldView,
        const Mtx34& /*view*/, const Mtx34& /*world*/, const Mtx34& local)
    {
        CalcScreenViewpoint(pWorldView, local);
    }

    //! @brief CalculateScreenViewpoint() の別名関数です。
    static void CalculateScreenViewpoint(Mtx34* pWorldView,
        const Mtx34& /*view*/, const Mtx34& /*world*/, const Mtx34& local)
    {
        CalcScreenViewpoint(pWorldView, local);
    }

    //! @brief ビルボード全種類を同じ引数で呼ぶためのラッパーです。
    static void CalcYAxis(Mtx34* pWorldView,
        const Mtx34& view, const Mtx34& world, const Mtx34& /*local*/)
    {
        CalcYAxis(pWorldView, view, world);
    }

    //! @brief CalcYAxis() の別名関数です。
    static void CalculateYAxis(Mtx34* pWorldView,
        const Mtx34& view, const Mtx34& world, const Mtx34& /*local*/)
    {
        CalcYAxis(pWorldView, view, world);
    }

    //! @brief ビルボード全種類を同じ引数で呼ぶためのラッパーです。
    static void CalcYAxisViewpoint(Mtx34* pWorldView,
        const Mtx34& view, const Mtx34& world, const Mtx34& /*local*/)
    {
        CalcYAxisViewpoint(pWorldView, view, world);
    }

    //! @brief CalcYAxisViewpoint() の別名関数です。
    static void CalculateYAxisViewpoint(Mtx34* pWorldView,
        const Mtx34& view, const Mtx34& world, const Mtx34& /*local*/)
    {
        CalcYAxisViewpoint(pWorldView, view, world);
    }

    //! @brief billboardMode に応じてビルボード計算を行います。
    //!
    //! billboardMode は res::ResBone::Flag で指定します。
    //!
    static void Calc(bit32 billboardMode, Mtx34* pWorldView,
        const Mtx34& view, const Mtx34& world, const Mtx34& local);

    //! @brief Calc() の別名関数です。
    static void Calculate(bit32 billboardMode, Mtx34* pWorldView,
        const Mtx34& view, const Mtx34& world, const Mtx34& local)
    {
        Calc(billboardMode, pWorldView, view, world, local);
    }

protected:
    //! @briefprivate Right 方向の設定を行う内部関数です。
    //!
    static void SetR(Mtx34* pWorldView, const Vec3& vx, const Vec3& vy, const Vec3& vz);

    //! @briefprivate Right 方向の設定を行う内部関数です。
    //!
    static void SetR(Mtx34* pWorldView, const Vec3& vy);
};

}} // namespace nw::g3d

#endif // NW_G3D_BILLBOARD_H_
