﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nw/g3d/edit/g3d_EditDefs.h>

#if NW_G3D_CONFIG_USE_HOSTIO

#if NW_G3D_IS_HOST_CAFE
#include <cafe/fs.h>
#endif

namespace nw { namespace g3d {

class ModelObj;

namespace res {

class ResShaderArchive;

} // namespace res

}} // namespace nw::g3d

namespace nw { namespace g3d { namespace edit {

class IAllocator;
class EditCallback;

//! @briefprivate nn::g3d::viewer 互換用です。
enum ViewerResult {
    ViewerResult_Success,
    ViewerResult_ViewerDisabled,
    ViewerResult_UnknownError,
};

//! @brief editの処理を行うサーバークラスです。
class EditServer
{
public:
    //! @brief アタッチモデル時に渡す構造体です。
    struct AttachModelArg
    {
        nw::g3d::ModelObj*  modelObj;//!< アタッチするModelObjです。
        const char* attachFileName;//!< ModelObjに対応するアタッチ対象のファイル名です。

        AttachModelArg()
            : modelObj(NULL)
            , attachFileName(NULL)
        {}

        explicit AttachModelArg(const char* path)
            : modelObj(NULL)
            , attachFileName(path)
        {}
    };

    //! @brief デタッチモデル時に渡す構造体です。
    struct DetachModelArg
    {
        nw::g3d::ModelObj*  modelObj;//!< デタッチするModelObjです。

        explicit DetachModelArg()
            : modelObj(NULL)
        {}
    };

    //! @brief アタッチシェーダアーカイブ時に渡す構造体です。
    struct AttachShaderArchiveArg
    {
        nw::g3d::res::ResShaderArchive* resShaderArchive;//!< アタッチするResShaderArchiveです。
        const char* attachFileName;//!< ResShaderArchiveに対応するアタッチ対象のファイル名です。
        AttachShaderArchiveArg()
            : resShaderArchive(NULL)
            , attachFileName(NULL)
        {}

        explicit AttachShaderArchiveArg(const char* path)
            : resShaderArchive(NULL)
            , attachFileName(path)
        {}
    };

    //! @brief デタッチシェーダアーカイブ時に渡す構造体です。
    struct DetachShaderArchiveArg
    {
        nw::g3d::res::ResShaderArchive* resShaderArchive;//!< デタッチするResShaderArchiveです。
    };

    /**
        @briefprivate モデル配置情報の登録時に渡す構造体です。
     */
    struct ModelLayoutArg
    {
        const ModelObj* modelObj;   //!< モデル配置対象の ModelObj です。
        f32 scale[3];               //!< モデル配置対象に設定する Scale です。
        f32 rotate[3];              //!< モデル配置対象に設定する Rotate です。
        f32 translate[3];           //!< モデル配置対象に設定する Translate です。

        ModelLayoutArg()
            : modelObj(NULL)
        {
            scale[0] = scale[1] = scale[2] = 1.0f;
            rotate[0] = rotate[1] = rotate[2] = 0.0f;
            translate[0] = translate[1] = translate[2] = 0.0f;
        }

        //! @briefprivate nn::g3d::viewer 互換用です。
        explicit ModelLayoutArg(const nw::g3d::ModelObj* pModelObj)
            : modelObj(pModelObj)
        {
        }

        //! @briefprivate nn::g3d::viewer 互換用です。
        void SetScale(f32 x, f32 y, f32 z)
        {
            scale[0] = x;
            scale[1] = y;
            scale[2] = z;
        }

        //! @briefprivate nn::g3d::viewer 互換用です。
        void SetRotate(f32 x, f32 y, f32 z)
        {
            rotate[0] = x;
            rotate[1] = y;
            rotate[2] = z;
        }

        //! @briefprivate nn::g3d::viewer 互換用です。
        void SetTranslate(f32 x, f32 y, f32 z)
        {
            translate[0] = x;
            translate[1] = y;
            translate[2] = z;
        }
    };

    //! @briefprivate nn::g3d::viewer 互換用です。
    typedef ModelLayoutArg LayoutModelArg;

    //! @brief サーバー生成時に渡す構造体です。
    struct CreateArg
    {
        IAllocator*             allocator;      //!< EditServerで使用するアロケータです。
        EditCallback*           editCallback;   //!< EditServerで使用するコールバックです。
        u16                     codePage;       //!< Editで使用するコードページです。

#if NW_G3D_IS_HOST_CAFE
        FSClient*               hostFileIOHandle; //!< HostIOをアプリケーションで使用している場合のみ設定します。
#endif

        CreateArg()
        : allocator(NULL)
        , editCallback(NULL)
        , codePage(0)
#if NW_G3D_IS_HOST_CAFE
        , hostFileIOHandle(NULL)
#endif
        {}

        //! @briefprivate nn::g3d::viewer 互換用です。
        void SetCallback(EditCallback* pCallbacks)
        {
            editCallback = pCallbacks;
        }

        //! @briefprivate nn::g3d::viewer 互換用です。
        void SetCodePage(int codePageValue)
        {
            codePage = static_cast<u16>(codePageValue);
        }
    };

    //! @briefprivate nn::g3d::viewer 互換用です。
    typedef CreateArg InitializeArg;

public:
    //!  @brief EditServerクラスのインスタンスを生成します。
    static bool CreateInstance(const CreateArg& arg);

    //! @briefprivate nn::g3d::viewer 互換用です。
    static ViewerResult Initialize(const InitializeArg& arg)
    {
        bool success = CreateInstance(arg);
        if (!success)
        {
            return ViewerResult_UnknownError;
        }

        return ViewerResult_Success;
    }

    //! @brief EditServerクラスのインスタンスを取得します。
    static EditServer& Instance()
    {
        return s_Instance;
    }

    //! @briefprivate nn::g3d::viewer 互換用です。
    static EditServer& GetInstance()
    {
        return Instance();
    }

    //! @brief EditServerクラスのインスタンスを破棄します。
    static void DeleteInstance();

    //! @briefprivate nn::g3d::viewer 互換用です。
    static void Finalize()
    {
        DeleteInstance();
    }

    //! @brief EditServerクラスが初期化済みであるかを判定します。
    static bool IsInitialized();

    //! @brief サーバーが3DEditorと接続済みか判定します。
    bool IsConnected() const;

    //! @brief 接続状況、パケットの送受信を監視します。
    void PollDataCommunication();

    //! @brief 受信したデータを元にデータ編集等の処理を行います。
    void PollDataEdit();

    //! @briefprivate nn::g3d::viewer 互換用です。
    void ExecuteCommands()
    {
        PollDataEdit();
    }

    //! @brief PollDataCommunication()、PollDataEdit() の処理を実行します。
    void Poll();

    //! @brief サーバーをオープンします。
    bool Open();

    //! @brief サーバーをクローズします。
    void Close();

    //! @brief 編集する対象モデルをアタッチします。
    bool AttachModel(const AttachModelArg& arg);

    //! @briefprivate nn::g3d::viewer 互換用です。
    ViewerResult QueueAttachModelCommand(nw::g3d::ModelObj* pAttachModelObj, const char* attachFileName)
    {
        AttachModelArg arg;
        arg.attachFileName = attachFileName;
        arg.modelObj = pAttachModelObj;
        bool success = AttachModel(arg);
        if (!success)
        {
            return ViewerResult_UnknownError;
        }

        return ViewerResult_Success;
    }

    //! @brief モデルをデタッチします。
    bool DetachModel(const DetachModelArg& arg);

    //! @briefprivate nn::g3d::viewer 互換用です。
    ViewerResult QueueDetachModelCommand(nw::g3d::ModelObj* pModelObj)
    {
        DetachModelArg arg;
        arg.modelObj = pModelObj;
        bool success = DetachModel(arg);
        if (!success)
        {
            return ViewerResult_UnknownError;
        }

        return ViewerResult_Success;
    }

    //! @brief 編集する対象シェーダアーカイブをアタッチします。
    bool AttachShaderArchive(const AttachShaderArchiveArg& arg);

    //! @briefprivate nn::g3d::viewer 互換用です。
    ViewerResult QueueAttachShaderArchiveCommand(nw::g3d::ResShaderArchive* pAttachResShaderArchive, const char* attachFileName)
    {
        AttachShaderArchiveArg arg;
        arg.attachFileName = attachFileName;
        arg.resShaderArchive = pAttachResShaderArchive;
        bool success = AttachShaderArchive(arg);
        if (!success)
        {
            return ViewerResult_UnknownError;
        }

        return ViewerResult_Success;
    }

    //! @brief シェーダアーカイブをデタッチします。
    bool DetachShaderArchive(const DetachShaderArchiveArg& arg);

    //! @briefprivate nn::g3d::viewer 互換用です。
    ViewerResult QueueDetachShaderArchiveCommand(nw::g3d::ResShaderArchive* pAttachResShaderArchive)
    {
        DetachShaderArchiveArg arg;
        arg.resShaderArchive = pAttachResShaderArchive;
        bool success = DetachShaderArchive(arg);
        if (!success)
        {
            return ViewerResult_UnknownError;
        }

        return ViewerResult_Success;
    }

    //! @brief 編集対象になっているアニメーションを計算します。
    void CalcAnimations();

    //! @briefprivate nn::g3d::viewer 互換用です。
    void CalculateAnimations()
    {
        CalcAnimations();
    }

    //! @brief 指定した modelObj が編集対象になっている場合に、バインドされているスケルタルアニメを計算します。
    void CalcSkeletalAnimations(const nw::g3d::ModelObj* modelObj);

    //! @briefprivate nn::g3d::viewer 互換用です。
    void CalculateSkeletalAnimations(const nw::g3d::ModelObj* modelObj)
    {
        CalcSkeletalAnimations(modelObj);
    }

    //! @brief 編集対象になっているModelObj か判定します。
    bool HasModelObj(const nw::g3d::ModelObj* modelObj) const;

    //! @briefprivate nn::g3d::viewer 互換用です。
    bool IsModelAttached(const nw::g3d::ModelObj* pModelObj) const
    {
        return HasModelObj(pModelObj);
    }

    //! @brief 編集対象になっているResShaderArchive か判定します。
    bool HasShaderArchive(const nw::g3d::res::ResShaderArchive* resShaderArchive) const;

    //! @briefprivate nn::g3d::viewer 互換用です。
    bool IsShaderArchiveAttached(const nw::g3d::res::ResShaderArchive* resShaderArchive) const
    {
        return HasShaderArchive(resShaderArchive);
    }

    /**
        @brief 3DEditor に送信する文字列型の描画情報の値の範囲を登録します。
        @param[in] labelName 描画情報のラベル名です。
        @param[in] itemName 描画情報のアイテム名です。(文字列型での設定値になります。）
        @param[in] aliasItemName 描画情報のアイテム名のエイリアス名です。(文字列型での設定値になります。未指定の場合は NULL です。）
     */
    bool PushRenderInfoChoice(const char* labelName, const char* itemName, const char* aliasItemName = NULL);

    //! @briefprivate nn::g3d::viewer 互換用です。
    ViewerResult QueueAddRenderInfoStringChoiceCommand(const char* labelName, const char* itemName, const char* aliasItemName = NULL)
    {
        bool success = PushRenderInfoChoice(labelName, itemName, aliasItemName);
        if (!success)
        {
            return ViewerResult_UnknownError;
        }

        return ViewerResult_Success;
    }

    /**
        @brief 3DEditor に送信するint型の描画情報の値の範囲を登録します。
        @param[in] labelName 描画情報のラベル名です。
        @param[in] minValue 描画情報の最小値です。
        @param[in] maxValue 描画情報の最大値です。
     */
    bool PushRenderInfoChoice(const char* labelName, s32 minValue, s32 maxValue);

    //! @briefprivate nn::g3d::viewer 互換用です。
    ViewerResult QueueSetRenderInfoIntRangeCommand(const char* labelName, int minValue, int maxValue)
    {
        bool success = PushRenderInfoChoice(labelName, minValue, maxValue);
        if (!success)
        {
            return ViewerResult_UnknownError;
        }

        return ViewerResult_Success;
    }

    /**
        @brief 3DEditor に送信するfloat型の描画情報の値の範囲を登録します。
        @param[in] labelName 描画情報のラベル名です。
        @param[in] minValue 描画情報の最小値です。
        @param[in] maxValue 描画情報の最大値です。
     */
    bool PushRenderInfoChoice(const char* labelName, f32 minValue, f32 maxValue);

    //! @briefprivate nn::g3d::viewer 互換用です。
    ViewerResult QueueSetRenderInfoFloatRangeCommand(const char* labelName, float minValue, float maxValue)
    {
        bool success = PushRenderInfoChoice(labelName, minValue, maxValue);
        if (!success)
        {
            return ViewerResult_UnknownError;
        }

        return ViewerResult_Success;
    }

    /**
        @brief 3DEditor に送信する文字列型の描画情報のデフォルト値を登録します。
        @param[in] labelName 描画情報のラベル名です。
        @param[in] value 描画情報のデフォルト値です。
     */
    bool PushRenderInfoDefault(const char* labelName, const char* value);

    //! @briefprivate nn::g3d::viewer 互換用です。
    ViewerResult QueuePushRenderInfoStringDefaultCommand(const char* labelName, const char* value)
    {
        bool success = PushRenderInfoDefault(labelName, value);
        if (!success)
        {
            return ViewerResult_UnknownError;
        }

        return ViewerResult_Success;
    }

    /**
        @brief 3DEditor に送信するint型の描画情報のデフォルト値を登録します。
        @param[in] labelName 描画情報のラベル名です。
        @param[in] value 描画情報のデフォルト値です。
     */
    bool PushRenderInfoDefault(const char* labelName, s32 value);

    //! @briefprivate nn::g3d::viewer 互換用です。
    ViewerResult QueuePushRenderInfoIntDefaultCommand(const char* labelName, int value)
    {
        bool success = PushRenderInfoDefault(labelName, value);
        if (!success)
        {
            return ViewerResult_UnknownError;
        }

        return ViewerResult_Success;
    }

    /**
        @brief 3DEditor に送信するfloat型の描画情報のデフォルト値を登録します。
        @param[in] labelName 描画情報のラベル名です。
        @param[in] value 描画情報のデフォルト値です。
     */
    bool PushRenderInfoDefault(const char* labelName, f32 value);

    //! @briefprivate nn::g3d::viewer 互換用です。
    ViewerResult QueuePushRenderInfoFloatDefaultCommand(const char* labelName, float value)
    {
        bool success = PushRenderInfoDefault(labelName, value);
        if (!success)
        {
            return ViewerResult_UnknownError;
        }

        return ViewerResult_Success;
    }

    /**
        @brief 3DEditor 上のマテリアルを選択状態にする情報を登録します。
        @param[in] modelObj 選択対象のモデルです。
        @param[in] materialIndex 選択対象のマテリアルインデックスです。
     */
    bool PushPickupMaterial(const ModelObj* modelObj, int materialIndex);

    //! @briefprivate nn::g3d::viewer 互換用です。
    ViewerResult QueueSelectMaterialCommand(const nw::g3d::ModelObj* pModelObj, int materialIndex)
    {
        bool success = PushPickupMaterial(pModelObj, materialIndex);
        if (!success)
        {
            return ViewerResult_UnknownError;
        }

        return ViewerResult_Success;
    }

    /**
        @brief 3DEditor 上のマテリアルを選択解除します。
     */
    bool ClearPickupMaterial();

    //! @briefprivate nn::g3d::viewer 互換用です。
    ViewerResult QueueClearMaterialSelectionCommand()
    {
        bool success = ClearPickupMaterial();
        if (!success)
        {
            return ViewerResult_UnknownError;
        }

        return ViewerResult_Success;
    }

    /**
        @briefprivate 3DEditor に送信するモデル配置情報を登録します。
     */
    bool PushModelLayout(const ModelLayoutArg& arg);

    //! @briefprivate nn::g3d::viewer 互換用です。
    ViewerResult QueueLayoutModelCommand(const LayoutModelArg& arg)
    {
        bool success = PushModelLayout(arg);
        if (!success)
        {
            return ViewerResult_UnknownError;
        }

        return ViewerResult_Success;
    }

    /**
        @brief サーバーの状態をクリアします。
        @param[in] isCallbackEnabled true を指定するとコールバックを実行します。
     */
    void Clear(bool isCallbackEnabled = false);

    /**
        @briefprivate アニメーションフラグを設定します。
        @param[in] enable true の場合は、再生、false の場合は停止状態にします。
     */
    void SetAnimFlag(bool enable);

    //! @briefprivate nn::g3d::viewer 互換用です。
    void SetAnimPlaying(bool enable)
    {
        SetAnimFlag(enable);
    }

    /**
        @briefprivate アニメーションフラグを取得します。
     */
    bool GetAnimFlag() const;

    //! @briefprivate nn::g3d::viewer 互換用です。
    bool IsAnimPlaying() const
    {
        return GetAnimFlag();
    }

    /**
        @briefprivate ループアニメーションか判定します。
     */
    bool IsLoopAnim() const;

    //! @briefprivate nn::g3d::viewer 互換用です。
    bool IsAnimPlayPolicyLoop() const
    {
        return IsLoopAnim();
    }

    /**
        @briefprivate フレームを取得します。
     */
    float GetFrame() const;

    //! @briefprivate nn::g3d::viewer 互換用です。
    float GetCurrentFrame() const
    {
        return GetFrame();
    }

    /**
        @briefprivate フレームを設定します。
     */
    void SetFrame(float value);

    //! @briefprivate nn::g3d::viewer 互換用です。
    void SetCurrentFrame(float frame)
    {
        SetFrame(frame);
    }

    /**
        @briefprivate フレームステップを取得します。
     */
    float GetFrameStep() const;

    /**
        @briefprivate フレームステップを設定します。
     */
    void SetFrameStep(float value);

    /**
        @briefprivate フレームステップにかけるプレビュー調整用の割合を設定します。
     */
    void SetPreviewFrameStepRate(float value);

    /**
        @briefprivate フレームステップにかけるプレビュー調整用の割合を取得します。
     */
    float GetPreviewFrameStepRate() const;

    /**
        @briefprivate フレーム数を取得します。
     */
    float GetFrameCount() const;

    /**
        @briefprivate フレーム数を取得します。
     */
    int GetModelAnimCount() const;

    /**
        @briefprivate 指定モデルにバインドされているアニメ数を取得します。
     */
    int GetModelAnimBoundCount(const ModelObj* modelObj) const;

    /**
        @briefprivate 指定モデルにバインドされているアニメが登録されているインデックスを取得します。
        GetModelAnimBoundCount で総数を取得して、boundAnimIndex には、総数 - 1 までの値を渡します。
        関数実行時の返り値が 0 以上であれば、GetModelAnimFrameCount、GetModelAnimName にこの値を引数に渡すことで、
        指定モデルのバインドされている各アニメの情報を取得する事ができます。
     */
    int GetActiveModelAnimIndex(const ModelObj* modelObj, int boundAnimIndex) const;

    /**
        @briefprivate フレーム数を取得します。
     */
    float GetModelAnimFrameCount(int modelAnimIndex) const;

    /**
        @briefprivate フレーム数を取得します。
     */
    const char* GetModelAnimName(int modelAnimindex) const;

    /**
        @briefprivate アニメーションの種類を取得します。
     */
    EditAnimKind GetModelAnimKind(int modelAnimIndex) const;

    /**
        @briefprivate 指定モデルの次のアニメへ設定を移動します。
     */
    void MoveNextModelAnim(const ModelObj* modelObj);

    /**
        @briefprivate 指定モデルの前のアニメへ設定を移動します。
     */
    void MovePrevModelAnim(const ModelObj* modelObj);

private:
    static EditServer s_Instance;

private:
    explicit EditServer();
    bool InitializeImpl(const CreateArg& arg);

    class Impl;
    Impl* m_pImpl;

    IAllocator* m_pAllocator;

private:
    NW_G3D_DISALLOW_COPY_AND_ASSIGN(EditServer);
};

}}} // namespace nw::g3d::edit

#endif // NW_G3D_CONFIG_USE_HOSTIO
