﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_FONT_WORDWRAPPING_H_
#define NW_FONT_WORDWRAPPING_H_

#include <wchar.h>
#include <nw/font/font_Types.h>

namespace nw {
namespace font {

//!
//! @brief ワードラッピングの設定です。
//!
//! @sa WordWrapping::CalcWordWrapping
//!
struct WordWrapConfig
{
    //! @name コンストラクタ / デストラクタ
    //@{

    //!
    //! @brief コンストラクタです。
    //!
    WordWrapConfig()
        : mMaxLineNum(0)
        , mIsIgnoreFirstSpOver2ndLine(true)
    {}

    //@}

    //! @name 設定
    //@{

    //! @brief 最大行数を設定します。
    //!
    //! @param[in] value  最大行数を指定します(初期値 = 0)。
    //!
    //! @return WordWrapConfig への参照を返します。
    //!
    //! @details
    //! 最大行数を指定した場合、 WordWrapping::CalcWordWrapping() で
    //! 行数が上限を超えた場合に処理を中断します。
    //!
    //! value に0を指定した場合は無制限を表します。
    //!
    //! @sa GetMaxLineNum
    //!
    WordWrapConfig& SetMaxLineNum(s32 value)
    {
        mMaxLineNum = value;
        return *this;
    }

    //! @brief 最大行数の設定を取得します。
    //!
    //! @return
    //! 設定値を返します。
    //!
    //! @sa SetMaxLineNum
    //!
    s32 GetMaxLineNum() const
    {
        return mMaxLineNum;
    }

    //! @brief 2行目以降の行頭の空白を無視します。
    //!
    //! @param[in] value  機能を有効にする場合は true を指定します(初期値 = true)。
    //!
    //! @return WordWrapConfig への参照を返します。
    //!
    //! @sa GetIgnoreFirstSpOver2ndLine
    //!
    WordWrapConfig& SetIgnoreFirstSpOver2ndLine(bool value)
    {
        mIsIgnoreFirstSpOver2ndLine = value;
        return *this;
    }

    //! @brief 2行目以降の行頭の空白を無視するかの設定を取得します。
    //!
    //! @return
    //! 設定値を返します。
    //!
    //! @sa SetIgnoreFirstSpOver2ndLine
    //!
    bool GetIgnoreFirstSpOver2ndLine() const
    {
        return mIsIgnoreFirstSpOver2ndLine;
    }

    //@}

    //! @cond PRIVATE

    s32 mMaxLineNum;
    bool mIsIgnoreFirstSpOver2ndLine;

    //! @endcond
};

//!
//! @brief ワードラッピングの処理中に呼び出されるコールバックです。
//!
//! @sa WordWrapping::CalcWordWrapping
//!
class WordWrapCallback
{
public:
    //!
    //! @brief 与えられた文字列のうち一行の表示に収まる範囲を求めます。
    //!
    //! @details
    //! 引数 (begin, end) が示す文字列の、一行で表示できる範囲の終端を返します。
    //! 戻り値が指す文字は一行の表示には含まれません。
    //!
    //! 本関数はワードラッピング関数( WordWrapping::CalcWordWrapping(),
    //! WordWrapping::FindLineBreak() ) から呼び出されます。
    //! 本関数により一行に表示できる範囲の上限が決定され、その範囲内で、
    //! ワードラッピング関数により適切な改行位置が決定されます。
    //!
    //! (begin, end) はユーザがワードラッピング関数に与えた文字列内を指しています。
    //! 文字列のヌル終端は保証されません。
    //!
    //! 一行に表示できる文字列の範囲を取得するには WideTextWriter::FindPosOfWidthLimit() が使用できます。
    //!
    //! @param[in] begin  現在処理中の行の先頭の文字を指します。
    //! @param[in] end  ユーザがワードラッピング関数に与えた文字列の終端です。
    //!
    //! @return
    //! 一行に収まらない最初の文字の位置を返します。
    //!
    //! @sa WideTextWriter::FindPosOfWidthLimit
    //! @sa WordWrapping::CalcWordWrapping
    //! @sa WordWrapping::FindLineBreak
    //!
    virtual const char16* GetLineBreakLimit(const char16* begin, const char16* end) = 0;
};

//!
//! @brief ワードラッピングを行うクラスです。
//!
//! @li WordWrapping::CalcWordWrapping() @n
//!     文字列の適切な位置に改行コードを挿入します。
//!
class WordWrapping
{
public:
    //!
    //! @brief 改行してよい場所を検索します。
    //!
    //! @param[in] begin  行の先頭文字を指します。
    //! @param[in] end  文字列の終端を指します。
    //! @param[in] func  一行の表示範囲を得るコールバックです。
    //! @param[in] config  ワードラップの動作を指定します。
    //!
    //! @return 改行直前の文字のアドレスを返します。
    //!
    //! @details
    //! 単体の機能として利用したい場合の関数です。
    //!
    //! func が示す一行の範囲のなかで改行して良い場所を探します。
    //!
    //! 改行して良い場所が無かった場合には func が示す範囲の最後の文字を行末とします。
    //!
    static const char16*
    FindLineBreak(
        const char16* begin,
        const char16* end,
        WordWrapCallback& func,
        const WordWrapConfig& config);

    //!
    //! @brief ワードラッピングを行います。
    //!
    //! @param[out] writeSize  out に格納された文字列の長さを返します。
    //! @param[out] out  適切な位置に改行コードの追加された文字列が格納されます。
    //! @param[in] outSize  out に格納できる文字数を指定します。終端文字を含みます。
    //! @param[in] in  処理の対象となる文字列です（要ヌル終端）。
    //! @param[in] func  一行の表示範囲を得るコールバックです。
    //! @param[in] config  ワードラップの動作を指定します。
    //!
    //! @return
    //! 処理に成功した場合は true を返します。処理を中断した場合は false を返します。
    //!
    //! @details
    //! 文字列に改行が含まれる場合、改行コードはLFである必要があります。CRやCRLFは扱えません。
    //!
    //! 改行して良い場所が無かった場合には func が示す位置で強制的に改行します。
    //!
    //! 以下の場合は処理を中断します。その際でも out はヌル終端されます。
    //! @li out のサイズが足りない場合。
    //! @li config で指定された行数の上限を超えた場合。
    //!
    //! writeSize にアドレスを指定すると、out に格納された文字列の長さが書き込まれます。
    //!
    static bool
    CalcWordWrapping(
        u32* writeSize,
        char16* out,
        u32 outSize,
        const char16* in,
        WordWrapCallback& func,
        const WordWrapConfig& config = WordWrapConfig())
    {
        return CalcWordWrapping(
            writeSize,
            out,
            outSize,
            in,
            static_cast<u32>(wcslen(in)),
            func,
            config);
    }

    //!
    //! @brief ワードラッピングを行います。
    //!
    //! @param[out] writeSize  out に格納された文字列の長さを返します。
    //! @param[out] out  適切な位置に改行コードの追加された文字列が格納されます。
    //! @param[in] outSize  out に格納できる文字数を指定します。終端文字を含みます。
    //! @param[in] in  処理の対象となる文字列です。
    //! @param[in] inSize  in の文字列長です。
    //! @param[in] func  一行の表示範囲を得るコールバックです。
    //! @param[in] config  ワードラップの動作を指定します。
    //!
    //! @return
    //! 処理に成功した場合は true を返します。処理を中断した場合は false を返します。
    //!
    //! @details
    //! コールバック func が示す１行の表示範囲で適切な位置に改行コードを挿入した文字列を生成します。
    //!
    //! 文字列に改行が含まれる場合、改行コードはLFである必要があります。CRやCRLFは扱えません。
    //!
    //! 改行して良い場所が無かった場合には func が示す位置で強制的に改行します。
    //!
    //! 以下の場合は処理を中断します。その際でも out はヌル終端されます。
    //! @li out のサイズが足りない場合。
    //! @li config で指定された行数の上限を超えた場合。
    //!
    //! writeSize にアドレスを指定すると、out に格納された文字列の長さが書き込まれます。
    //!
    static bool
    CalcWordWrapping(
        u32* writeSize,
        char16* out,
        u32 outSize,
        const char16* in,
        u32 inSize,
        WordWrapCallback& func,
        const WordWrapConfig& config = WordWrapConfig());
};

}   // namespace font
}   // namespace nw

#endif // NW_FONT_WORDWRAPPING_H_
