﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nw/eft/eft2_Resource.h>
#include <nw/eft/eft2_Heap.h>
#include <nw/eft/eftcom_Guid.h>
#include <nw/eft/eftvw2_FileSystem.h>
#include <nw/eft/eft2_System.h>

namespace nw    {
namespace eftvw2{

#if EFT_IS_CAFE
static void* savedBuffer = NULL;    //!< TBD
static u32   savedSize = 0;         //!< TBD
//---------------------------------------------------------------------------
//! @brief TBD
//! @param[in] heap     TBD
//! @param[in] loadTextPath     TBD
//! @param[in] size     TBD
//! @return TBD
//---------------------------------------------------------------------------
static void* _binaryLoadCafe( nw::eft2::Heap*          heap,
               const char*         loadTextPath,
               u32*                size )
{
    void* buffer = NULL;

    FSMountSource  mountSrc;
    char           mountPath[FS_MAX_MOUNTPATH_SIZE];
    char           path[FS_MAX_ARGPATH_SIZE];
    FSStatus       status;
    FSStat         stat;
    FSFileHandle   fh;

    memset( path, 0, FS_MAX_ARGPATH_SIZE );
    FSClient* pClient = (FSClient *)heap->Alloc( sizeof( FSClient ) );
    FSCmdBlock* pCmd  = (FSCmdBlock *)heap->Alloc( sizeof( FSCmdBlock ) );
    FSAddClient( pClient, FS_RET_NO_ERROR );
    FSInitCmdBlock( pCmd );

    FSGetMountSource( pClient, pCmd, FS_SOURCETYPE_HFIO, &mountSrc, FS_RET_NO_ERROR );
    FSMount( pClient, pCmd, &mountSrc, mountPath, sizeof(mountPath), FS_RET_NO_ERROR );

    snprintf( path, sizeof(path), "%s/%s", mountPath, loadTextPath );
    for ( int i = 0; i < FS_MAX_ARGPATH_SIZE; i++ )
    {
        if ( path[i] == '\\' || path[i] == ':' )
        {
            path[i] = '/';
        }
    }

    status = FSGetStat( pClient, pCmd, path, &stat, FS_RET_NOT_FOUND );
    if ( status == FS_STATUS_OK )
    {
        buffer = heap->Alloc( stat.size + 4, nw::eft2::EFT_PTCL_BINARY_ALIGNMENT );
        memset( buffer, 0, stat.size + 4 );

        FSOpenFile( pClient, pCmd, path, "r", &fh, FS_RET_NO_ERROR );
        status = FSReadFile( pClient, pCmd, buffer, 1, stat.size, fh, 0, FS_RET_NO_ERROR );
        *size  = stat.size;
        FSCloseFile( pClient, pCmd, fh, FS_RET_NO_ERROR );
    }

    FSUnmount(pClient, pCmd, mountPath, FS_RET_NO_ERROR);
    FSDelClient(pClient, FS_RET_NO_ERROR);
    heap->Free( pClient );
    heap->Free( pCmd );

    return buffer;
}

//---------------------------------------------------------------------------
//! @brief TBD
//! @param[in] heap     TBD
//! @param[in] outBinPath     TBD
//! @param[in] buffer     TBD
//! @param[in] length     TBD
//! @return TBD
//---------------------------------------------------------------------------
static s32 _binaryWriteCafe( nw::eft2::Heap*     heap,
                      const char*         outBinPath,
                      const void*         buffer,
                      u32                 length )
{

    FSMountSource  mountSrc;
    char           mountPath[FS_MAX_MOUNTPATH_SIZE];
    char           path[FS_MAX_ARGPATH_SIZE];
    FSStatus       status;
    FSFileHandle   fh;

    memset( path, 0, FS_MAX_ARGPATH_SIZE );
    FSClient* pClient = (FSClient *)heap->Alloc( sizeof( FSClient ) );
    FSCmdBlock* pCmd  = (FSCmdBlock *)heap->Alloc( sizeof( FSCmdBlock ) );
    FSAddClient( pClient, FS_RET_NO_ERROR );
    FSInitCmdBlock( pCmd );

    FSGetMountSource( pClient, pCmd, FS_SOURCETYPE_HFIO, &mountSrc, FS_RET_NO_ERROR );
    FSMount( pClient, pCmd, &mountSrc, mountPath, sizeof(mountPath), FS_RET_NO_ERROR );

    snprintf( path, sizeof(path), "%s/%s", mountPath, outBinPath );
    for ( int i = 0; i < FS_MAX_ARGPATH_SIZE; i++ )
    {
        if ( path[i] == '\\' || path[i] == ':' )
        {
            path[i] = '/';
        }
    }

    status = FSOpenFile( pClient, pCmd, path, "w", &fh, FS_RET_NO_ERROR );
    status = FSWriteFile( pClient, pCmd, buffer, 1, 32, fh, 0, FS_RET_ALL_ERROR );
//    status = FSWriteFile( pClient, pCmd, buffer, 1, length, fh, 0, FS_RET_NO_ERROR );
    FSCloseFile( pClient, pCmd, fh, FS_RET_NO_ERROR );

    FSUnmount(pClient, pCmd, mountPath, FS_RET_NO_ERROR);
    FSDelClient(pClient, FS_RET_NO_ERROR);
    heap->Free( pClient );
    heap->Free( pCmd );

    return (s32)status;
}

#endif

//---------------------------------------------------------------------------
//! @brief        リソースマネージャクラス
//---------------------------------------------------------------------------
class ResourceManager
{
public:

    //----------------------------------------------------------------------
    //! @brief  エフェクトリソース管理構造体。
    //----------------------------------------------------------------------
    struct EffectResource
    {
        void*           binary;         //!< TBD
        eftcom::Guid    guid;           //!< TBD
        s32             resId;          //!< TBD
        s32             releaseCnt;     //!< TBD

        //----------------------------------------------------------------------
        //! @brief  TBD
        //----------------------------------------------------------------------
        void Initialize()
        {
            binary      = NULL;
            for( u32 i = 0; i < 16; i++ ) guid.data[i] = 0;
            releaseCnt  = -1;
            resId     = -1;
        }
    };

    //---------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //---------------------------------------------------------------------------
    ResourceManager()
    {
        mResourceNum        = 0;
        mResourceStartId    = 8;
        mCacheResourceNo    = -1;
    }

    //---------------------------------------------------------------------------
    //! @brief  リソースマネージャの初期化を行います。
    //! @param[in]      eftHeap     パケットデータです.
    //! @param[in]      ctrlResourceNum     パケットデータです.
    //! @param[in]      userResourceNum     パケットデータです.
    //---------------------------------------------------------------------------
    void Initialize( nw::eft2::Heap* eftHeap, u32 ctrlResourceNum, u32 userResourceNum )
    {
        mEftHeap            = eftHeap;
        mResourceNum        = ctrlResourceNum;
        mResourceStartId    = userResourceNum;

        // 管理用ワークを確保し、初期化
        mEffectResourceArray = static_cast<EffectResource*>( mEftHeap->Alloc(
                                            sizeof( EffectResource ) * mResourceNum ) );
        for ( u32 i = 0; i < mResourceNum; ++ i ) { mEffectResourceArray[i].Initialize(); }
    }

    //---------------------------------------------------------------------------
    //! @brief  リソースマネージャの終了処理を行います。
    //---------------------------------------------------------------------------
    void Finalize()
    {
        for ( u32 i = 0; i < mResourceNum; ++ i )
        {
            if ( mEffectResourceArray[i].binary )
            {
                mEftHeap->Free( mEffectResourceArray[i].binary );
            }
        }

        mEftHeap->Free( mEffectResourceArray );
    }


    //---------------------------------------------------------------------------
    //! @brief  保持するリソースの破棄登録を行います。
    //---------------------------------------------------------------------------
    void EntryResDestroy()
    {
        for ( u32 i = 0; i < mResourceNum; ++ i )
        {
            if ( mEffectResourceArray[i].binary )
            {
                Release( mEffectResourceArray[i].guid );
            }
        }
    }

    //---------------------------------------------------------------------------
    //! @brief  リソースを登録します。
    //! @param[in]      effectBinaryPath     パケットデータです.
    //! @param[in]      guid     パケットデータです.
    //! @return TBD
    //---------------------------------------------------------------------------
    bool Entry( const char* effectBinaryPath, eftcom::Guid guid )
    {
        // 登録可能かどうか。
        s32 slot = GetEmptySlot();
        if ( slot == -1 ) { return false; }

        EffectResource* res = &mEffectResourceArray[slot];
        res->Initialize();

        // 指定パスのエフェクトバイナリをロードする
        u32   bufferSize = 0;
        FileSystem::Load( mEftHeap, effectBinaryPath, &res->binary, &bufferSize );

        // 実際のリソースIDにオフセット
        res->resId  = slot + mResourceStartId;
        res->guid   = guid;

        return true;
    }

    //---------------------------------------------------------------------------
    //! @brief  リリースフラグの立っているリソースを解放する。
    //! @return TBD
    //---------------------------------------------------------------------------
    void Flush( nw::eft2::System* pEftSystem, nw::eft2::Heap* pViewerHeap )
    {
        for ( u32 i = 0; i < mResourceNum; ++ i )
        {
            if ( mEffectResourceArray[i].releaseCnt > 0 )
            {
                mEffectResourceArray[i].releaseCnt--;

                if ( mEffectResourceArray[i].releaseCnt == 0 )
                {
                    const s32 resId = mEffectResourceArray[i].resId;
                    if ( resId != -1 )
                    {
                        pEftSystem->ClearResource( pViewerHeap, resId );
                    }

                    if ( mEffectResourceArray[i].binary )
                    {
                        mEftHeap->Free( mEffectResourceArray[i].binary );
                    }
                    else
                    {
                        nw::eft2::OutputError( "[EFTVW] Resource Rleease Error Occured. \n" );
                    }
                    mEffectResourceArray[i].Initialize();
                }
            }
        }
    }

    //---------------------------------------------------------------------------
    //! @brief  guidを指定してリソースを破棄します。
    //! @param[in]      guid     TBD
    //! @return TBD
    //---------------------------------------------------------------------------
    bool Release( eftcom::Guid guid )
    {
        s32 slot = SerchResourceSlot( guid );
        if ( slot == -1 ) return false;

        // 遅延解放の為の待機フレームを設定
        mEffectResourceArray[slot].releaseCnt = 10;

        return true;
    }

    //---------------------------------------------------------------------------
    //! @brief  guidを指定して、リソースの登録先スロットを検索します。
    //! @param[in]      guid     TBD
    //! @return TBD
    //---------------------------------------------------------------------------
    s32 SerchResourceSlot( eftcom::Guid guid )
    {
        for ( u32 i = 0; i < mResourceNum; ++ i )
        {
            if ( mEffectResourceArray[i].guid       == guid &&
                 mEffectResourceArray[i].releaseCnt == -1 )
            {
                return i;
            }
        }

        return -1;
    }

    //---------------------------------------------------------------------------
    //! @brief  エミッタセット名を指定して、リソースを取得します。
    //! @param[in] guid     エフェクトリソースguid
    //! @return エフェクトリソース
    //---------------------------------------------------------------------------
    EffectResource* GetResource( eftcom::Guid guid )
    {
        s32 slot = SerchResourceSlot( guid );
        if ( slot == -1 ) return NULL;
        return &mEffectResourceArray[slot];
    }

    //---------------------------------------------------------------------------
    //! @brief  エミッタセット名を指定して、リソースを取得します。
    //! @param[in] guid     エフェクトリソースインデックス
    //! @return エフェクトリソース
    //---------------------------------------------------------------------------
    EffectResource* GetResource( u32 index )
    {
        if ( mResourceNum <= index )
        {
            return NULL;
        }
        return &mEffectResourceArray[ index ];
    }

    //---------------------------------------------------------------------------
    //! @brief  管理しているリソース数を取得します。
    //! @return エフェクトリソース数
    //---------------------------------------------------------------------------
    u32 GetResourceCount() const
    {
        return mResourceNum;
    }
private:
    //---------------------------------------------------------------------------
    //! @brief   空いているスロットを取得します。
    //! @return TBD
    //---------------------------------------------------------------------------
    s32 GetEmptySlot()
    {
        for ( u32 i = 0; i < ( mResourceNum - mResourceStartId ); ++i )
        {
            if ( !mEffectResourceArray[i].binary )
            {
                return i;
            }
        }

        return -1;
    }

    nw::eft2::Heap*     mEftHeap;               //!< TBD
    u32                 mResourceNum;           //!< 管理可能な最大リソース数
    s32                 mCacheResourceNo;       //!< 管理可能な最大リソース数
    u32                 mResourceStartId;       //!< ビューアとして利用可能なリソースID
    EffectResource*     mEffectResourceArray;   //!< エフェクトリソース配列
};


} // namespace eftvw
} // namespace nw
