﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once



#ifndef _EFT_TOOL
#include <nw/math.h>
#endif//_EFT_TOOL

#include <nw/eft/eft2_EndianUtil.h>

namespace nw   {
namespace eftvw2 {

//------------------------------------------------------------------------------
//! @brief      カメラリソース
//------------------------------------------------------------------------------
struct ResCamera
{
    u8                                                isLinearMode;                                             //!< リニアモード
    u8                                                isRealtimeEditing;                                        //!< リアルタイム編集かどうかを判別するためのフラグです。ランタイム側のみで使用します。
    u8                                                displayDebugInformation;                                  //!< デバッグ情報表示のon/off
    u8                                                displayGpuProfile;                                        //!< GPU処理の計測結果を表示on/off
    u32                                               gridType;                                                 //!< グリッドタイプ
    f32                                               frameRate;                                                //!< フレームレート
    nw::math::VEC3                                    position;                                                 //!< カメラ位置です
    nw::math::VEC3                                    target;                                                   //!< カメラの注視点です
    f32                                               rotation;                                                 //!< カメラの回転角です
    s32                                               projectionType;                                           //!< 射影タイプです
    f32                                               aspectRatio;                                              //!< アスペクト比です
    f32                                               fieldOfView;                                              //!< パースペクティブ時の垂直画角です
    f32                                               imageHeight;                                              //!< オルソ時の画像の高さ
    f32                                               nearClip;                                                 //!< 近クリップ平面への距離です
    f32                                               farClip;                                                  //!< 遠クリップ平面への距離です
    nw::math::VEC4                                    bgColor;                                                  //!< 背景色
    u32                                               guid;                                                     //!< 背景画像テクスチャID
    nw::math::VEC4                                    imageColor;                                               //!< 背景画像への乗算カラー
    u8                                                enablePad;                                                //!< Padの有効化/無効化
    bool                                              displayGrid;                                              //!< グリッド表示のon/off
    bool                                              displayParentEmitterShape;                                //!< 親エミッタ形状表示のon/off

    bool                                              displayChildEmitterShape;                                 //!< 子エミッタ形状表示のon/off
    bool                                              displayFieldShape;                                        //!< フィールド形状表示のon/off
    bool                                              restrainViewrWarnings;                                    //!< ビューアの警告を抑制
    u8                                                dummay[2];                                                //!< アライン用

    //--------------------------------------------------------------------------
    //! @brief      エンディアンをフリップします.
    //--------------------------------------------------------------------------
    void FlipEndian()
    {
        nw::eft2::EndianUtil::Flip( &gridType );
        nw::eft2::EndianUtil::Flip( &frameRate );
        nw::eft2::EndianUtil::Flip( &position );
        nw::eft2::EndianUtil::Flip( &target );
        nw::eft2::EndianUtil::Flip( &rotation );
        nw::eft2::EndianUtil::Flip( &projectionType );
        nw::eft2::EndianUtil::Flip( &aspectRatio );
        nw::eft2::EndianUtil::Flip( &fieldOfView );
        nw::eft2::EndianUtil::Flip( &imageHeight );
        nw::eft2::EndianUtil::Flip( &nearClip );
        nw::eft2::EndianUtil::Flip( &farClip );
        nw::eft2::EndianUtil::Flip( &bgColor );
        nw::eft2::EndianUtil::Flip( &guid );
        nw::eft2::EndianUtil::Flip( &imageColor );

    }
};

//------------------------------------------------------------------------------
//! @brief      コンフィグ設定データです。
//------------------------------------------------------------------------------
struct ResConfig
{
    s32                                               frameRate;                                                //!< フレームレートです。
    s32                                               resolution;                                               //!< 解像度です。
    nw::math::VEC4                                    clearColor;                                               //!< 背景カラーです。
    f32                                               worldScaleRange;                                          //!< ワールドスケール範囲です。
    f32                                               worldScaleTime;                                           //!< ワールドスケール時間です。
    f32                                               gridScale;                                                //!< グリッドスケールです。
    s32                                               isLinearEditMode;                                         //!< リニア編集モードかどうかを示すフラグです。

    //--------------------------------------------------------------------------
    //! @brief      エンディアンをフリップします.
    //--------------------------------------------------------------------------
    void FlipEndian()
    {
        nw::eft2::EndianUtil::Flip( &frameRate );
        nw::eft2::EndianUtil::Flip( &resolution );
        nw::eft2::EndianUtil::Flip( &clearColor );
        nw::eft2::EndianUtil::Flip( &worldScaleRange );
        nw::eft2::EndianUtil::Flip( &worldScaleTime );
        nw::eft2::EndianUtil::Flip( &gridScale );
        nw::eft2::EndianUtil::Flip( &isLinearEditMode );

    }
};

//------------------------------------------------------------------------------
//! @brief      再生範囲設定データです。
//------------------------------------------------------------------------------
struct ResPlayRange
{
    s32                                               startFrame;                                               //!< 開始フレームです。
    s32                                               endFrame;                                                 //!< 終了フレームです。
    s32                                               isLoop;                                                   //!< ループさせるかどうかのフラグです。

    //--------------------------------------------------------------------------
    //! @brief      エンディアンをフリップします.
    //--------------------------------------------------------------------------
    void FlipEndian()
    {
        nw::eft2::EndianUtil::Flip( &startFrame );
        nw::eft2::EndianUtil::Flip( &endFrame );
        nw::eft2::EndianUtil::Flip( &isLoop );

    }
};

//------------------------------------------------------------------------------
//! @brief      コンストレインデータです。
//------------------------------------------------------------------------------
struct ResConstrain
{
    s32                                               matrixSetType;                                            //!< 行列設定タイプです。
    s32                                               matrixApplyType;                                          //!< マトリクス適用タイプです。
    nw::eftcom::Guid                     modelGuid;                                                //!< 接続モデルです。
    s32                                               boneIndex;                                                //!< 接続ボーンです。
    nw::math::VEC3                                    offsetPosition;                                           //!< オフセット行列 座標値です。
    nw::math::VEC3                                    offsetScale;                                              //!< オフセット行列 スケール値です。
    f32                                               offsetRotateX;                                            //!< オフセット行列 回転Xです。
    f32                                               offsetRotateY;                                            //!< オフセット行列 回転Yです。
    f32                                               offsetRotateZ;                                            //!< オフセット行列 回転Zです。
    s32                                               isEmitterBillboard;                                       //!< エミッタビルボードです。

    //--------------------------------------------------------------------------
    //! @brief      エンディアンをフリップします.
    //--------------------------------------------------------------------------
    void FlipEndian()
    {
        nw::eft2::EndianUtil::Flip( &matrixSetType );
        nw::eft2::EndianUtil::Flip( &matrixApplyType );
        nw::eft2::EndianUtil::Flip( &boneIndex );
        nw::eft2::EndianUtil::Flip( &offsetPosition );
        nw::eft2::EndianUtil::Flip( &offsetScale );
        nw::eft2::EndianUtil::Flip( &offsetRotateX );
        nw::eft2::EndianUtil::Flip( &offsetRotateY );
        nw::eft2::EndianUtil::Flip( &offsetRotateZ );
        nw::eft2::EndianUtil::Flip( &isEmitterBillboard );

    }
};

//------------------------------------------------------------------------------
//! @brief      プレビュー基本設定データです。
//------------------------------------------------------------------------------
struct ResPreviewFnd
{
    s32                                               startFrame;                                               //!< 再生開始フレームです。
    s32                                               forceStartFrame;                                          //!< 強制開始フレームです。

    //--------------------------------------------------------------------------
    //! @brief      エンディアンをフリップします.
    //--------------------------------------------------------------------------
    void FlipEndian()
    {
        nw::eft2::EndianUtil::Flip( &startFrame );
        nw::eft2::EndianUtil::Flip( &forceStartFrame );

    }
};

//------------------------------------------------------------------------------
//! @brief      カラーデータです。
//------------------------------------------------------------------------------
struct ResPreviewColor
{
    nw::math::VEC4                                    color;                                                    //!< カラーです。

    //--------------------------------------------------------------------------
    //! @brief      エンディアンをフリップします.
    //--------------------------------------------------------------------------
    void FlipEndian()
    {
        nw::eft2::EndianUtil::Flip( &color );

    }
};

//------------------------------------------------------------------------------
//! @brief      パーティクル放出データです。
//------------------------------------------------------------------------------
struct ResPreviewEmission
{
    f32                                               emissionRate;                                             //!< 放出レートです。
    f32                                               emissionInterval;                                         //!< 放出間隔です。

    //--------------------------------------------------------------------------
    //! @brief      エンディアンをフリップします.
    //--------------------------------------------------------------------------
    void FlipEndian()
    {
        nw::eft2::EndianUtil::Flip( &emissionRate );
        nw::eft2::EndianUtil::Flip( &emissionInterval );

    }
};

//------------------------------------------------------------------------------
//! @brief      パーティクルスケールデータです。
//------------------------------------------------------------------------------
struct ResPreviewPtclScale
{
    nw::math::VEC3                                    particleScale;                                            //!< パーティクルの大きさです。
    nw::math::VEC3                                    emissionParticleScale;                                    //!< 放出時のみ作用するパーティクルの大きさ。
    nw::math::VEC3                                    emitterVolumeScale;                                       //!< エミッタ形状スケールです。

    //--------------------------------------------------------------------------
    //! @brief      エンディアンをフリップします.
    //--------------------------------------------------------------------------
    void FlipEndian()
    {
        nw::eft2::EndianUtil::Flip( &particleScale );
        nw::eft2::EndianUtil::Flip( &emissionParticleScale );
        nw::eft2::EndianUtil::Flip( &emitterVolumeScale );

    }
};

//------------------------------------------------------------------------------
//! @brief      パーティクル制御データです。
//------------------------------------------------------------------------------
struct ResPreviewPtclControl
{
    f32                                               life;                                                     //!< ライフスケールです。
    f32                                               allDirectionVel;                                          //!< 全方向初速スケールです。
    f32                                               directionalVel;                                           //!< 指定方向初速スケールです。
    f32                                               randomVel;                                                //!< 初速度のランダムスケールです。
    nw::math::VEC3                                    addVel;                                                   //!< 速度に加算するワールドベクトルです。
    nw::math::VEC3                                    emitterColor0;                                            //!< エミッタカラー0です。
    nw::math::VEC3                                    emitterColor1;                                            //!< エミッタカラー1です。
    u8                                                enableEmitterColor;                                       //!< エミッタカラー有効かどうかのフラグです。
    u8                                                dummy0;                                                   //!< パディング領域
    u8                                                dummy1[2];                                                //!< パディング領域

    //--------------------------------------------------------------------------
    //! @brief      エンディアンをフリップします.
    //--------------------------------------------------------------------------
    void FlipEndian()
    {
        nw::eft2::EndianUtil::Flip( &life );
        nw::eft2::EndianUtil::Flip( &allDirectionVel );
        nw::eft2::EndianUtil::Flip( &directionalVel );
        nw::eft2::EndianUtil::Flip( &randomVel );
        nw::eft2::EndianUtil::Flip( &addVel );
        nw::eft2::EndianUtil::Flip( &emitterColor0 );
        nw::eft2::EndianUtil::Flip( &emitterColor1 );

    }
};

//------------------------------------------------------------------------------
//! @brief      自動移動データです。
//------------------------------------------------------------------------------
struct ResPreviewAutoMove
{
    s32                                               moveType;                                                 //!< 自動移動タイプです。
    f32                                               radiusXZ;                                                 //!< XZ移動半径です。
    f32                                               speedXZ;                                                  //!< XZ移動速度です。
    f32                                               amplitudeY;                                               //!< Y揺れ振幅です。
    f32                                               speedY;                                                   //!< Y揺れ速度です。
    f32                                               rollZ;                                                    //!< Zロール速度です。

    //--------------------------------------------------------------------------
    //! @brief      エンディアンをフリップします.
    //--------------------------------------------------------------------------
    void FlipEndian()
    {
        nw::eft2::EndianUtil::Flip( &moveType );
        nw::eft2::EndianUtil::Flip( &radiusXZ );
        nw::eft2::EndianUtil::Flip( &speedXZ );
        nw::eft2::EndianUtil::Flip( &amplitudeY );
        nw::eft2::EndianUtil::Flip( &speedY );
        nw::eft2::EndianUtil::Flip( &rollZ );

    }
};

//------------------------------------------------------------------------------
//! @brief      プレビューデータです。
//------------------------------------------------------------------------------
struct ResPreview
{
    eftvw2::ResConstrain             constrain;                                                //!< SRT制御データです。
    eftvw2::ResPreviewAutoMove       autoMove;                                                 //!< 自動移動データです。

    //--------------------------------------------------------------------------
    //! @brief      エンディアンをフリップします.
    //--------------------------------------------------------------------------
    void FlipEndian()
    {
        /* DO_NOTHING */

        constrain.FlipEndian();
        autoMove.FlipEndian();
    }
};

//------------------------------------------------------------------------------
//! @brief      モデルプレビュー
//------------------------------------------------------------------------------
struct ResModelPreview
{
    f32                                               animSpeed;                                                //!< アニメーション再生速度です。
    f32                                               animStartFrame;                                           //!< アニメーション再生開始フレームです。

    //--------------------------------------------------------------------------
    //! @brief      エンディアンをフリップします.
    //--------------------------------------------------------------------------
    void FlipEndian()
    {
        nw::eft2::EndianUtil::Flip( &animSpeed );
        nw::eft2::EndianUtil::Flip( &animStartFrame );

    }
};

//------------------------------------------------------------------------------
//! @brief      モデルプレビューのデータセットです。
//------------------------------------------------------------------------------
struct ResModelPreviewPack
{
    eftvw2::ResModelPreview          modelPreview;                                             //!< モデルプレビューデータです。
    eftvw2::ResPreview               preview;                                                  //!< プレビューデータです。

    //--------------------------------------------------------------------------
    //! @brief      エンディアンをフリップします.
    //--------------------------------------------------------------------------
    void FlipEndian()
    {
        /* DO_NOTHING */

        modelPreview.FlipEndian();
        preview.FlipEndian();
    }
};

//------------------------------------------------------------------------------
//! @brief      エフェクトプレビューデータです。
//------------------------------------------------------------------------------
struct ResEffectPreview
{
    eftvw2::ResPreviewFnd            basicSetting;                                             //!< 基本設定データです。
    eftvw2::ResPreviewColor          color;                                                    //!< カラーデータです。
    eftvw2::ResPreviewEmission       emission;                                                 //!< パーティクル放出データです。
    eftvw2::ResPreviewPtclScale      ptclScale;                                                //!< パーティクルスケールデータです。
    eftvw2::ResPreviewPtclControl    ptclControl;                                              //!< パーティクル制御データです。

    //--------------------------------------------------------------------------
    //! @brief      エンディアンをフリップします.
    //--------------------------------------------------------------------------
    void FlipEndian()
    {
        /* DO_NOTHING */

        basicSetting.FlipEndian();
        color.FlipEndian();
        emission.FlipEndian();
        ptclScale.FlipEndian();
        ptclControl.FlipEndian();
    }
};

//------------------------------------------------------------------------------
//! @brief      エフェクトプレビューのデータセットです。
//------------------------------------------------------------------------------
struct ResEffectPreviewPack
{
    eftvw2::ResEffectPreview         effectPreview;                                            //!< エフェクトプレビューデータです。
    eftvw2::ResPreview               preview;                                                  //!< プレビューデータです。

    //--------------------------------------------------------------------------
    //! @brief      エンディアンをフリップします.
    //--------------------------------------------------------------------------
    void FlipEndian()
    {
        /* DO_NOTHING */

        effectPreview.FlipEndian();
        preview.FlipEndian();
    }
};

//------------------------------------------------------------------------------
//! @brief      ptclバイナリデータです。
//------------------------------------------------------------------------------
struct ResPtclBinary
{
    s32                                               reloadCtrl;                                               //!< リロード呼び出しを差分形式でコントロールします。
    s32                                               esetIndexCtrl;                                            //!< 再生するエミッタセットのインデックスを差分形式でコントロールします。
    s32                                               pathLength;                                               //!< ファイルパスの文字数。
    u8                                                filePath[4];                                              //!< ファイルパス。

    //--------------------------------------------------------------------------
    //! @brief      エンディアンをフリップします.
    //--------------------------------------------------------------------------
    void FlipEndian()
    {
        nw::eft2::EndianUtil::Flip( &reloadCtrl );
        nw::eft2::EndianUtil::Flip( &esetIndexCtrl );
        nw::eft2::EndianUtil::Flip( &pathLength );

    }
};


} // namespace eftvw2
} // namespace nw

