﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nw/eft/eft2_Callback.h>

namespace nw    {
namespace eft2  {

//------------------------------------------------------------------------------
//! @brief  ストライプ定数
//------------------------------------------------------------------------------
enum SuperStripeConst
{
    EFT_SUPER_STRIPE_HISTRY_NUM = 256,                      //!< 最大履歴数
};

//------------------------------------------------------------------------------
//! @brief  テクスチャマッピング方式
//------------------------------------------------------------------------------
enum TextureMappingType
{
    EFT_TEXTURE_MAPPING_TYPE_UNIFORM        = 0,            //!< 均等割り当て
    EFT_TEXTURE_MAPPING_TYPE_DISTANCE_BASED = 1,            //!< 距離依存
};

//------------------------------------------------------------------------------
//! @brief  ストライプ履歴。パーティクルの移動の記録。
//------------------------------------------------------------------------------
struct SuperStripeHistory
{
    nw::math::VEC3 pos;                                     //!< エミッタ（or ワールド）座標系での座標
    nw::math::VEC3 vec;                                     //!< 履歴点の速度
    nw::math::VEC3 dir;                                     //!< 背骨の方向（履歴点の速度ではない）
    nw::math::VEC3 wing;                                    //!< ポリゴンを展開する方向.xyz
    nw::math::VEC3 emitterSRT_Y;                            //!< その時のエミッタ行列のY軸成分
    f32            scale;                                   //!< スケール
};

//---------------------------------------------------------------------------
// ストライプ頂点バッファ
//---------------------------------------------------------------------------
struct SuperStripeVertexBuffer
{
    // EPパラメータ
    nw::math::VEC4* pos;                                    //!< ( 履歴の位置.xyz / 羽の符号付長さ )
    nw::math::VEC4* dir;                                    //!< ( 背骨の方向.xyz / )
    nw::math::VEC4* wing;                                   //!< ( 羽の方向.xyz /  )
    nw::math::VEC4* tex;                                    //!< ( テクスチャリング.012 / )
    nw::math::VEC4* emat;                                   //!< ( matv.xyz / )
};

//------------------------------------------------------------------------------
//! @brief  ストライプ
//------------------------------------------------------------------------------
struct SuperStripe
{
    SuperStripeHistory      hist[ EFT_SUPER_STRIPE_HISTRY_NUM ];    //!< ヒストリ
    SuperStripeVertexBuffer vertexBuffer;                           //!< 頂点バッファ
    nw::math::VEC4          color0;                                 //!< カラー0
    nw::math::VEC4          color1;                                 //!< カラー1
    nw::math::VEC4          random;                                 //!< ランダム
    nw::math::VEC3          interpolateNextDir;                     //!< 補間ベクトル
    SuperStripe*            next;                                   //!< 遅延終了処理用のリスト
    u32                     numHistory;                             //!< ヒストリが何個分あるか？
    u32                     numDelayHistory;                        //!< 遅延ストライプに入った時のストライプ数
    u32                     vertexBufferNum;                        //!< 頂点バッファに書き込んだ数
    f32                     time;                                   //!< 生成時間
    f32                     life;                                   //!< 寿命
    f32                     prevFrame;                              //!< 前回のフレーム値
    f32                     journey;                                //!< ストライプが飛んだ距離
    bool                    used;                                   //!< 利用中かどうか
};

//---------------------------------------------------
//! @brief  ストライプを管理するクラスです。
//---------------------------------------------------
class SuperStripeSystem
{
public:
    static const EmitterPluginResourceIndex  PluginId = EFT_EMITTER_PLUGIN_ID_3;   //!< エミッタプラグインID
public:
    //------------------------------------------------------------------------------
    //! @brief  ストライプシステムの初期化処理を行います。
    //! @param[in] heap         ヒープへのポインタ
    //! @param[in] system       Systemへのポインタ
    //! @param[in] bufferMode   バッファリングモード
    //! @param[in] stripeNum    ストライプの最大数
    //------------------------------------------------------------------------------
    static void Initialize( nw::eft2::Heap* heap, nw::eft2::System* system, BufferMode bufferMode, u32 stripeNum );

    //------------------------------------------------------------------------------
    //! @brief  ストライプシステムの終了処理を行います。
    //! @param[in] heap         ヒープへのポインタ
    //------------------------------------------------------------------------------
    static void Finalize( nw::eft2::Heap* heap );

    //------------------------------------------------------------------------------
    //! @brief  ストライプシステムで確保したワークサイズを取得します。
    //! @return 確保したワークサイズ
    //------------------------------------------------------------------------------
    static u32 GetWorkSize();

    //------------------------------------------------------------------------------
    //! @brief      ワンタイムエミッタ用の追加のエミッタ寿命を取得
    //! @param[in]  pEmitter    エミッタへのポインタ
    //! @return                 ワンタイムエミッタ用の追加のエミッタ寿命
    //------------------------------------------------------------------------------
    static s32 GetExtendedEndTimeForOneTimeEmitter( nw::eft2::Emitter* pEmitter );

    //------------------------------------------------------------------------------
    //! @brief          エミッタ生成後コールバック
    //! @param[in] arg  コールバック引数
    //! @return         成功すればtrue
    //------------------------------------------------------------------------------
    static bool _EmitterInitializeCallback( EmitterInitializeArg& arg );

    //------------------------------------------------------------------------------
    //! @brief          パーティクル生成コールバック
    //! @param[in] arg  コールバック引数
    //! @return         成功すればtrue
    //------------------------------------------------------------------------------
    static bool _ParticleEmitCallback( ParticleEmitArg& arg );

    //------------------------------------------------------------------------------
    //! @brief          パーティクル削除コールバック
    //! @param[in] arg  コールバック引数
    //! @return         成功すればtrue
    //------------------------------------------------------------------------------
    static bool _ParticleRemoveCallback( ParticleRemoveArg& arg );

    //------------------------------------------------------------------------------
    //! @brief          パーティクル計算コールバック
    //! @param[in] arg  コールバック引数
    //------------------------------------------------------------------------------
    static void _ParticleCalcCallback( ParticleCalcArg& arg );

    //------------------------------------------------------------------------------
    //! @brief          エミッタ計算処理前コールバック
    //! @param[in] arg  コールバック引数
    //------------------------------------------------------------------------------
    static void _EmitterPreCalcCallback( EmitterPreCalcArg& arg );

    //------------------------------------------------------------------------------
    //! @brief          エミッタ計算処理後コールバック
    //! @param[in] arg  コールバック引数
    //------------------------------------------------------------------------------
    static void _EmitterPostCalcCallback( EmitterPostCalcArg& arg );

    //------------------------------------------------------------------------------
    //! @brief          エミッタ描画コールバック
    //! @param[in] arg  コールバック引数
    //! @return         成功すればtrue
    //------------------------------------------------------------------------------
    static bool _EmitterDrawCallback( EmitterDrawArg& arg );

    //------------------------------------------------------------------------------
    //! @brief          エミッタ破棄後コールバック
    //! @param[in] arg  コールバック引数
    //------------------------------------------------------------------------------
    static void _EmitterFinalizeCallback( EmitterFinalizeArg& arg );

private:
    //----------------------------------------
    //! @name コンストラクタ／デストラクタ
    //@{

    //---------------------------------------------------------------------------
    //! @brief  ストライプシステムの生成を行います。
    //! @param[in] heap         ヒープへのポインタ
    //! @param[in] system       Systemへのポインタ
    //! @param[in] bufferMode   バッファリングモード
    //! @param[in] stripeNum    ストライプ最大数
    //---------------------------------------------------------------------------
    explicit SuperStripeSystem( nw::eft2::Heap* heap, nw::eft2::System* system, BufferMode bufferMode, u32 stripeNum );

    //---------------------------------------------------------------------------
    //! @brief  ストライプシステムの破棄を行います。
    //---------------------------------------------------------------------------
    virtual ~SuperStripeSystem();

    //---------------------------------------------------------------------------
    //! @brief  ストライプの履歴を更新します。
    //! @param[in] arg              コールバック引数
    //! @param[in] stripe           ストライプへのポインタ
    //! @param[in] stripeData       ストライプ設定へのポインタ
    //! @param[in] doPushHistory    履歴を積むかどうか
    //---------------------------------------------------------------------------
    void UpdateHistory(
        ParticleCalcArg& arg,
        SuperStripe* stripe,
        ResStripeSuper* stripeData, const bool doPushHistory );

    //---------------------------------------------------------------------------
    //! @brief  空きストライプインスタンスを確保します。
    //! @return ストライプへのポインタ
    //---------------------------------------------------------------------------
    SuperStripe* Alloc();

    //---------------------------------------------------------------------------
    //! @brief              ストライプインスタンスを返却します。
    //! @param[in] stripe   ストライプへのポインタ
    //---------------------------------------------------------------------------
    void Free( SuperStripe* stripe );

    //---------------------------------------------------------------------------
    //! @brief              ストライプインスタンスを遅延リストへ追加します。
    //! @param[in] emitter  エミッタへのポインタ
    //! @param[in] stripe   ストライプへのポインタ
    //---------------------------------------------------------------------------
    void AddDelayList( Emitter* emitter, SuperStripe* stripe );

    //---------------------------------------------------------------------------
    //! @brief  バッファをスワップする。
    //! @param[in] emitter     エミッタへのポインタ
    //---------------------------------------------------------------------------
    void SwapBuffer( Emitter* emitter );

    //---------------------------------------------------------------------------
    //! @brief  ポリゴン用ワークを確保します。
    //! @param[in] emitter     エミッタへのポインタ
    //! @return                 成功した場合 true
    //---------------------------------------------------------------------------
    bool AllocStripeVertexBuffer( Emitter* emitter );

    //---------------------------------------------------------------------------
    //! @brief  遅延解放ストライプ計算処理を行います。
    //! @param[in] emitter     エミッタへのポインタ
    //---------------------------------------------------------------------------
    void CalcDelayStripe( Emitter* emitter );

    //---------------------------------------------------------------------------
    //! @brief  遅延解放ストライトを破棄します。
    //! @param[in] emitter     エミッタへのポインタ
    //---------------------------------------------------------------------------
    void FreeDelayStripeList( Emitter* emitter );

    //---------------------------------------------------------------------------
    //! @brief  ストライプ計算処理を行います。
    //! @param[in] arg      コールバック引数
    //! @param[in] stripe   ストライプへのポインタ
    //---------------------------------------------------------------------------
    void CalcStripe( ParticleCalcArg& arg, SuperStripe* stripe );

    //---------------------------------------------------------------------------
    //! @brief  エミッタ描画処理を行います。
    //! @param[in] emitter      エミッタへのポインタ
    //! @param[in] shaderType   シェーダタイプ
    //! @param[in] userParam    ユーザーパラメータへのポインタ
    //---------------------------------------------------------------------------
    void Draw( Emitter* emitter, ShaderType shaderType, void* userParam );

    //---------------------------------------------------------------------------
    //! @brief  現在のストライプ数取得を行います。
    //! @return 現在のストライプ数
    //---------------------------------------------------------------------------
    u32 GetProcessingStripeNum() const { return mStripeNum; }

    //---------------------------------------------------------------------------
    //! @brief          ストライプのポリゴン生成を行います。
    //! @param[out]     vertexBuffer 頂点バッファへのポインタ
    //! @param[in,out]  stripe       ストライプへのポインタ
    //! @param[in]      stripeData   ストライプ設定へのポインタ
    //! @param[in]      emitter      エミッタへのポインタ
    //! @param[in]      indexOffset  テクスチャオフセット値
    //---------------------------------------------------------------------------
    void MakeStripeVertexBuffer(
        SuperStripeVertexBuffer* vertexBuffer,
        SuperStripe* stripe,
        const ResStripeSuper* stripeData,
        const Emitter* emitter,
        f32 indexOffset = 0.0f );

    //---------------------------------------------------------------------------
    //! @brief          ストライプのポリゴン生成を行います。
    //! @param[out]     vertexBuffer 頂点バッファへのポインタ
    //! @param[in,out]  stripe       ストライプへのポインタ
    //! @param[in]      stripeData   ストライプ設定へのポインタ
    //! @param[in]      emitter      エミッタへのポインタ
    //! @param[in]      indexOffset  テクスチャオフセット値
    //---------------------------------------------------------------------------
    void MakeStripeVertexBufferWithDivision(
        SuperStripeVertexBuffer* vertexBuffer,
        SuperStripe* stripe,
        const ResStripeSuper* stripeData,
        const Emitter* emitter,
        f32 indexOffset = 0.0f );

    //------------------------------------------------------------------------------
    //! @brief  ストライプシステムで確保したワークサイズを取得します。
    //! @return 確保したワークサイズ
    //------------------------------------------------------------------------------
    u32 _GetWorkSize() const { return mStripeWorkSize; }
    //@}

private:
    nw::eft2::System*       mSystem;            //!< エフェクトシステム
    nw::eft2::Heap*         mHeap;              //!< ヒープ
    SuperStripe*            mStripeArray;       //!< ストライプ配列
    u32                     mStripeArrayNum;    //!< ストライプ配列数
    u32                     mStripeArrayIdx;    //!< ストライプ配列インデックス
    u32                     mStripeNum;         //!< ストライプ数
    BufferMode              mBufferMode;        //!< バッファモード
    u32                     mStripeWorkSize;    //!< ストライプワークサイズ
};

} // namespace eft2
} // namespace nw
