﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nw/eft/typeDef2.h>
#include <nw/eft/eft2_ShaderRes.h>
#include <nw/eft/eft2_UniformBlock.h>
#include <nw/eft/eft2_TextureSampler.h>

namespace nw   {
namespace eft2 {

struct Emitter;

//---------------------------------------------------------------------------
//! @brief  シェーダ
//---------------------------------------------------------------------------
struct Shader
{
public:
    //---------------------------------------------------------------------------
    //! @brief アトリビュート
    //---------------------------------------------------------------------------
    enum AttributeType
    {
        EFT_ATTR_IDX_POSITION          =  0,        //!< TBD
        EFT_ATTR_IDX_NORMAL            =  1,        //!< TBD
        EFT_ATTR_IDX_TANGENT           =  2,        //!< TBD
        EFT_ATTR_IDX_VERTEX_COLOR0     =  3,        //!< TBD
        EFT_ATTR_IDX_VERTEX_COLOR1     =  4,        //!< TBD
        EFT_ATTR_IDX_TEXTURE_COORD0    =  5,        //!< TBD
        EFT_ATTR_IDX_TEXTURE_COORD1    =  6,        //!< TBD
        EFT_ATTR_IDX_MAX               =  7,        //!< TBD
    };

    //---------------------------------------------------------------------------
    //! @brief パーティクル情報
    //---------------------------------------------------------------------------
    enum ParticleAttrType
    {
        EFT_PTCL_ATTR_IDX_LOCAL_POSITION    =  0,   //!< TBD
        EFT_PTCL_ATTR_IDX_LOCAL_VECTOR      =  1,   //!< TBD
        EFT_PTCL_ATTR_IDX_LOCAL_POS_DELTA   =  2,   //!< TBD
        EFT_PTCL_ATTR_IDX_SCALE             =  3,   //!< TBD
        EFT_PTCL_ATTR_IDX_RANDOM            =  4,   //!< TBD
        EFT_PTCL_ATTR_IDX_ROTATE            =  5,   //!< TBD
        EFT_PTCL_ATTR_IDX_COLOR0            =  6,   //!< TBD
        EFT_PTCL_ATTR_IDX_COLOR1            =  7,   //!< TBD
        EFT_PTCL_ATTR_IDX_EMTR_SRT_MTX0     =  8,   //!< TBD
        EFT_PTCL_ATTR_IDX_EMTR_SRT_MTX1     =  9,   //!< TBD
        EFT_PTCL_ATTR_IDX_EMTR_SRT_MTX2     =  10,  //!< TBD
        EFT_PTCL_ATTR_IDX_EMTR_RT_MTX0      =  11,  //!< TBD
        EFT_PTCL_ATTR_IDX_EMTR_RT_MTX1      =  12,  //!< TBD
        EFT_PTCL_ATTR_IDX_EMTR_RT_MTX2      =  13,  //!< TBD
        EFT_PTCL_ATTR_IDX_AMX               =  14,  //!< TBD
    };

    //---------------------------------------------------------------------------
    //! @brief ストリームアウト用アトリビュート配列
    //---------------------------------------------------------------------------
    enum StreamOutAttributeType
    {
        EFT_STREAM_ATTR_IDX_POSITION   =  0,        //!< TBD
        EFT_STREAM_ATTR_IDX_VECTOR     =  1,        //!< TBD
        EFT_STREAM_ATTR_IDX_MAX        =  2,        //!< TBD
    };

    //---------------------------------------------------------------------------
    //! @brief エミッタプラグイン用アトリビュート配列
    //---------------------------------------------------------------------------
    enum EmitterPluginAttributeType
    {
        EFT_EMTR_PLUGIN_ATTR_IDX_0     =  0,        //!< TBD
        EFT_EMTR_PLUGIN_ATTR_IDX_1     =  1,        //!< TBD
        EFT_EMTR_PLUGIN_ATTR_IDX_2     =  2,        //!< TBD
        EFT_EMTR_PLUGIN_ATTR_IDX_3     =  3,        //!< TBD
        EFT_EMTR_PLUGIN_ATTR_IDX_4     =  4,        //!< TBD
        EFT_EMTR_PLUGIN_ATTR_IDX_MAX   =  5,        //!< TBD
    };


#ifdef EFT_USE_UNIFORM_BLOCK
public:
    //---------------------------------------------------------------------------
    //! @brief ユニフォームブロックタイプ
    //---------------------------------------------------------------------------
    enum UniformBlockType
    {
        EFT_UBO_TYPE_VIEW                 =  0,         //!< eft_ParticleDeclaration.glsl の記述と合わせる
        EFT_UBO_TYPE_EMITTER_STATIC       =  1,         //!< TBD
        EFT_UBO_TYPE_EMITTER_DYNAMIC      =  2,         //!< TBD
        EFT_UBO_TYPE_EMITTER_FIELD        =  3,         //!< TBD
        EFT_UBO_TYPE_EMITTER_PLUGIN       =  4,         //!< TBD
        EFT_UBO_TYPE_RESERVED_PARAM       =  5,         //!< TBD
        EFT_UBO_TYPE_MAX                  =  6,         //!< TBD
    };
private:
    //---------------------------------------------------------------------------
    //! @brief ユニフォームブロックバインドポイント
    //---------------------------------------------------------------------------
    enum
    {
        EFT_UBO_BINDING_POINT_VIEW                 =  6,        //!< eft_ParticleDeclaration.glsl の記述と合わせる
        EFT_UBO_BINDING_POINT_EMITTER_STATIC       =  7,        //!< TBD
        EFT_UBO_BINDING_POINT_EMITTER_DYNAMIC      =  8,        //!< TBD
        EFT_UBO_BINDING_POINT_EMITTER_FIELD        =  9,        //!< TBD
        EFT_UBO_BINDING_POINT_EMITTER_PLUGIN       =  10,       //!< TBD
        EFT_UBO_BINDING_POINT_RESERVED_PARAM       =  11,       //!< TBD
        EFT_UBO_BINDING_POINT_CUSTOM_SHADER_UBO_0  =  12,       //!< TBD
        EFT_UBO_BINDING_POINT_CUSTOM_SHADER_UBO_1  =  13,       //!< TBD
        EFT_UBO_BINDING_POINT_CUSTOM_SHADER_UBO_2  =  14,       //!< TBD
        EFT_UBO_BINDING_POINT_CUSTOM_SHADER_UBO_3  =  15,       //!< TBD
    };

    //---------------------------------------------------------------------------
    //! @brief ユニフォームブロックが有効かどうかのビットフラグ
    //---------------------------------------------------------------------------
    enum
    {
        EFT_UBO_FLAG_VIEW                  =  0x01 << 0,        //!< TBD
        EFT_UBO_FLAG_EMITTER_STATIC        =  0x01 << 1,        //!< TBD
        EFT_UBO_FLAG_EMITTER_DYNAMIC       =  0x01 << 2,        //!< TBD
        EFT_UBO_FLAG_EMITTER_FIELD         =  0x01 << 3,        //!< TBD
        EFT_UBO_FLAG_EMITTER_PLUGIN        =  0x01 << 4,        //!< TBD
        EFT_UBO_FLAG_RESERVED_PARAM        =  0x01 << 5,        //!< TBD
        EFT_UBO_FLAG_CUSTOM_SHADER_UBO_0   =  0x01 << 6,        //!< TBD
        EFT_UBO_FLAG_CUSTOM_SHADER_UBO_1   =  0x01 << 7,        //!< TBD
        EFT_UBO_FLAG_CUSTOM_SHADER_UBO_2   =  0x01 << 8,        //!< TBD
        EFT_UBO_FLAG_CUSTOM_SHADER_UBO_3   =  0x01 << 9,        //!< TBD

        EFT_UBO_FLAG_VIEW_F                =  0x01 << 15,       //!< TBD
        EFT_UBO_FLAG_EMITTER_STATIC_F      =  0x01 << 16,       //!< TBD
        EFT_UBO_FLAG_EMITTER_DYNAMIC_F     =  0x01 << 17,       //!< TBD
        EFT_UBO_FLAG_EMITTER_FIELD_F       =  0x01 << 18,       //!< TBD
        EFT_UBO_FLAG_EMITTER_PLUGIN_F      =  0x01 << 19,       //!< TBD
        EFT_UBO_FLAG_RESERVED_PARAM_F      =  0x01 << 20,       //!< TBD
        EFT_UBO_FLAG_CUSTOM_SHADER_UBO_0_F =  0x01 << 21,       //!< TBD
        EFT_UBO_FLAG_CUSTOM_SHADER_UBO_1_F =  0x01 << 22,       //!< TBD
        EFT_UBO_FLAG_CUSTOM_SHADER_UBO_2_F =  0x01 << 23,       //!< TBD
        EFT_UBO_FLAG_CUSTOM_SHADER_UBO_3_F =  0x01 << 24,       //!< TBD
    };
#endif
#ifdef EFT_USE_UNIFORM_REGISTER
    //---------------------------------------------------------------------------
    //! @brief レジスタータイプ
    //---------------------------------------------------------------------------
    enum RegisterType
    {
        EFT_REGISTER_LOC_VIEW_MATRIX        =  0,   //!< TBD
        EFT_REGISTER_LOC_PROJ_MATRIX        =  1,   //!< TBD
        EFT_REGISTER_LOC_VIEW_PROJ_MATRIX   =  2,   //!< TBD
        EFT_REGISTER_LOC_BILLBOARD_MATRIX   =  3,   //!< TBD
        EFT_REGISTER_LOC_CAMERA_POS         =  4,   //!< TBD
        EFT_REGISTER_LOC_CAMERA_VEC         =  5,   //!< TBD
        EFT_REGISTER_LOC_VIEW_PARAM         =  6,   //!< TBD
        EFT_REGISTER_LOC_EMITTER_PARAM0     =  7,   //!< TBD
        EFT_REGISTER_LOC_EMITTER_PARAM1     =  8,   //!< TBD
        EFT_REGISTER_LOC_EMITTER_COLOR0     =  9,   //!< TBD
        EFT_REGISTER_LOC_EMITTER_COLOR1     = 10,   //!< TBD
        EFT_REGISTER_LOC_EMITTER_SRT_MATIRX = 11,   //!< TBD
        EFT_REGISTER_LOC_EMITTER_RT_MATIRX  = 12,   //!< TBD
        EFT_REGISTER_LOC_MAX                = 13,   //!< TBD
    };
#endif

public:
    //---------------------------------------------------------------------------
    //! @brief  無効化
    //---------------------------------------------------------------------------
    void Invalidate();

    //---------------------------------------------------------------------------
    //! @brief  シェーダリソースを取得します。
    //! @return TBD
    //---------------------------------------------------------------------------
    const ShaderResource* GetShaderResource() const { return &m_ShaderRes; }

    //---------------------------------------------------------------------------
    //! @brief  シェーダをコンパイル済かどうかを取得します。
    //! @return TBD
    //---------------------------------------------------------------------------
    bool IsCompiled() const { return m_IsInitialized; }

    //---------------------------------------------------------------------------
    //! @brief        ロケーションを初期化します。
    //---------------------------------------------------------------------------
    void InitializeLocation();

    //---------------------------------------------------------------------------
    //! @brief  シェーダの終了処理を行います。
    //! @param[in] heap TBD
    //---------------------------------------------------------------------------
    void Finalize( Heap* heap );

    //---------------------------------------------------------------------------
    //! @brief  シェーダを有効にします。
    //---------------------------------------------------------------------------
    void Bind() const { m_ShaderRes.Bind(); }

    //---------------------------------------------------------------------------
    //! @brief      インスタンシングを有効にします。
    //!
    //!             Windows/OpenGL版のみ機能します。
    //---------------------------------------------------------------------------
    void EnableInstanced();

    //---------------------------------------------------------------------------
    //! @brief      インスタンシングを無効にします。
    //!
    //!             Windows/OpenGL版のみ機能します。
    //---------------------------------------------------------------------------
    void DisableInstanced();

    //----------------------------------------
    //! @name ユニフォームブロック
    //@{
#ifdef EFT_USE_UNIFORM_BLOCK
    //---------------------------------------------------------------------------
    //! @brief  ユニフォームブロックを初期化します。
    //---------------------------------------------------------------------------
    static void InitializeUniformBlock();

    //---------------------------------------------------------------------------
    //! @brief  ユニフォームブロックとシェーダの関連付けをします。
    //---------------------------------------------------------------------------
    void BindingUniformBlock();

    //---------------------------------------------------------------------------
    //! @brief  ビューユニフォームブロックを終了処理をします。
    //---------------------------------------------------------------------------
    static void FinalizeUniformBlock();

    //---------------------------------------------------------------------------
    //! @brief  ビューユニフォームブロックをバインドします。
    //! @param[in] viewUniformBlockBuffer TBD
    //---------------------------------------------------------------------------
    static void BindViewUniformBlock( const UniformBlockBuffer* viewUniformBlockBuffer );

    //---------------------------------------------------------------------------
    //! @brief  各エミッタ共通ユニフォームブロックをバインドします。
    //! @param[in] id               ID
    //! @param[in] uniformBlock     ユニフォームブロックへのポインタ
    //! @param[in] uniformBlockSize ユニフォームブロックサイズ
    //! @param[in] endianSwap       エンディアンスワップの有無
    //---------------------------------------------------------------------------
    static void BindCommonCustomShaderUniformBlock( CustomShaderUboID id, void* uniformBlock, u32 uniformBlockSize, bool endianSwap );

    //---------------------------------------------------------------------------
    //  ユニフォームブロックをバインドします。
    //! @param[in] uniformBuffer TBD
    //---------------------------------------------------------------------------
    void BindUniformBlock( UniformBlockType type, const UniformBlockBuffer* uniformBuffer );

    //---------------------------------------------------------------------------
    //! @brief  ユーザーパラメータユニフォームブロックをバインドします。
    //!
    //!         本メソッドは"sysCustomShaderReservedUniformBlockParam"に対してユニフォームブロックの設定を行います。
    //! @param[in] emitter TBD
    //! @return TBD
    //---------------------------------------------------------------------------
    bool BindReservedCustomShaderUniformBlock( Emitter* emitter );

    //---------------------------------------------------------------------------
    //! @brief  カスタムシェーダユニフォームブロックをバインドします。
    //! @param[in] id TBD
    //! @param[in] uniformBlock TBD
    //! @param[in] uniformBlockSize TBD
    //! @param[in] swapEndian TBD
    //---------------------------------------------------------------------------
    void BindCustomShaderUniformBlock( CustomShaderUboID id, void* uniformBlock, u32 uniformBlockSize, bool swapEndian = true );

    //---------------------------------------------------------------------------
    //! @brief  エミッタプラグインユニフォームブロックをバインドします。
    //! @param[in] uniformBuffer TBD
    //---------------------------------------------------------------------------
    void BindEmitterPluginUniformBlock( const UniformBlockBuffer* uniformBuffer );

    //---------------------------------------------------------------------------
    //! @brief  バイナリ化されているエミッタプラグインデータをユニフォームブロックにバインドします。
    //! @param[in] emitter TBD
    //! @return TBD
    //---------------------------------------------------------------------------
    bool BindEmitterPluginUniformBlockAuto( Emitter* emitter );
#endif
    //@}

    //----------------------------------------
    //! @name ユニフォームレジスタ
    //@{
#ifdef EFT_USE_UNIFORM_REGISTER
    //---------------------------------------------------------------------------
    //! @brief ユニフォームレジスタロケーションを取得します。
    //! @param[in] Type TBD
    //! @return TBD
    //---------------------------------------------------------------------------
    u32 GetRegisterLocation( const RegisterType Type ) const
    {
        return m_Register[ Type ];
    }
#endif
    //@}

    //----------------------------------------
    //! @name アトリビュート取得
    //@{

    //---------------------------------------------------------------------------
    //! @brief アトリビュートを取得します。
    //! @param[in] Type TBD
    //! @return TBD
    //---------------------------------------------------------------------------
    u32 GetAttribute( const AttributeType Type ) const { return m_Attr[ Type ]; }

    //---------------------------------------------------------------------------
    //! @brief アトリビュートを取得します。
    //! @param[in] Type TBD
    //! @return TBD
    //---------------------------------------------------------------------------
    u32 GetParticleAttribute( const ParticleAttrType Type ) const { return m_PtclAttr[ Type ]; }

#ifndef EFT_DEGRADATION_SPEC
    //---------------------------------------------------------------------------
    //! @brief アトリビュートを取得します。
    //! @param[in] Type TBD
    //! @return TBD
    //---------------------------------------------------------------------------
    u32 GetStreamOutAttribute( const StreamOutAttributeType Type ) const { return m_StreamAttr[ Type ]; }

    //---------------------------------------------------------------------------
    //! @brief  エミッタプラグイン アトリビュートを取得します。
    //! @param[in] Type TBD
    //! @return TBD
    //---------------------------------------------------------------------------
    u32 GetEmitterPluginAttribute( const EmitterPluginAttributeType Type ) const { return m_EmtrPluginAttr[ Type ]; }
#endif

    //@}

    //----------------------------------------
    //! @name テクスチャロケーション
    //@{

    //---------------------------------------------------------------------------
    //! @brief  フラグメント テクスチャサンプラロケーションを取得します。
    //! @param[in] slot TBD
    //! @return TBD
    //---------------------------------------------------------------------------
    u32 GetFragmentTextureSamplerLocation( TextureSlot slot ) const { return m_TexSmpLoc[slot].GetFragmentSmpLocation(); }

    //---------------------------------------------------------------------------
    //! @brief  頂点 テクスチャサンプラロケーションを取得します。
    //! @param[in] slot TBD
    //! @return TBD
    //---------------------------------------------------------------------------
    u32 GetVertexTextureSamplerLocation( TextureSlot slot ) const { return m_TexSmpLoc[slot].GetVertexSmpLocation(); }

    //---------------------------------------------------------------------------
    //! @brief  フラグメント フレームバッファテクスチャサンプラロケーションを取得します。
    //! @return TBD
    //---------------------------------------------------------------------------
    u32 GetFrameBufferTextureFragmentSamplerLocation() const { return m_FrameBuferTexSmpLoc.GetFragmentSmpLocation(); }

    //---------------------------------------------------------------------------
    //! @brief  頂点フレームバッファテクスチャサンプラロケーションを取得します。
    //! @return TBD
    //---------------------------------------------------------------------------
    u32 GetFrameBufferTextureVertexSamplerLocation() const { return m_FrameBuferTexSmpLoc.GetVertexSmpLocation(); }

    //---------------------------------------------------------------------------
    //! @brief  フラグメント デプスバッファテクスチャサンプラロケーションを取得します。
    //! @return TBD
    //---------------------------------------------------------------------------
    u32 GetDepthBufferTextureFragmentSamplerLocation() const { return m_DepthBuferTexSmpLoc.GetFragmentSmpLocation(); }

    //---------------------------------------------------------------------------
    //! @brief  頂点 デプスバッファテクスチャサンプラロケーションを取得します。
    //! @return TBD
    //---------------------------------------------------------------------------
    u32 GetDepthBufferTextureVertexSamplerLocation() const { return m_DepthBuferTexSmpLoc.GetVertexSmpLocation(); }

    //---------------------------------------------------------------------------
    //! @brief  フラグメント カールノイズテクスチャサンプラロケーションを取得します。
    //! @return TBD
    //---------------------------------------------------------------------------
    u32 GetCurlNoiseTextureFragmentSamplerLocation() const { return m_CurlNoiseTexSmpLoc.GetFragmentSmpLocation(); }

    //---------------------------------------------------------------------------
    //! @brief  頂点 カールノイズテクスチャサンプラロケーションを取得します。
    //! @return TBD
    //---------------------------------------------------------------------------
    u32 GetCurlNoiseTextureVertexSamplerLocation() const { return m_CurlNoiseTexSmpLoc.GetVertexSmpLocation(); }

    //---------------------------------------------------------------------------
    //! @brief  テクスチャサンプラロケーションを取得します。
    //! @param[in] type TBD
    //! @return TBD
    //---------------------------------------------------------------------------
    u32 GetCustomTextureFragmentSamplerLocation( CustomShaderTextureType type ) const
    {
        return m_CtmTextureLoc[type].GetFragmentSmpLocation();
    }
    //---------------------------------------------------------------------------
    //! @brief  テクスチャサンプラロケーションを取得します。
    //! @param[in] type TBD
    //! @return TBD
    //---------------------------------------------------------------------------
    u32 GetCustomTextureVertexSamplerLocation( CustomShaderTextureType type ) const
    {
        return m_CtmTextureLoc[type].GetVertexSmpLocation();
    }

    //@}


#ifdef EFT_OGL
    //----------------------------------------
    //! @name OpenGL版 専用メソッド
    //@{

    //---------------------------------------------------------------------------
    //! @brief  シェーダクラスを初期化します。
    //          シェーダのコンパイルは行われません、
    //! @param[in] emitterResSet TBD
    //! @param[in] addShaderDef TBD
    //---------------------------------------------------------------------------
    void Initialize( const EmitterResource* emitterResSet, const char* addShaderDef );

    //---------------------------------------------------------------------------
    //! @brief  シェーダをコンパイルします。
    //! @param[in] emitterResSet TBD
    //! @param[in] addCompileDef TBD
    //! @return TBD
    //---------------------------------------------------------------------------
    bool CompileShader();

    //---------------------------------------------------------------------------
    //! @brief  ストリームアウトシェーダを初期化します。
    //---------------------------------------------------------------------------
    void InitializeStreamOutShader();

    //---------------------------------------------------------------------------
    //! @brief  エミッタリソースを取得します。
    //! @return TBD
    //---------------------------------------------------------------------------
    const EmitterResource* GetEmitterRes() const
    {
        return m_EmitterResSet;
    }

    //---------------------------------------------------------------------------
    //! @brief  追加シェーダコンパイル定義を取得します。
    //! @return TBD
    //---------------------------------------------------------------------------
    const char* GetAddCompileDef() const
    {
        return m_AddShaderDef;
    }

    //---------------------------------------------------------------------------
    //! @brief  頂点位置アトリビュートを取得します。
    //! @return TBD
    //---------------------------------------------------------------------------
    u32 GetProgramID() { return m_ShaderRes.GetProgramID(); }
    u32 GetProgramId() { return m_ShaderRes.GetProgramID(); }

    //@}
#endif

#if EFT_GX2
    //----------------------------------------
    //! @name GX2版 専用メソッド
    //@{

    //---------------------------------------------------------------------------
    //! @brief  シェーダを初期化します。
    //! @param[in] heap TBD
    //! @param[in] binary TBD
    //! @param[in] vertexIdx TBD
    //! @param[in] fragIdx TBD
    //---------------------------------------------------------------------------
    void InitializeShader( Heap* heap, void* binary, u32 vertexIdx, u32 fragIdx );

    //------------------------------------------------------------------------------
    //! @brief  シェーダをセットアップする
    //! @param[in] heap TBD
    //! @return TBD
    //------------------------------------------------------------------------------
    void SetupShader( Heap* heap )
    {
        m_ShaderRes.SetupShader( heap );
    }

    //------------------------------------------------------------------------------
    //! @brief  シェーダインデックスを設定する
    //! @param[in] vertexIdx TBD
    //! @param[in] fragIdx TBD
    //! @return TBD
    //------------------------------------------------------------------------------
    void SetShaderId( u32 vertexIdx, u32 fragIdx)
    {
        m_VertexIdx = vertexIdx;
        m_FragIdx   = fragIdx;
    }

    //------------------------------------------------------------------------------
    //! @brief  頂点シェーダインデックスを取得する
    //! @return TBD
    //------------------------------------------------------------------------------
    u32  GetVertexShaderId() const { return m_VertexIdx; }

    //------------------------------------------------------------------------------
    //! @brief  フラグメントシェーダインデックスを所得する
    //! @return TBD
    //------------------------------------------------------------------------------
    u32  GetFragmentShaderId() const { return m_FragIdx; }

    //@}
#endif

private:
    ShaderResource          m_ShaderRes;                                        //!< シェーダリソース
    const EmitterResource*  m_EmitterResSet;                                    //!< エミッタリソース
    const char*             m_AddShaderDef;                                     //!< 追加シェーダ定義
    bool                    m_IsInitialized;                                    //!< 初期化済かどうか

    u32                     m_Attr[EFT_ATTR_IDX_MAX];                           //!< アトリビュート

#ifndef EFT_DEGRADATION_SPEC
    u32                     m_StreamAttr[EFT_STREAM_ATTR_IDX_MAX];              //!< StreamOut用アトリビュート
    u32                     m_EmtrPluginAttr[EFT_EMTR_PLUGIN_ATTR_IDX_MAX];     //!< エミッタプラグイン用アトリビュート
#endif
    u32                     m_PtclAttr[EFT_PTCL_ATTR_IDX_AMX];                  //!< インスタンスアトリビュート

#ifdef EFT_USE_UNIFORM_BLOCK
    u32                     m_UboEnableFlag;
    static UniformBlockID   g_UniformBlock[EFT_UBO_TYPE_MAX];                   //!< ビューユニフォームブロック
    static UniformBlockID   g_CtmShaderUboID[EFT_CUSTOM_SHADER_UBO_MAX];        //!< カスタムユニフォームブロック
#endif
#ifdef EFT_USE_UNIFORM_REGISTER
    u32                     m_Register[EFT_REGISTER_LOC_MAX];
#endif

    TextureSmplerLoc        m_CtmTextureLoc[EFT_CUSTOM_SHADER_TEXTURE_MAX];     //!< カスタムテクスチャ
    TextureSmplerLoc        m_TexSmpLoc[EFT_TEXTURE_SLOT_MAX];                  //!< テクスチャ0-2 サンプラロケーション
    TextureSmplerLoc        m_FrameBuferTexSmpLoc;                              //!< フレームバッファテクスチャ サンプラロケーション
    TextureSmplerLoc        m_DepthBuferTexSmpLoc;                              //!< デプスバッファテクスチャ   サンプラロケーション
    TextureSmplerLoc        m_CurlNoiseTexSmpLoc;                               //!< カールノイズテクスチャ   サンプラロケーション

#if EFT_GX2
    u32                     m_VertexIdx;                                        //!< 頂点シェーダインデックス
    u32                     m_FragIdx;                                          //!< フラグメントシェーダインデックス
#endif
};


} // namespace nw
} // namespace eft2
