﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nw/eft/typeDef2.h>

namespace nw    {
namespace eft2  {

//---------------------------------------------------
//! @brief メモリ操作ユーティリティー
//---------------------------------------------------
class MemUtil
{
public:
    //---------------------------------------------------------------------------
    //! @brief        メモリのある範囲を一定の値で埋める。
    //!
    //! @param[in]    buf           メモリ位置。
    //! @param[in]    val           埋める値。
    //! @param[in]    size          サイズ。
    //! @return                     先頭アドレス
    //---------------------------------------------------------------------------
    inline static void* Fill( void* buf, u8 val, u32 size )
    {
        return memset(buf,val,size);
    }

    //---------------------------------------------------------------------------
    //! @brief        メモリのある範囲をゼロで埋める。
    //!
    //! @param[in]    buf           メモリ位置。
    //! @param[in]    size          サイズ。
    //! @return                     先頭アドレス
    //---------------------------------------------------------------------------
    inline static void* FillZero( void* buf, u32 size )
    {
        return memset(buf, 0, size);
    }

    //---------------------------------------------------------------------------
    //! @brief        メモリのある範囲を別の範囲へコピー（高速だが範囲が重なるとエラー）。
    //!
    //! @param[in]    dst           コピー先。
    //! @param[in]    src           コピー元。
    //! @param[in]    size          サイズ。
    //! @return                     コピー先のアドレス
    //---------------------------------------------------------------------------
    inline static void* Copy( void* dst, const void* src, u32 size )
    {
#if EFT_IS_CAFE
        return OSBlockMove(dst, src, size, false);
#else
        return memcpy(dst, src, size);
#endif

    }

    //---------------------------------------------------------------------------
    //! @brief        キャッシュをフラッシュする。
    //!
    //! @param[in]    ptr           フラッシュ先のアドレス。
    //! @param[in]    size          フラッシュサイズ。
    //---------------------------------------------------------------------------
    inline static void FlushCache( void* ptr, u32 size )
    {
#if EFT_IS_CAFE
//      EFT_ASSERT( (((u32)ptr)%32) == 0 );
//      EFT_ASSERT( ((size)%32) == 0 );
        DCFlushRange( ptr, size );
#else
        EFT_UNUSED_VARIABLE( ptr );
        EFT_UNUSED_VARIABLE( size );
#endif
    }

    //---------------------------------------------------------------------------
    //! @brief        指定のメモリ範囲をキャッシュに載せて0を書き込む。
    //!
    //! @param[in]    ptr           先頭のアドレス。
    //! @param[in]    size          サイズ。
    //---------------------------------------------------------------------------
    inline static void ZeroRange( void* ptr, u32 size )
    {
#if EFT_IS_CAFE
//        EFT_ASSERT( (((u32)ptr)%32) == 0 );
//        EFT_ASSERT( ((size)%32) == 0 );
        DCZeroRange( ptr, size );
#endif
#if EFT_IS_WIN
        EFT_UNUSED_VARIABLE( ptr );
        EFT_UNUSED_VARIABLE( size );
//        EFT_ASSERT( (((u32)ptr)%32) == 0 );
//        EFT_ASSERT( ((size)%32) == 0 );
#endif
    }
};

//---------------------------------------------------------------------------
//! @briefprivate   64bitと解釈して、二つの変数を同時にコピーする。
//---------------------------------------------------------------------------
#define EFT_F32_DOUBLE_ASSIGNMENT( dst, src )   \
    *( reinterpret_cast< f64* >( &dst ) ) = *( reinterpret_cast< f64* >( const_cast< f32* >( &src ) ) )

//---------------------------------------------------------------------------
//! @briefprivate   nw::math::VEC2としてコピー。
//!                 代入2回が1回で済む。
//---------------------------------------------------------------------------
#define EFT_F32_VEC2_COPY( dst, src )                   \
    EFT_F32_DOUBLE_ASSIGNMENT( (dst)->x, (src)->x );

//---------------------------------------------------------------------------
//! @briefprivate   nw::math::VEC3としてコピー。
//!                 代入3回が2回で済む。
//---------------------------------------------------------------------------
#define EFT_F32_VEC3_COPY( dst, src )                   \
    EFT_F32_DOUBLE_ASSIGNMENT( (dst)->x, (src)->x );    \
    (dst)->z = (src)->z;

//---------------------------------------------------------------------------
//! @briefprivate   nw::math::VEC4としてコピー。
//!                 代入4回が2回で済む。
//---------------------------------------------------------------------------
#define EFT_F32_VEC4_COPY( dst, src )                   \
    EFT_F32_DOUBLE_ASSIGNMENT( (dst)->x, (src)->x );    \
    EFT_F32_DOUBLE_ASSIGNMENT( (dst)->z, (src)->z );

} // namespace eftut
} // namespace nw
