﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nw/eft/eft2_Enum.h>
#include <nw/eft/eft2_Texture.h>
#include <nw/eft/eft2_TextureSampler.h>
#include <nw/eft/eft2_Primitive.h>
#include <nw/eft/eft2_UniformBlock.h>


namespace nw    {
namespace eft2  {

class ShaderManager;
struct BinaryData;

//------------------------------------------------------------------------------
//! @brief  フィールドのユニフォームブロック
//------------------------------------------------------------------------------
struct EmitterFieldUniformBlock
{
    nw::math::VEC4      randomParam;            //!< ランダム幅XYZ / 適用間隔
    nw::math::VEC4      randomParam1;           //!< フィールドランダム ランダム周期比率（ 0 / 1 / 2 / 3 ）
    nw::math::VEC4      randomParam2;           //!< フィールドランダム 合成重み（固定）（ 0 / 1 / 2 / 3 ）
    nw::math::VEC4      randomParam3;           //!< 位相の統一 / 詳細設定有効 / / 空気抵抗の有無
    nw::math::VEC4      randomParam4;           //!< 位相の変化速度　/ 位相の分布 / /

    nw::math::VEC4      randomFe1Param;         //!< ランダム幅XYZ / 適用間隔

    nw::math::VEC4      posAddParam;            //!< 位置に加算 加算値XYZ / ワールド座標系で処理するかどうか

    nw::math::VEC4      magnetParam;            //!< 磁力 位置XYZ / 磁力
    nw::math::VEC4      magnetParam1;           //!< 磁力 エミッタ追従 / / /

    nw::math::VEC4      convergenceParam;       //!< 収束 位置XYZ / 収束力
    nw::math::VEC4      convergenceParam1;      //!< 収束 エミッタ追従 / / /

    nw::math::VEC4      spinParam;              //!< スピン 回転速度 / 拡散速度 / 軸 /

    nw::math::VEC4      collisionParam;         //!< シンプルコリジョン タイプ / 反射率 / 摩擦 / 反射座標

    nw::math::VEC4      curlNoiseParam;         //!< カールノイズ パワーXYZ / スケール
    nw::math::VEC4      curlNoiseParam1;        //!< カールノイズ スピードXYZ / オフセット
    nw::math::VEC4      curlNoiseParam2;        //!< カールノイズ ノイズのオフセットランダム /

    nw::math::VEC4      customFieldParam0;      //!< カスタムフィールド パラメータ0
    nw::math::VEC4      customFieldParam1;      //!< カスタムフィールド パラメータ1
};

//------------------------------------------------------------------------------
//! @brief デフォルトのカスタムアクションパラメータ
//!
//!        ユーザーデータ定義を使用しない場合の、カスタムアクションに渡されるデフォルトのパラメータです。
//------------------------------------------------------------------------------
struct ResCustomActionData
{
    u32         bit;                //!< ビット
    union
    {
        s32     iValue[ 6 ];        //!< 整数
        int     integerValue[ 6 ];  //!< 整数
    };
    union
    {
        f32     fValue[ 8 ];        //!< 浮動小数点数
        float   floatValue[ 8 ];    //!< 浮動小数点数
    };

    //--------------------------------------------------------------------------
    //! @brief          エンディアンをフリップします.
    //! @param[in] size フリップするサイズ
    //--------------------------------------------------------------------------
    void FlipEndian( u32 size )
    {
#ifdef  EFT_ENDIAN_LITTLE
        u32* temp = reinterpret_cast<u32*>( &bit );
        u32 cnt = size/4;
        for( u32 i=0; i<cnt; i++ )
        {
            nw::eft2::EndianUtil::Flip( temp );
            temp++;
        }
//        nw::eft2::EndianUtil::Flip( &bit );
//        nw::eft2::EndianUtil::FlipArray( 8, fValue );
//        nw::eft2::EndianUtil::FlipArray( 6, iValue );
#endif//EFT_ENDIAN_LITTLE
    }
};

//------------------------------------------------------------------------------
//! @brief デフォルトのエミッタ拡張パラメータ
//!
//!        ユーザーデータ定義を使用しない場合の、エミッタ拡張パラメータに渡されるデータです。
//------------------------------------------------------------------------------
struct ResEmitterExtParamData
{
    u32 bit;            //!< ビット
    int iValue[6];      //!< 整数
    f32 fValue[8];      //!< 浮動小数点数

    //--------------------------------------------------------------------------
    //! @brief          エンディアンをフリップします.
    //! @param[in] size フリップするサイズ
    //--------------------------------------------------------------------------
    void FlipEndian( u32 size )
    {
#ifdef  EFT_ENDIAN_LITTLE
        u32* temp = reinterpret_cast<u32*>( &bit );
        u32 cnt = size/4;
        for( u32 i=0; i<cnt; i++ )
        {
            nw::eft2::EndianUtil::Flip( temp );
            temp++;
        }
#endif //EFT_ENDIAN_LITTLE
    }
};

//------------------------------------------------------------------------------
//! @brief エミッタ管理データ
//!
//!       １つのエミッタに関するデータが集約された構造体です。
//------------------------------------------------------------------------------
struct EmitterResource
{
    u32                         emitterIndex;                               //!< エミッタインデックス( 親から見て何番目のエミッタか)
    union
    {
        ResEmitter*             emitterData;                                //!< エミッタリソース
        ResEmitter*             m_pResEmitter;
    };
    StaticUniformBlockBuffer    emitterStaticUbo;                           //!< エミッタ静的ユニフォームブロック
    EmitterStaticUniformBlock*  resEmitterStaticUniformBlock;               //!< エミッタ静的ユニフォームブロック( CPUエミッタの場合は実体ではなくコピー )
    StaticUniformBlockBuffer    fieldStaticUbo;                             //!< フィールド静的ユニフォームブロック
    EmitterFieldUniformBlock*   fieldUniformBlock;                          //!< フィールドユニフォームブロック
    TextureResource*            texture[EFT_TEXTURE_SLOT_MAX];              //!< テクスチャバイナリ
    TextureSampler              textureSampler[EFT_TEXTURE_SLOT_MAX];       //!< テクスチャサンプラー
    union
    {
        Primitive*              primitive;                                  //!< 利用するプリミティブ
        Primitive*              m_pPrimitive;                               //!< 利用するプリミティブ
    };
    Primitive*                  volumePrimitive;                            //!< エミッタ形状で利用するプリミティブ
    u32                         childEmitterResCount;                       //!< チャイルドリソース数
    EmitterResource*            childEmitterResSet[EFT_EMITTER_INSET_NUM];  //!< チャイルドエミッタリソースセット
    EmitterResource*            next;                                       //!< 次のエミッタリソースセット
    nw::math::MTX34             resMatrixSRT;                               //!< リソース内で解決されるSRTマトリクス
    nw::math::MTX34             resMatrixRT;                                //!< リソース内で解決されるRTマトリクス
    u32                         shaderIndex;                                //!< シェーダインデックス
    u32                         drawPathShader1;                            //!< 描画パスシェーダ1
    u32                         drawPathShader2;                            //!< 描画パスシェーダ2
    bool                        visibility;                                 //!< 表示/非表示
    ResFieldRandom*             fieldRandomData;                            //!< フィールドランダムデータ
    ResFieldRandomFe1*          fieldRandomDataFe1;                         //!< 旧フィールドランダムデータ
    ResFieldMagnet*             fieldMagnetData;                            //!< フィールド磁力データ
    ResFieldSpinData*           fieldSpinData;                              //!< フィールドスピンデータ
    ResFieldCollisionData*      fieldCollisionData;                         //!< フィールドコリジョンデータ
    ResFieldConvergenceData*    fieldConvergenceData;                       //!< フィールド収束データ
    ResFieldPosAddData*         fieldPosAddData;                            //!< フィールド位置加算データ
    ResFieldCustom*             fieldCustomData;                            //!< カスタムフィールドのデータ
    ResFieldCurlNoiseData*      fieldCurlNoiseData;                         //!< フィールドカールノイズデータ
    ResAnimEmitterKeyParamSet*  emitterAnimArray[EFT_EMITTER_ANIM_MAX];     //!< エミッタアニメ配列
    bool                        isUseEmitterAnim;                           //!< エミッタ時間アニメを含むか
    bool                        isEmitterSRTAnim;                           //!< エミッタSRTアニメを含むか。
    void*                       customActionParam;                          //!< カスタムアクションパラメータ
    void*                       customShaderParam;                          //!< カスタムシェーダパラメータ
    void*                       customDataParam;                            //!< エミッタ拡張パラメータ
    u32                         customShaderParamSize;                      //!< カスタムシェーダパラメータサイズ
    nw::math::VEC3              rotateInit;                                 //!< 回転初期値
    u32                         emitterPluginIndex;                         //!< エミッタプラグインインデックス
    void*                       emitterPluginData;                          //!< エミッタプラグインデータ

    //---------------------------------------------------------------------------
    //! @brief        emitterDataを元に他メンバーを更新します。
    //---------------------------------------------------------------------------
    void Update();
};


//------------------------------------------------------------------------------
//! @brief エミッタセット管理データ
//!
//!       １つのエミッタセットに関するデータが集約された構造体です。
//------------------------------------------------------------------------------
struct EmitterSetResource
{
    u32                 emitterNum;                                 //!< 親エミッタ数
    u32                 emitterAllNum;                              //!< 総エミッタ数(子を含む)
    BinaryData*         emSetData;                                  //!< エミッタセットバイナリデータ
    ResEmitterSet*      emitterSetRes;                              //!< エミッタセットリソース
    EmitterResource*    emitterResSet;                              //!< エミッタバイナリノード
    bool                visibility;                                 //!< 表示/非表示
    bool                isLoopEffect;                               //!< ループエフェクトかどうか
    bool                isLifeInfinity;                             //!< 無限寿命エフェクトかどうか
    ShaderManager*      shaderManager;                              //!< シェーダマネージャリソース
    u32                 textureNum;                                 //!< テクスチャ数
    TextureResource*    textureArray;                               //!< テクスチャリソース
    u32                 primitiveNum;                               //!< プリミティブ数
    Primitive*          primitiveArray;                             //!< プリミティブリソース

    //------------------------------------------------------------------------------
    //! @brief テクスチャを検索する
    //! @param[in] guid テクスチャのGUID
    //! @return         見つかった場合、TextureResouceへのポインタ
    //------------------------------------------------------------------------------
    TextureResource* GetTexture( u32 guid )
    {
        for ( u32 i = 0; i < textureNum; i++ )
        {
            if ( textureArray[i].GetGlobalID() == guid )
            {
                return &textureArray[i];
            }
        }

        return NULL;
    }

    //------------------------------------------------------------------------------
    //! @brief プリミティブを検索する
    //! @param[in] guid プリミティブのGUID
    //! @return         見つかった場合、Primitiveへのポインタ
    //------------------------------------------------------------------------------
    Primitive* GetPrimitive( u32 guid )
    {
        for ( u32 i = 0; i < primitiveNum; i++ )
        {
            if ( primitiveArray[i].GetGlobalID() == guid )
            {
                return &primitiveArray[i];
            }
        }

        return NULL;
    }
};


//---------------------------------------------------------------------------
//! @brief  エミッタ 動的ユニフォームブロック
//---------------------------------------------------------------------------
struct EmitterDynamicUniformBlock
{
    nw::math::VEC4      emitterColor0;                              //!< エミッタカラー0(セット*アニメカラー0*フェードカラー)
    nw::math::VEC4      emitterColor1;                              //!< エミッタカラー1(セット*アニメカラー1*フェードカラー)
    nw::math::VEC4      emitterParam0;                              //!< エミッタパラメータ( time / emScaleX / emScaleY / frameRate )
    nw::math::VEC4      emitterParam1;                              //!< エミッタパラメータ( fadeOutAlpha / empty / empty / empty )
    nw::math::MTX44     emitterMatrix;                              //!< エミッタマトリクス
    nw::math::MTX44     emitterMatrixRT;                            //!< エミッタマトリクス
};

} // namespace eft2
} // namespace nw
