﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nw/eft/typeDef2.h>

namespace nw   {
namespace eft2 {

class Heap;

//---------------------------------------------------
//! @brief  エフェクトシステム生成時に設定するコンフィグです。
//!
//!         エフェクト用ヒープの設定(SetEffectHeap)は必須になります。
//!         必要に応じて、適切にバッファサイズを設定してください。
//!         詳しくは、エフェクト チュートリアル ドキュメントをご覧ください。
//!         SetEffectHeap / SetEffectDynamicHeap でのヒープの指定はシステムの起動に必須になります。
//---------------------------------------------------
class Config
{
public:
    //---------------------------------------------------------------------------
    //! @brief  コンフィグ　デフォルト値。
    //---------------------------------------------------------------------------
    enum ConfigDefault
    {
        EFT_DEFAULT_EMITTER_NUM                     = 64,           //!< デフォルトのエミッタ数(64)
        EFT_DEFAULT_EMITTER_SET_NUM                 = 32,           //!< デフォルトのエミッタセット数(32)
        EFT_DEFAULT_RESOURCE_NUM                    = 8,            //!< デフォルトのリソース数(8)
        EFT_DEFAULT_RESOURCE_VIEWER_RESERVE_NUM     = 32,           //!< デフォルトのビューア利用リソース数(16)
        EFT_DEFAULT_STRIPE_NUM                      = 32,           //!< デフォルトのストライプ数(32)
        EFT_DEFAULT_TEMPORARY_BUFFER_SIZE           = 64 * 1024,    //!< デフォルトのバッファサイズ(64*1024)
        EFT_DEFAULT_PTCL_SORT_BUFFER_COUNT          = 1024,         //!< デフォルトのソートバッファサイズ(Ptcl数)
        EFT_DEFAULT_FORCE_4BYTE = 0x80000000
    };

    //---------------------------------------------------------------------------
    //! @brief  コンストラクタ。
    //!         各パラメータの初期値が設定されます。
    //---------------------------------------------------------------------------
    Config() : m_Heap                    ( NULL )
             , m_DynamicHeap             ( NULL )
             , m_EmitterNum              ( EFT_DEFAULT_EMITTER_NUM )
             , m_EmitterSetNum           ( EFT_DEFAULT_EMITTER_SET_NUM )
             , m_ResourceNum             ( EFT_DEFAULT_RESOURCE_NUM )
             , m_StripeNum               ( EFT_DEFAULT_STRIPE_NUM )
             , m_SuperStripeNum          ( EFT_DEFAULT_STRIPE_NUM )
             , m_SuppressLog             ( false )
             , m_UseMultiCore            ( true )
             , m_IsEnableTripleBuffer    ( false )
             , m_IsUniformBlockCpuChacheFlush ( true )
             , m_IsUniformBlockGpuChacheFlush ( false )
             , m_StreamOutBatchProcess( false )
    {
        m_TemporaryBufferSize[EFT_CPU_CORE_0]   = EFT_DEFAULT_TEMPORARY_BUFFER_SIZE;
        m_TemporaryBufferSize[EFT_CPU_CORE_1]   = EFT_DEFAULT_TEMPORARY_BUFFER_SIZE;
        m_TemporaryBufferSize[EFT_CPU_CORE_2]   = EFT_DEFAULT_TEMPORARY_BUFFER_SIZE;
        m_SortBufferCount[EFT_CPU_CORE_0]       = EFT_DEFAULT_PTCL_SORT_BUFFER_COUNT;
        m_SortBufferCount[EFT_CPU_CORE_1]       = EFT_DEFAULT_PTCL_SORT_BUFFER_COUNT;
        m_SortBufferCount[EFT_CPU_CORE_2]       = EFT_DEFAULT_PTCL_SORT_BUFFER_COUNT;
    }

    //---------------------------------------------------------------------------
    //! @brief  デストラクタです。
    //---------------------------------------------------------------------------
    virtual ~Config() {}

    //---------------------------------------------------------------------------
    //! @brief      メモリ確保先ヒープを設定します。システムの生成とリソース設定関数からメモリ確保関数が呼ばれます。
    //! @param[in]  heap    確保先ヒープ。
    //---------------------------------------------------------------------------
    void SetEffectHeap( Heap *heap ){ m_Heap = heap; }

    //---------------------------------------------------------------------------
    //! @brief      メモリ確保先ヒープを取得します。
    //! @return     ヒープへのインスタンス。
    //---------------------------------------------------------------------------
    Heap* GetEffectHeap() const { return m_Heap; }

    //---------------------------------------------------------------------------
    //! @brief      動的メモリ確保先ヒープを設定します。
    //!
    //!             エミッタ生成時にGPUで挙動行う場合にメモリ確保関数が呼ばれます。
    //!             確保されるメモリ量は、　sizeof( PtclAttributeBuffer ) * 放出するパーティクル数 になります。
    //! @param[in]  heap    確保先ヒープ。
    //---------------------------------------------------------------------------
    void SetEffectDynamicHeap( Heap *heap ){ m_DynamicHeap = heap; }

    //---------------------------------------------------------------------------
    //! @brief      動的メモリ確保先ヒープを取得します。
    //! @return     ヒープへのインスタンス。
    //---------------------------------------------------------------------------
    Heap* GetEffectDynamicHeap() const { return m_DynamicHeap; }

    //---------------------------------------------------------------------------
    //! @brief          最大同時計算エミッタ数を設定します。
    //! @param[in] num  最大同時計算エミッタ数
    //---------------------------------------------------------------------------
    void SetEmitterNum( u32 num ) { m_EmitterNum = num; }

    //---------------------------------------------------------------------------
    //! @brief  最大同時計算エミッタ数を取得します。
    //! @return 最大同時計算エミッタ数
    //---------------------------------------------------------------------------
    u32 GetEmitterNum() const { return m_EmitterNum; }

    //---------------------------------------------------------------------------
    //! @brief          最大同時計算エミッタセット数を設定します。
    //! @param[in] num  最大同時計算エミッタセット数
    //---------------------------------------------------------------------------
    void SetEmitterSetNum( u32 num ) { m_EmitterSetNum = num; }

    //---------------------------------------------------------------------------
    //! @brief  最大同時計算エミッタセット数を取得します。
    //! @return 最大同時計算エミッタセット数
    //---------------------------------------------------------------------------
    u32 GetEmitterSetNum() const { return m_EmitterSetNum; }

    //---------------------------------------------------------------------------
    //! @brief          最大リソース数を設定します。
    //! @param[in] num  最大リソース数
    //---------------------------------------------------------------------------
    void SetResourceNum( u32 num ){ m_ResourceNum = num; }

    //---------------------------------------------------------------------------
    //! @brief  最大リソース数を取得します。
    //! @return 最大リソース数
    //---------------------------------------------------------------------------
    u32 GetResourceNum() const { return m_ResourceNum; }

    //---------------------------------------------------------------------------
    //! @brief          最大ストライプ数を設定します。
    //! @param[in] num  最大ストライプ数
    //---------------------------------------------------------------------------
    void SetStripeNum( u32 num ) { m_StripeNum = num; }

    //---------------------------------------------------------------------------
    //! @brief  最大ストライプ数を取得します。
    //! @return 最大ストライプ数
    //---------------------------------------------------------------------------
    u32 GetStripeNum() const { return m_StripeNum; }

    //---------------------------------------------------------------------------
    //! @brief          最大スーパーストライプ数を設定します。
    //! @param[in] num  最大スーパーストライプ数
    //---------------------------------------------------------------------------
    void SetSuperStripeNum( u32 num ) { m_SuperStripeNum = num; }

    //---------------------------------------------------------------------------
    //! @brief  最大スーパーストライプ数を取得します。
    //! @return 最大スーパーストライプ数
    //---------------------------------------------------------------------------
    u32 GetSuperStripeNum() const { return m_SuperStripeNum; }

    //---------------------------------------------------------------------------
    //! @brief          コア毎にテンポラリバッファサイズを設定します。
    //! @param[in] core コアID
    //! @param[in] size バッファサイズ
    //---------------------------------------------------------------------------
    void SetTemporaryBufferSize( CpuCore core, u32 size )
    {
        m_TemporaryBufferSize[core] = size;
    }

    //---------------------------------------------------------------------------
    //! @brief  ストリームアウト処理を一括で行うモードを有効にする。
    //---------------------------------------------------------------------------
    void SetEnableStreamOutBatchProcess()
    {
        m_StreamOutBatchProcess = true;
    }

    //---------------------------------------------------------------------------
    //! @brief          テンポラリバッファサイズを設定します。
    //! @param[in] size テンポラリバッファサイズ
    //---------------------------------------------------------------------------
    void SetTemporaryBufferSize( u32 size )
    {
        m_TemporaryBufferSize[EFT_CPU_CORE_0] = size;
        m_TemporaryBufferSize[EFT_CPU_CORE_1] = size;
        m_TemporaryBufferSize[EFT_CPU_CORE_2] = size;
    }

    //---------------------------------------------------------------------------
    //! @brief              パーティクルソート用バッファサイズを設定します。
    //! @param[in] count    パーティクルソート用バッファサイズ
    //---------------------------------------------------------------------------
    void SetParticleSortBufferCount( u32 count )
    {
        m_SortBufferCount[EFT_CPU_CORE_0] = count;
        m_SortBufferCount[EFT_CPU_CORE_1] = count;
        m_SortBufferCount[EFT_CPU_CORE_2] = count;
    }

    //---------------------------------------------------------------------------
    //! @brief          テンポラリバッファサイズを設定します。(廃止予定)
    //! @param[in] size テンポラリバッファサイズ
    //---------------------------------------------------------------------------
    void SetDoubleBufferSize( u32 size ){ return SetTemporaryBufferSize( size ); }

    //---------------------------------------------------------------------------
    //! @brief  テンポラリバッファサイズを取得します。
    //! @return テンポラリバッファサイズ
    //---------------------------------------------------------------------------
    u32 GetTemporaryBufferSize() const { return m_TemporaryBufferSize[EFT_CPU_CORE_1]; }

    //---------------------------------------------------------------------------
    //! @brief  パーティクルソート用のバッファ長（ソート可能なPtclの個数）を取得します。
    //! @return パーティクルソート用のバッファ長
    //---------------------------------------------------------------------------
    u32 GetParticleSortBufferCount() const { return m_SortBufferCount[EFT_CPU_CORE_1]; }

    //---------------------------------------------------------------------------
    //! @brief  ダブルバッファサイズを取得します。(廃止予定)
    //! @return ダブルバッファサイズ
    //---------------------------------------------------------------------------
    u32 GetDoubleBufferSize() const { return GetTemporaryBufferSize(); }

    //---------------------------------------------------------------------------
    //! @brief          ランタイムログ出力を抑制します。
    //! @param[in] flag ランタイムログ出力を押さえるかどうかを示すbool値
    //---------------------------------------------------------------------------
    void SetSuppressionLog( bool flag ) { m_SuppressLog = flag; }

    //---------------------------------------------------------------------------
    //! @brief  ランタイムログ出力を抑制するかどうかを取得します。
    //! @return ランタイムログ出力を押さえるかどうかを示すbool値
    //---------------------------------------------------------------------------
    bool IsSuppressionLog() const { return m_SuppressLog; }

    //---------------------------------------------------------------------------
    //! @brief  マルチコア動作させるかどうか取得します。
    //! @return マルチコア動作させるかどうか
    //---------------------------------------------------------------------------
    bool IsUseMultiCoreProcess() const { return m_UseMultiCore; }

    //---------------------------------------------------------------------------
    //! @brief                      トリプルバッファを有効にする場合に呼び出してください。
    //! @param[in] isUseTripeBuffer トリプルバッファが有効かどうか
    //---------------------------------------------------------------------------
    void SetEnableTripleBuffer( bool isUseTripeBuffer ) { m_IsEnableTripleBuffer = isUseTripeBuffer; }

    //---------------------------------------------------------------------------
    //! @brief  トリプルバッファを利用するかどうか取得します。
    //! @return トリプルバッファが有効かどうか
    //---------------------------------------------------------------------------
    bool IsEnableTripleBuffer() const { return m_IsEnableTripleBuffer; }

    //---------------------------------------------------------------------------
    //! @brief          ユニフォームブロックのCpuキャッシュフラッシュを行うかどうか。
    //! @param[in] flag 使用する場合true
    //---------------------------------------------------------------------------
    void SetUniformBlockCpuChacheFlush( bool flag ) { m_IsUniformBlockCpuChacheFlush = flag; }

    //---------------------------------------------------------------------------
    //! @brief  ユニフォームブロックのCpuキャッシュフラッシュを行うかどうか。
    //! @return 使用する場合true
    //---------------------------------------------------------------------------
    bool IsUniformBlockCpuChacheFlush() const { return m_IsUniformBlockCpuChacheFlush; }

    //---------------------------------------------------------------------------
    //! @brief          ユニフォームブロックのCpuキャッシュフラッシュを行うかどうか。
    //! @param[in] flag 使用する場合true
    //---------------------------------------------------------------------------
    void SetUniformBlockGpuChacheFlush( bool flag ) { m_IsUniformBlockGpuChacheFlush = flag; }

    //---------------------------------------------------------------------------
    //! @brief  ユニフォームブロックのGpuキャッシュフラッシュを行うかどうか。
    //! @return 使用する場合true
    //---------------------------------------------------------------------------
    bool IsUniformBlockGpuChacheFlush() const { return m_IsUniformBlockGpuChacheFlush; }

    //---------------------------------------------------------------------------
    //! @brief  ストリームアウト処理を一括で行うモードが有効かどうか。
    //! @return 使用する場合true
    //---------------------------------------------------------------------------
    bool IsEnableStreamOutBatchProcess() const { return m_StreamOutBatchProcess; }

protected:
    Heap*           m_Heap;                                     //!< メモリ確保先ヒープ
    Heap*           m_DynamicHeap;                              //!< 動的メモリ確保先ヒープ
    u32             m_EmitterNum;                               //!< 最大同時計算エミッタ数
    u32             m_EmitterSetNum;                            //!< 最大同時計算エミッタセット数
    u32             m_StripeNum;                                //!< 最大同時計算ストライプ数
    u32             m_SuperStripeNum;                           //!< 最大同時計算スーパーストライプ数
    u32             m_ResourceNum;                              //!< 最大リソース数
    u32             m_TemporaryBufferSize[EFT_CPU_CORE_MAX];    //!< テンポラリバッファのサイズ
    u32             m_SortBufferCount[EFT_CPU_CORE_MAX];        //!< ソート用バッファのサイズ（パーティクル数）
    bool            m_SuppressLog;                              //!< ランタイムログ出力を抑制します。
    bool            m_UseMultiCore;                             //!< マルチコア動作するかどうか。
    bool            m_IsEnableTripleBuffer;                     //!< トリプルバッファが有効かどうか。
    bool            m_IsUniformBlockCpuChacheFlush;             //!< ユニフォームブロックのCpuキャッシュフラッシュを行うかどうか
    bool            m_IsUniformBlockGpuChacheFlush;             //!< ユニフォームブロックのGpuキャッシュフラッシュを行うかどうか
    bool            m_StreamOutBatchProcess;                    //!< ストリームアウト処理を一括で行うかどうか
};

} // namespace eft2
} // namespace nw
