﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_DW_WINDOW_MANAGER_H_
#define NW_DW_WINDOW_MANAGER_H_

#include <nw/types.h>
#include <nw/math.h>
#include <nw/dev/dev_Pad.h>
#include <nw/dw/system/dw_Pointer.h>
#include <nw/dw/system/dw_UIRenderer.h>
#include <nw/dw/window/dw_Window.h>
#include <nw/dw/window/dw_WindowMenu.h>
#include <nw/ut/ut_Memory.h>

#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
#include <nw/config/undef_windows.h>
#endif

namespace nw {
namespace internal {
namespace dw {

class WindowManager
{
private:
    static const f32 ALPHA_MIN;
    static const f32 ALPHA_MAX;
    static const f32 WINDOW_MARGIN;

public:
    WindowManager();
    ~WindowManager();

public:
    void Initialize(f32 screenWidth, f32 screenHeight, nw::ut::IAllocator* allocator);
    void Finalize();

    // ウィンドウの作成・破棄
    bool CreateWindow( Window* window, f32 x, f32 y );
    bool CreateWindow( Window* window, f32 x, f32 y, f32 width, f32 height );
    void DestroyWindow( Window* window );

    // ウィンドウの更新
    void UpdateInputs(const nw::internal::dw::Inputs& inputs);
    void Update(UIRenderer& renderer);

    // ウィンドウの描画
    void Draw( UIRenderer& renderer );

    const nw::math::Vector2& GetScreenSize() const;

    // ポインター位置
    void SetPointerPosition( const nw::math::Vector2& pos );
    const nw::math::Vector2& GetPointerPosition() const { return m_Pointer.GetPosition(); }

    // アルファ
    void SetAlpha( f32 alpha ) { m_MasterAlpha = alpha; }
    f32 GetAlpha() const { return m_MasterAlpha; }

    // ウィンドウコントロールフラグ
    void SetWindowControlFlag( bool enable ) { m_EnableWindowControl = enable; }
    bool GetWindowControlFlag() const { return m_EnableWindowControl; }
    bool IsPressWindowControlKey( const nw::dev::Pad& pad ) const;

    // ウィンドウ描画フラグ
    void SetWindowVisibleFlag( bool flag ) { m_VisibleFlag = flag; }
    bool GetWindowVisibleFlag() const { return m_VisibleFlag; }

    // アクティブウィンドウ
    Window* GetActiveWindow() { return m_ActiveWindow; }
    void SetActiveWindow( Window* window );

    void SetForegroundWindow( Window* window );

    // ウィンドウリスト
    WindowList& GetWindowList() { return m_WindowList; }
    const WindowList& GetWindowList() const { return m_WindowList; }

    // ウィンドウオーダー
    WindowZOrderList& detail_GetWindowZOrderList() { return m_WindowDrawList; }
    const WindowZOrderList& detail_GetWindowZOrderList() const { return m_WindowDrawList; }
    void detail_SortWindowList();
    void detail_SortWindowZOrderList();

private:
    void UpdatePointer(const nw::internal::dw::Inputs& inputs);

    void UpdatePointerWindow(const Inputs& inputs);
    void UpdateActiveWindow();
    void UpdateHoldWindow( const nw::dev::Pad& pad );
    void UpdateOrder();

    void EnterWindowListMode( const nw::math::Vector2& screenSize );
    void LeaveWindowListMode();

    Window* GetPointerWindow();
    bool IsPointerOverWindow( const Window* window, bool onTitleBar ) const;

    Window* GetWindowByIndex( int index );
    int GetWindowIndex( Window* window );

    void ChangeMasterAlpha( f32 add );
    void ChangeWindowSize( Window* window, const nw::math::Vector2& size );

    void OnUpdatePad( const nw::dev::Pad& pad );
    void OnUpdatePadForWindowControl(const nw::internal::dw::Inputs& inputs);

private:
    bool m_EnableWindowControl;
    bool m_WindowMenuMode;
    bool m_VisibleFlag;

    detail::Pointer m_Pointer;

    bool m_WindowControlMode;

    Window* m_HoldWindow;
    bool m_HoldWithA;

    Window* m_PointerWindow;
    Window* m_ActiveWindow;
    bool m_ActiveWindowToTop;

    f32 m_MasterAlpha;
    WindowList m_WindowList;
    WindowZOrderList m_WindowDrawList;

    detail::WindowMenu m_WindowMenu;

    WindowContext m_WindowContext;
};

} // namespace nw::internal::dw
} // namespace nw::internal
} // namespace nw

#endif // NW_DW_WINDOW_MANAGER_H_
