﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef DW_PARAMETER_LIST_ITEM_H_
#define DW_PARAMETER_LIST_ITEM_H_

#include <nw/dw/control/dw_FixedUIElementList.h>
#include <nw/dw/control/dw_FixedDockPanel.h>
#include <nw/dw/control/dw_FixedLabel.h>

namespace nw {
namespace internal {
namespace dw {

//---------------------------------------------------------------------------
//! @brief     パラメータリストアイテムの基底クラスです。
//---------------------------------------------------------------------------
class ParameterListItem : public nw::internal::dw::UIElement
{
public:
    //---------------------------------------------------------------------------
    //! @brief     コンストラクタです。
    //---------------------------------------------------------------------------
    ParameterListItem();

    //---------------------------------------------------------------------------
    //! @brief     デストラクタです。
    //---------------------------------------------------------------------------
    virtual ~ParameterListItem() {}

    //---------------------------------------------------------------------------
    //! @brief        ラベルの文字列を取得します。
    //!
    //! @return       ラベルの文字列を返します。
    //---------------------------------------------------------------------------
    const char* GetLabelText() const;

    //---------------------------------------------------------------------------
    //! @brief        ラベルの文字列を設定します。
    //!
    //! @param[in]    value     ラベルの文字列です。
    //---------------------------------------------------------------------------
    void SetLabelText(const char* value);

    //---------------------------------------------------------------------------
    //! @brief        ラベルの幅を設定します。
    //!
    //! @param[in]    value     ラベルの幅です。
    //---------------------------------------------------------------------------
    void SetLabelWidth(f32 value);

    //---------------------------------------------------------------------------
    //! @brief        値をリセットします。
    //---------------------------------------------------------------------------
    virtual void ResetValue() = 0;

    //---------------------------------------------------------------------------
    //! @brief        値がループするかどうかを指定します。
    //!
    //! @param[in]    value     ループさせる場合、 true を指定します。
    //---------------------------------------------------------------------------
    void SetValueLoop(bool value) { m_ValueLoop = value; }

    //---------------------------------------------------------------------------
    //! @brief        値がループするかどうかを取得します。
    //!
    //! @return       値がループする設定の場合、 true を返します。
    //---------------------------------------------------------------------------
    bool GetValueLoop() const { return m_ValueLoop; }


protected:
    //! @brief 次の値にします。
    virtual void NextValue(bool isLargeChange) = 0;
    //! @brief 前の値にします。
    virtual void PreviousValue(bool isLargeChange) = 0;
    //! @brief 値を表示します。
    virtual void PrintValue(char* pText, u32 bufferLength) = 0;
    //! @brief 利用可能かどうかを取得します。
    virtual bool IsAvailable() const = 0;

    //! @brief 入力を更新します。
    /*override*/ bool OnUpdateFocusedInput(const nw::internal::dw::Inputs& inputs);
    //! @brief UI要素サイズ計算の準備を行います。
    /*override*/ void OnPrepareMeasure();

    nw::internal::dw::FixedUIElementList<1> m_Contents;
    nw::internal::dw::FixedDockPanel<3>     m_Container;
    nw::internal::dw::FixedLabel<32>        m_Label;
    nw::internal::dw::FixedLabel<256>       m_ValueText;
    bool                          m_ValueLoop;
};



//---------------------------------------------------------------------------
//! @brief     ハンドラに渡す TModel をもつパラメータリストアイテムです。
//---------------------------------------------------------------------------
template<class TModel>
class ModelParameterListItem : public ParameterListItem
{
public:
    //---------------------------------------------------------------------------
    //! @brief     コンストラクタです。
    //---------------------------------------------------------------------------
    ModelParameterListItem()
        : m_pModel( NULL ) {}

    //---------------------------------------------------------------------------
    //! @brief     デストラクタです。
    //---------------------------------------------------------------------------
    virtual ~ModelParameterListItem() {}


    //---------------------------------------------------------------------------
    //! @brief        ハンドラに渡す TModel を取得します。
    //!
    //! @return       TModel オブジェクトへのポインタを返します。
    //---------------------------------------------------------------------------
    TModel* GetModel() const { return m_pModel; }

    //---------------------------------------------------------------------------
    //! @brief        ハンドラに渡す TModel を設定します。
    //!
    //! @param[in]    value     TModel オブジェクトへのポインタです。
    //---------------------------------------------------------------------------
    void SetModel(TModel* value) { m_pModel = value; }


protected:
    //! @brief 利用可能かどうかを取得します。
    virtual bool IsAvailable() const { return m_pModel != NULL; }

private:
    TModel* m_pModel;
};

} /* namespace dw */
} /* namespace internal */
} /* namespace nw */

#endif //  DW_PARAMETER_LIST_ITEM_H_
