﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

namespace nw {
namespace internal {
namespace dw {

//---------------------------------------------------------------------------
template<class TModel, typename TValue>
NumericParameterListItem<TModel, TValue>::NumericParameterListItem()
 : m_DefaultValue( static_cast<TValue>(0) ),
   m_ValueChange( static_cast<TValue>(1) ),
   m_ValueLargeChange( static_cast<TValue>(10) ),
   m_MinimumValue( static_cast<TValue>(0) ),
   m_MaximumValue( static_cast<TValue>(100) ),
   m_pGetValueHandler(NULL),
   m_pSetValueHandler(NULL),
   m_pPrintValueHandler(NULL)
{
}

//---------------------------------------------------------------------------
template<class TModel, typename TValue>
void
NumericParameterListItem<TModel, TValue>::Initialize(
    GetValueHandler pGetValueHandler,
    SetValueHandler pSetValueHandler,
    PrintValueHandler pPrintValueHandler
)
{
    NW_ASSERTMSG( m_pGetValueHandler == NULL, "NumericParameterListItem is already initialized." );
    NW_ASSERTMSG( m_pSetValueHandler == NULL, "NumericParameterListItem is already initialized." );
    NW_ASSERTMSG( m_pPrintValueHandler == NULL, "NumericParameterListItem is already initialized." );
    NW_ASSERT( pGetValueHandler != NULL );
    NW_ASSERT( pSetValueHandler != NULL );
    NW_ASSERT( pPrintValueHandler != NULL );

    m_pGetValueHandler = pGetValueHandler;
    m_pSetValueHandler = pSetValueHandler;
    m_pPrintValueHandler = pPrintValueHandler;
}

//---------------------------------------------------------------------------
template<class TModel, typename TValue>
TValue
NumericParameterListItem<TModel, TValue>::GetValue() const
{
    NW_ASSERTMSG(m_pGetValueHandler != NULL, "NumericParameterListItem is not initialized.");
    return (*m_pGetValueHandler)( GetModel() );
}

//---------------------------------------------------------------------------
template<class TModel, typename TValue>
void
NumericParameterListItem<TModel, TValue>::SetValue(TValue value)
{
    NW_ASSERTMSG(m_pSetValueHandler != NULL, "NumericParameterListItem is not initialized.");
    (*m_pSetValueHandler)( GetModel(), value );
}

//---------------------------------------------------------------------------
template<class TModel, typename TValue>
void
NumericParameterListItem<TModel, TValue>::ResetValue()
{
    SetValue( m_DefaultValue );
}

//---------------------------------------------------------------------------
template<class TModel, typename TValue>
TValue
NumericParameterListItem<TModel, TValue>::GetDefaultValue() const
{
    return m_DefaultValue;
}

//---------------------------------------------------------------------------
template<class TModel, typename TValue>
void
NumericParameterListItem<TModel, TValue>::SetDefaultValue(TValue value)
{
    m_DefaultValue = value;
}

//---------------------------------------------------------------------------
template<class TModel, typename TValue>
TValue
NumericParameterListItem<TModel, TValue>::GetValueChange() const
{
    return m_ValueChange;
}

//---------------------------------------------------------------------------
template<class TModel, typename TValue>
void
NumericParameterListItem<TModel, TValue>::SetValueChange(TValue value)
{
    m_ValueChange = value;
}

//---------------------------------------------------------------------------
template<class TModel, typename TValue>
TValue
NumericParameterListItem<TModel, TValue>::GetValueLargeChange() const
{
    return m_ValueLargeChange;
}

//---------------------------------------------------------------------------
template<class TModel, typename TValue>
void
NumericParameterListItem<TModel, TValue>::SetValueLargeChange(TValue value)
{
    m_ValueLargeChange = value;
}

//---------------------------------------------------------------------------
template<class TModel, typename TValue>
TValue
NumericParameterListItem<TModel, TValue>::GetMinimumValue() const
{
    return m_MinimumValue;
}

//---------------------------------------------------------------------------
template<class TModel, typename TValue>
void
NumericParameterListItem<TModel, TValue>::SetMinimumValue(TValue value)
{
    m_MinimumValue = value;
}

//---------------------------------------------------------------------------
template<class TModel, typename TValue>
TValue
NumericParameterListItem<TModel, TValue>::GetMaximumValue() const
{
    return m_MaximumValue;
}

//---------------------------------------------------------------------------
template<class TModel, typename TValue>
void
NumericParameterListItem<TModel, TValue>::SetMaximumValue(TValue value)
{
    m_MaximumValue = value;
}

//---------------------------------------------------------------------------
template<class TModel, typename TValue>
void
NumericParameterListItem<TModel, TValue>::NextValue(bool isLargeChange)
{
    TValue newValue = GetValue() + (isLargeChange ? m_ValueLargeChange : m_ValueChange);

    if ( newValue < m_MinimumValue || newValue > m_MaximumValue )
    {
        if ( m_ValueLoop )
        {
            newValue = m_MinimumValue;
        }
        else
        {
            newValue = m_MaximumValue;
        }
    }

    SetValue( nw::ut::Min(newValue, m_MaximumValue) );
}

//---------------------------------------------------------------------------
template<class TModel, typename TValue>
void
NumericParameterListItem<TModel, TValue>::PreviousValue(bool isLargeChange)
{
    TValue newValue = GetValue() - (isLargeChange ? m_ValueLargeChange : m_ValueChange);

    if ( newValue < m_MinimumValue || newValue > m_MaximumValue )
    {
        if ( m_ValueLoop )
        {
            newValue = m_MaximumValue;
        }
        else
        {
            newValue = m_MinimumValue;
        }
    }

    SetValue( nw::ut::Max(newValue, m_MinimumValue) );
}

//---------------------------------------------------------------------------
template<class TModel, typename TValue>
void
NumericParameterListItem<TModel, TValue>::PrintValue(char* pText, u32 bufferLength)
{
    NW_ASSERTMSG(m_pPrintValueHandler != NULL, "NumericParameterListItem is not initialized.");
    (*m_pPrintValueHandler)( GetModel(), pText, bufferLength, GetValue() );
}

} /* namespace dw */
} /* namespace internal */
} /* namespace nw */
