﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_DW_FIXED_TAB_CONTROL_H_
#define NW_DW_FIXED_TAB_CONTROL_H_

#include <nw/dw/control/dw_TabControl.h>

namespace nw {
namespace internal {
namespace dw {

//---------------------------------------------------------------------------
//! @brief     タブ数固定のタブコントロールクラスです。
//---------------------------------------------------------------------------
template<u32 TTabCount>
class FixedTabControl : public TabControl
{
public:
    //---------------------------------------------------------------------------
    //! @brief     コンストラクタです。
    //---------------------------------------------------------------------------
    FixedTabControl();


    //---------------------------------------------------------------------------
    //! @brief     初期化処理を行います。
    //---------------------------------------------------------------------------
    void Initialize();


    //---------------------------------------------------------------------------
    //! @brief     タブアイテムを追加します。
    //!
    //! @param[in] pItem 追加するタブアイテムです。
    //! @param[in] tabName 追加するタブアイテムの名前です。
    //---------------------------------------------------------------------------
    /*override*/ void AddTabItem( UIElement* pItem, const char* tabName );

    //---------------------------------------------------------------------------
    //! @brief     タブアイテムの数を取得します。
    //!
    //! @return    タブアイテムの数です。
    //---------------------------------------------------------------------------
    /*override*/ const s32 GetTabCount() const
    {
        return m_Tabs.GetContents().GetCount();
    }

    //---------------------------------------------------------------------------
    //! @brief     タブが境界をもつかどうかを設定します。
    //!
    //! @param[in] value 境界を持つかどうかです。
    //---------------------------------------------------------------------------
    /*override*/ void SetTabIsBordered( bool value );

    //---------------------------------------------------------------------------
    //! @brief     選択中タブの背景色を設定します。
    //!
    //! @param[in] value 設定する色です。
    //---------------------------------------------------------------------------
    /*override*/ void SetTabSelectedBackgroundColor(bool value );

    //---------------------------------------------------------------------------
    //! @brief     タブの背景色を設定します。
    //!
    //! @param[in] value 設定する色です。
    //---------------------------------------------------------------------------
    /*override*/ void SetTabBackgroundColor( bool value );

    //---------------------------------------------------------------------------
    //! @brief     選択中タブの文字色を設定します。
    //!
    //! @param[in] value 設定する色です。
    //---------------------------------------------------------------------------
    /*override*/ void SetTabSelectedTextColor( bool value );

    //---------------------------------------------------------------------------
    //! @brief     タブの文字色を設定します。
    //!
    //! @param[in] value 設定する色です。
    //---------------------------------------------------------------------------
    /*override*/ void SetTabTextColor( bool value );


protected:
    //---------------------------------------------------------------------------
    //! @brief       UI要素を更新します。
    //!
    //! @details     UI要素を更新します。
    //!              特別な更新処理を行う場合は、この関数をオーバーライドしてください。
    //!
    //! @param[in]   context  UIElementTreeContext を指定します。
    //---------------------------------------------------------------------------
    /*override*/ void OnUpdate( const UIElementTreeContext& context );

    //---------------------------------------------------------------------------
    //! @brief       指定タブを選択します。
    //!
    //! @details     指定タブを選択します。
    //!
    //! @param[in]   index  選択対象のインデックスを指定します。
    //---------------------------------------------------------------------------
    /*override*/ void SelectTab( s32 index );

    //! @brief タブの背景色を設定します。
    void SetTabsBackgroundColor();

    //! @brief タブの文字色を設定します。
    void SetTabsTextColor();


    FixedUIElementList<1>      m_Contents;
    FixedStackPanel<2>         m_Container;

    UIElement*                 m_TabItem[TTabCount];
    FixedUIElementList<1>      m_RootContents;

    FixedStackPanel<TTabCount> m_Tabs;
    FixedLabel<32>             m_TabLabel[TTabCount];


#if defined(NW_DEBUG) || defined(NW_DEVELOP)
public:
    virtual const char* ToString() const
    {
        return "FixedTabControl";
    }
#endif
};

} /* namespace dw */
} /* namespace internal */
} /* namespace nw */

#include <nw/dw/control/dw_FixedTabControl.hpp>

#endif //  NW_DW_FIXED_TAB_CONTROL_H_
