﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_DEV_JOY_PAD_DEVICE_WIN_H_
#define NW_DEV_JOY_PAD_DEVICE_WIN_H_

#include <nw/types.h>
#include <nw/ut/ut_Inlines.h>

namespace nw
{
namespace dev
{

//---------------------------------------------------------------------------
//! @brief   PC 用ジョイパッドのデバイス情報を取得するクラスです。
//!
//! @details :category     入力デバイス
//---------------------------------------------------------------------------
class JoyPadDeviceWin
{
protected:
    static const u32 JOY_INFO_EX_SIZE = 52;
    static const u32 JOY_CAPS_SIZE = 728;
    static const u32 JOY_ALIGN = 4;

public:
    //! @param 扱うことのできるジョイパッドの最大数です。
    enum
    {
        NUM_MAX = 4     //!< 扱うことのできるジョイパッドの最大数です。
    };

    //! @brief ジョイパッドの状態を表します。
    struct PadStatus
    {
        u32  info[ NW_UT_ROUNDUP( JOY_INFO_EX_SIZE, JOY_ALIGN ) / JOY_ALIGN ];   //!< 操作情報です。
        u32  caps[ NW_UT_ROUNDUP( JOY_CAPS_SIZE, JOY_ALIGN ) / JOY_ALIGN ];      //!< 性能情報です。
        bool isEnable;  //!< 有効か否かです。
        u32  id;        //!< 状態を取得するときに用いる ID です。

        PadStatus() : isEnable(false), id(0) {}
    };


    //---------------------------------------------------------------------------
    //! @brief        インスタンスを取得します。
    //!
    //! @return       シングルトンのインスタンスを返します。
    //---------------------------------------------------------------------------
    static JoyPadDeviceWin* GetInstance()
    {
        static JoyPadDeviceWin s_Instance;
        return &s_Instance;
    }


    //---------------------------------------------------------------------------
    //! @brief        初期化処理です。
    //---------------------------------------------------------------------------
    void Initialize();

    //---------------------------------------------------------------------------
    //! @brief        終了処理です。
    //---------------------------------------------------------------------------
    void Finalize() {}

    //---------------------------------------------------------------------------
    //! @brief        デバイスを更新します。
    //---------------------------------------------------------------------------
    void Update();

    //---------------------------------------------------------------------------
    //! @brief        パッドの状態の配列を取得します。
    //!
    //! @param[in]    joyPadNumber 取得するジョイパッドの番号です。
    //!
    //! @return       パッドの状態の配列を返します。
    //---------------------------------------------------------------------------
    const PadStatus& GetPads( int joyPadNumber )
    {
        NW_ASSERT_MINMAX( joyPadNumber, 0, m_PadNum );
        return m_Pads[ joyPadNumber ];
    }

    //---------------------------------------------------------------------------
    //! @brief        パッドの数を取得します。
    //!
    //! @return       パッドの数を返します。
    //---------------------------------------------------------------------------
    s32 GetPadNum() const { return m_PadNum; }


protected:
    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //---------------------------------------------------------------------------
    JoyPadDeviceWin()
        : m_PadNum( 0 )
    {
        if (s_pInstance != NULL)
        {
            NW_FATAL_ERROR("nw::dev::JoyPadDeviceWin instance already exists.");
        }
        s_pInstance = this;
    }


    PadStatus m_Pads[ NUM_MAX ];
    s32 m_PadNum;

    static JoyPadDeviceWin* s_pInstance;             //!< シングルトンオブジェクトへのポインタです。
};

} // namespace dev
} // namespace nw

#endif // NW_DEV_JOY_PAD_DEVICE_WIN_H_
