﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_DEV_PROFILE_H_
#define NW_DEV_PROFILE_H_

#include <nw/dev/dev_Config.h>

#ifdef NW_PLATFORM_CTR
//#define NW_PERFORMANCE_COUNTER_ENABLED
#endif
//#define NW_PROFILE_DUMP_HIERARCHICAL_REPORTS_ENABLED

#include <nw/types.h>
#include <nw/ut/ut_Memory.h>
#include <nw/ut/ut_Preprocessor.h>
#include <nw/ut/ut_Inlines.h>
#include <nw/ut/os/ut_Time.h>

#include <nw/dev/dev_Utility.h>
#include <nw/dev/dev_Memory.h>

#if defined( NW_DEV_ENABLED )

#include <set>
#include <vector>
#include <limits>

#if defined( NW_PROFILE_ENABLED )

// NW_DEV_ENABLED と NW_PROFILE_ENABLED がともに定義されていたらプロファイルマクロを有効にします。
#define NW_INITIALIZE_PROFILE(maxReport, allocator) nw::dev::ProfileCenter::Initialize(maxReport, allocator)
#define NW_FINALIZE_PROFILE(allocator) nw::dev::ProfileCenter::Finalize(allocator)
#define NW_PROFILE(name) nw::dev::AutoProfile NW_profile(name, nw::dev::ProfileCenter::GetProfileManager())
#define NW_START_PROFILE(name) nw::dev::ProfileCenter::GetProfileManager()->StartProfile(name)
#define NW_STOP_PROFILE() nw::dev::ProfileCenter::GetProfileManager()->StopProfile()
#define NW_DUMP_PROFILE() nw::dev::ProfileCenter::GetProfileManager()->DumpReports()
#define NW_CLEAR_PROFILE() nw::dev::ProfileCenter::GetProfileManager()->ClearReports()

#else

#define NW_INITIALIZE_PROFILE(maxReport, allocator) (void)0
#define NW_FINALIZE_PROFILE(allocator) (void)0
#define NW_PROFILE(name) (void)0
#define NW_START_PROFILE(name) (void)0
#define NW_STOP_PROFILE() (void)0
#define NW_DUMP_PROFILE() (void)0
#define NW_CLEAR_PROFILE() (void)0

#endif // NW_PROFILE_ENABLED

#endif // NW_DEV_ENABLED

namespace nw
{
namespace dev
{

#if defined(NW_DEV_ENABLED) && defined(NW_PROFILE_ENABLED)

class AutoProfile;

//! @brief ログの出力形式の定義です。
enum ExportFormat
{
    CSV, //!< コンマ区切り形式です。
    CHART //!< チャート形式です。
};

//! @brief ログの出力精度の定義です。
enum Accuracy
{
    MILI_SECOND, //!< ミリセコンドです。
    MICRO_SECOND //!< マイクロセコンドです。
};

//! @brief 計測を行う項目です。
enum Mode
{
    NONE = 0x0, //!< 計測を行いません。
    TIMER = 0x1 << 0, //!< タイマーです。
    I_CACHE_MISS = 0x1 << 1, //!< 命令キャッシュミスです。
    D_CACHE_READ_MISS = 0x1 << 2, //!< データキャッシュミスです。
    D_CACHE_WRITE_MISS = 0x1 << 3 //!< データキャッシュミスです。
};

//---------------------------------------------------------------------------
//! @brief プロファイル結果です。
//---------------------------------------------------------------------------
struct ProfileReport
{
    //! @brief コンストラクタです。
    ProfileReport()
    : name(NULL)
    , elapsedTime(0)
    , callCount(0)
    , maxElapsedTime(std::numeric_limits<s64>::min())
    , minElapsedTime(std::numeric_limits<s64>::max())
#ifdef NW_PERFORMANCE_COUNTER_ENABLED
    , iCacheMiss(0)
    , dCacheReadMiss(0)
    , dCacheWriteMiss(0)
#endif
    {}

    void Combine(const ProfileReport& report);
    void DumpReport(u32 mode, ExportFormat exportFormat, float accuracy);
    static float TickToTime(ut::Tick tick, float accuracy)
    {
        return static_cast<float>(tick.ToTimeSpan().GetNanoSeconds()) / accuracy;
    }

    const char* name; //!< プロファイルの名前です。
    ut::Tick elapsedTime; //!< 経過時間です。
    s32 callCount; //!< 呼び出し回数です。
    ut::Tick maxElapsedTime; //!< 最大の経過時間です。
    ut::Tick minElapsedTime; //!< 最小の経過時間です。
#ifdef NW_PERFORMANCE_COUNTER_ENABLED
    u64 iCacheMiss; //!< 命令キャッシュミス数です。
    u64 dCacheReadMiss; //!< データキャッシュリードミス数です。
    u64 dCacheWriteMiss; //!< データキャッシュライトミス数です。
#endif
};

NW_INLINE void
ProfileReport::Combine(const ProfileReport& report)
{
    elapsedTime += report.elapsedTime;
    callCount += report.callCount;
    maxElapsedTime = ut::Max(maxElapsedTime, report.maxElapsedTime);
    minElapsedTime = ut::Min(minElapsedTime, report.minElapsedTime);
}

//----------------------------------------
struct CompareProfileReport : public std::binary_function<const ProfileReport&, const ProfileReport&, bool>
{
    bool operator()(const ProfileReport& x, const ProfileReport& y) const
    {
        return std::strcmp(x.name, y.name) < 0;
    }
};

//----------------------------------------
typedef std::set<ProfileReport, CompareProfileReport, HeapAllocator<ProfileReport> > ProfileReportSet;

//---------------------------------------------------------------------------
//! @brief 階層を構成するプロファイル結果を保持するノードクラスです。
//---------------------------------------------------------------------------
class ProfileNode
{
public:
    typedef std::vector<ProfileNode*, HeapAllocator<ProfileNode*> > ChildrenList;
    typedef ChildrenList::allocator_type Allocator;
    typedef std::pair<ChildrenList::iterator, ChildrenList::iterator> ChildrenRange;

    //! @brief コンストラクタです。
    ProfileNode(Allocator allocator, const char* name, ProfileNode* parent)
    : m_Parent(parent)
    , m_Children(allocator)
    , m_StartTime(0)
    , m_RecursionCount(0)
    {
        m_Report.name = name;
    }

    //! @brief コピーコンストラクタです。
    ProfileNode(const ProfileNode& src)
    : m_Report(src.m_Report)
    , m_Parent(src.m_Parent)
    , m_Children(src.m_Children)
    , m_StartTime(src.m_StartTime)
    , m_RecursionCount(src.m_RecursionCount)
    {}

    //! @brief デストラクタです。
    ~ProfileNode()
    {
        Clear();
    }

    //! @brief リポートを取得します。
    ProfileReport& GetReport() { return m_Report; }
    //! @brief リポートを取得します。
    const ProfileReport& GetReport() const { return m_Report; }

    //! @brief 引数で渡した名前とレポートの名前が等しい場合に true を返します。
    bool EqualsName(const char* name)
    {
        return m_Report.name == name;
    }

    //! @brief 親ノードを取得します。
    ProfileNode* GetParent() { return m_Parent; }
    //! @brief 親ノードを取得します。
    const ProfileNode* GetParent() const { return m_Parent; }

    //! @brief 引数の名前と等しい名前のレポートを持つ子のノードを取得します。
    ProfileNode* GetChild(const char* name);

    //! @brief 子のノードの先頭と末尾のイテレータを取得します。
    ChildrenRange GetChildren() { return std::make_pair(m_Children.begin(), m_Children.end()); }

    //! @brief アロケータを取得します。
    Allocator GetAllocator() { return m_Children.get_allocator(); }

    //! @brief 計測状態に入ります。
    void Call();
    //! @brief 計測状態から戻ります。
    bool Return();

    //! @brief 子のノードをクリアします。このノードの子以下のノードは全て破棄されます。
    void Clear()
    {
        ChildrenList::iterator end = m_Children.end();
        for (ChildrenList::iterator child = m_Children.begin(); child != end; ++child)
        {
            (*child)->~ProfileNode();
            m_Children.get_allocator().deallocate(*child, sizeof(ProfileNode));
        }
        m_Children.clear();
    }

    //! @brief 呼び出しの階層関係が分かるように計測結果を出力します。
    void DumpHierarchicalReports(u32 mode, ExportFormat exportFormat, float accuracy);

    //! @brief 引数のリストに計測結果を集計して結果を格納します。
    void TabulateReport(ProfileReportSet& reports);

private:
    ProfileReport m_Report;
    ProfileNode* m_Parent;
    ChildrenList m_Children;
    ut::Tick m_StartTime;
    s32 m_RecursionCount;
};

//----------------------------------------
NW_INLINE void
ProfileNode::Call()
{
    ++m_Report.callCount;
    if (m_RecursionCount == 0)
    {
        m_StartTime = ut::Tick::GetSystemCurrent();
    }
    ++m_RecursionCount;
}

//----------------------------------------
NW_INLINE bool
ProfileNode::Return()
{
    --m_RecursionCount;
    if (m_RecursionCount == 0 && m_Report.callCount != 0)
    {
        ut::Tick time = ut::Tick::GetSystemCurrent();
        time -= m_StartTime;
        m_Report.elapsedTime += time;
        m_Report.maxElapsedTime = ut::Max(m_Report.maxElapsedTime, time);
        m_Report.minElapsedTime = ut::Min(m_Report.minElapsedTime, time);
    }
    return m_RecursionCount == 0;
}

//----------------------------------------
NW_INLINE ProfileNode*
ProfileNode::GetChild(const char* name)
{
    ChildrenList::iterator found = std::find_if(
        m_Children.begin(),
        m_Children.end(),
        std::bind2nd(std::mem_fun(&ProfileNode::EqualsName), name));

    if (found == m_Children.end())
    {
        Allocator::rebind<ProfileNode>::other otherAllocator(m_Children.get_allocator());
        //NW_LOG("#### left node: %d, create node: %s\n", otherAllocator.max_size(), name);

        ProfileNode* node = otherAllocator.allocate(1);
        otherAllocator.construct(node, ProfileNode(m_Children.get_allocator(), name, this));
        m_Children.push_back(node);
        return node;
    }
    return *found;
}

//---------------------------------------------------------------------------
//! @brief        プロファイル結果を保存するためのクラスです。
//---------------------------------------------------------------------------
class ProfileManager
{
private:
    NW_DISALLOW_COPY_AND_ASSIGN(ProfileManager);

public:
    //! @brief 設定内容です。
    struct Description
    {
        //! @brief コンストラクタです。
        Description()
            : maxReport(DEFAULT_MAX_REPORT),
              exportFormat(CHART),
              accuracy(MILI_SECOND),
              mode(TIMER)
        {}

        size_t maxReport; //!< レポートの最大数です。
        ExportFormat exportFormat; //!< 出力形式です。
        Accuracy accuracy; //!< 出力精度です。
        u32 mode; //!< 計測を行う項目です。

    };

    //! @brief プロファイルマネージャーを生成します。
    //!
    //! @param[in]    description 設定内容です。
    //! @param[in]    allocator アロケータです。
    //!
    //! @return       生成されたプロファイルマネージャーです。
    //!
    static ProfileManager* CreateProfileManager(
        ProfileManager::Description& description,
        ut::IAllocator* allocator);

    //! @brief プロファイルマネージャーを破棄します。
    //!
    //! @param[in]    allocator アロケータです。
    //!
    void Destroy(ut::IAllocator* allocator);

    //----------------------------------------
    //! @name 取得／設定
    //@{

    const Description& GetDescription() const { return m_Description; }

    //! @brief 保存したレポートを削除します。
    void ClearReports();

    //@}

    //! @brief プロファイルを開始します。
    void StartProfile(const char* name)
    {
        NW_ASSERT_NOT_NULL(m_CurrentNode);
        if (name != m_CurrentNode->GetReport().name)
        {
            m_CurrentNode = m_CurrentNode->GetChild(name);
            NW_ASSERT_NOT_NULL(m_CurrentNode);
        }
        m_CurrentNode->Call();
    }

    //! @brief プロファイルを停止します。
    void StopProfile()
    {
        NW_ASSERT_NOT_NULL(m_CurrentNode);
        if (m_CurrentNode->Return())
        {
            m_CurrentNode = m_CurrentNode->GetParent();
            NW_ASSERT_NOT_NULL(m_CurrentNode);
        }
    }

    //! @brief レポートをコンソールに出力します。
    void DumpReports();

private:
    static const int DEFAULT_MAX_REPORT = 50; //!< デフォルトの最大レポート数です。

    ProfileManager(
        ProfileNode::Allocator allocator,
        const ProfileManager::Description& description);

    ~ProfileManager();

    ProfileManager::Description m_Description;

    ProfileNode m_RootNode;
    ProfileNode* m_CurrentNode;
    static void* m_NodeMemory;
};

//---------------------------------------------------------------------------
//! @brief        デストラクタで自動的にプロファイル結果を作成するクラスです。
//---------------------------------------------------------------------------
class AutoProfile
{
private:
    NW_DISALLOW_COPY_AND_ASSIGN(AutoProfile);

public:
    //! @brief コンストラクタです。
    AutoProfile(const char* name, ProfileManager* profileManager)
    : m_ProfileManager(profileManager)
    {
        m_ProfileManager->StartProfile(name);
    }

    //! @brief デストラクタです。
    ~AutoProfile()
    {
        m_ProfileManager->StopProfile();
    }

private:
    ProfileManager* m_ProfileManager;
};

//---------------------------------------------------------------------------
//! @brief        ProfileManager を管理するクラスです。
//!               ProfileCenter は必要に応じて作り直してください。
//---------------------------------------------------------------------------
class ProfileCenter
{
private:
    NW_DISALLOW_COPY_AND_ASSIGN(ProfileCenter);

public:
    //! @brief プロファイルの管理クラスを初期化します。
    static void Initialize(int maxReport, ut::IAllocator* allocator);

    //! @brief プロファイルの管理クラスを削除します。
    static void Finalize(ut::IAllocator* allocator);

    //! @brief プロファイルマネージャーを設定します。
    static void SetProfileManager(ProfileManager* profile)
    {
        ProfileCenter::GetInstance().SetProfileManagerInternal(profile);
    }

    //! @brief プロファイルマネージャーを取得します。
    static ProfileManager* GetProfileManager()
    {
        return ProfileCenter::GetInstance().GetProfileManagerInternal();
    }

private:
    void SetProfileManagerInternal(ProfileManager* profileManager)
    {
        this->m_ProfileManager = profileManager;
    }

    ProfileManager* GetProfileManagerInternal()
    {
        return m_ProfileManager;
    }

    //! @brief インスタンスを取得します。
    //!
    //! @return シングルトンのインスタンスを返します。
    static ProfileCenter& GetInstance()
    {
        NW_ASSERT_NOT_NULL(m_Instance);
        return *m_Instance;
    }

    ProfileCenter()
        : m_ProfileManager(NULL)
    {}

    static ProfileCenter* m_Instance;
    ProfileManager* m_ProfileManager;
};

#endif // NW_DEV_ENABLED

} // namespace nw::dev
} // namespace nw

#endif // NW_DEV_PROFILE_H_
