﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_DEV_MAP_FILE_H_
#define NW_DEV_MAP_FILE_H_

#include <nw/types.h>
#include <nw/ut/ut_Inlines.h>
#include <nw/ut/os/ut_Mutex.h>

#include <nw/dev/dev_Fs.h>

#if !defined(NW_MAP_ENABLE) && !defined(NW4F_RELEASE)
#define NW_MAP_ENABLE 1
#endif

namespace nw
{
namespace dev
{

//---------------------------------------------------------------------------
//! @brief        マップファイル解析を行うクラスです。
//!
//! @details :category     デバッグ
//---------------------------------------------------------------------------
class MapFile
{
public:
    //! マップファイルの登録を行うための構造体です。
    typedef struct MapFileBuf
    {
        u8*                mapBuf;         //!< マップファイルのバッファへのポインタです。
        FSClient*          client;         //!< DVD ファイルを扱うためのクライアントです。
        FSCmdBlock*        cmdBlock;       //!< DVD ファイルを扱うためのコマンドブロックです。
        FSFileHandle       fileHandle;     //!< DVD ファイルを扱うためのファイルハンドルです。
        struct MapFileBuf* next;           //!< 次のマップファイル構造体へのポインタです。
    }
    MapFileBuf;

    typedef MapFileBuf*    MapFileHandle;


    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //---------------------------------------------------------------------------
    MapFile()
      : GetCharPtr( NULL ),
        m_MapFileList( NULL )
        {}

    //---------------------------------------------------------------------------
    //! @brief        デストラクタです。
    //---------------------------------------------------------------------------
    ~MapFile(){}

#if defined(NW_MAP_ENABLE)
    //---------------------------------------------------------------------------
    //! @brief        メモリ上に展開済みのマップファイルデータを使用して
    //!               マップファイル構造体の初期化をおこないます。
    //!
    //! @param[in]    buffer         マップファイル構造体を作成するためのバッファです。
    //! @param[in]    mapDataBuf     メモリ上に展開されているマップファイルデータへのポインタです。
    //! @param[in]    mapDataLength  マップファイルデータのデータサイズです。
    //!
    //! @return       生成されたマップファイルのハンドルを返します。
    //---------------------------------------------------------------------------
    MapFileHandle RegistOnMem(
        void*       buffer,
        u8*         mapDataBuf,
        u32         mapDataLength
    );

    //---------------------------------------------------------------------------
    //! @brief        ディスク上のファイルのクライアントハンドルおよびファイルハンドルを
    //!               指定してマップファイル構造体の初期化をおこないます。
    //!
    //! @param[in]    buffer         マップファイル構造体を作成するためのバッファです。
    //! @param[in]    client         DVD ファイルを扱うためのクライアントです。
    //! @param[in]    cmdBlock       DVD ファイルを扱うためのコマンドブロックです。
    //! @param[in]    fileHandle     DVD ファイルを扱うためのファイルハンドルです。
    //!
    //! @return       生成されたマップファイルのハンドルを返します。
    //---------------------------------------------------------------------------
    MapFileHandle RegistOnDvd(
        void*            buffer,
        FSClient*        client,
        FSCmdBlock*      cmdBlock,
        FSFileHandle     fileHandle
    );


    //---------------------------------------------------------------------------
    //! @brief        登録されているマップファイルを解除します。
    //!
    //! @param[in]    hMapFile       マップファイルのハンドルです。
    //!
    //! @return       登録時に渡されたバッファ領域へのポインタを返します。
    //---------------------------------------------------------------------------
    void* Unregist( MapFileHandle hMapFile )
    {
        RemoveMapFile( hMapFile );
        return (void*)hMapFile;
    }

    //---------------------------------------------------------------------------
    //! @brief        登録されているマップファイルを全て解除します。
    //---------------------------------------------------------------------------
    void UnregistAll()
    {
        m_MapFileList = NULL;
    }

    //---------------------------------------------------------------------------
    //! @brief        登録されているマップファイルが存在するかどうかをチェックします
    //!
    //! @return       存在する場合、 true を返します。
    //---------------------------------------------------------------------------
    bool Exists()
    {
        return ( m_MapFileList != NULL ) ? true : false;
    }

    //---------------------------------------------------------------------------
    //! @brief        登録されたマップファイルから address に対応するシンボル名を検索し、
    //!               発見された場合には、シンボル名をコピーします。
    //!
    //! @param[in]    address        シンボルを検索する対象となるアドレスです。
    //! @param[in]    strBuf         シンボル名を格納するためのバッファへのポインタです。
    //! @param[in]    strBufSize     シンボル名を格納するためのバッファのサイズです。
    //!
    //! @return       address に対応するシンボルが見つかった場合、 true を返します。
    //---------------------------------------------------------------------------
    bool QuerySymbol(
        u32 address,
        u8* strBuf,
        u32 strBufSize
    );

#else // if !defined(NW_MAP_ENABLE)
    MapFileHandle RegistOnMem(
        void*       buffer,
        u8*         mapDataBuf,
        u32         mapDataLength,
        const void* moduleInfo = NULL
    )
    {
        NW_UNUSED_VARIABLE( buffer );
        NW_UNUSED_VARIABLE( mapDataBuf );
        NW_UNUSED_VARIABLE( mapDataLength );
        NW_UNUSED_VARIABLE( moduleInfo );
        return (MapFileHandle)NULL;
    }

    MapFileHandle RegistOnDvd(
        void*       buffer,
        const char* filePath,
        const void* moduleInfo = NULL
    )
    {
        NW_UNUSED_VARIABLE( buffer );
        NW_UNUSED_VARIABLE( filePath );
        NW_UNUSED_VARIABLE( moduleInfo );
        return (MapFileHandle)NULL;
    }

    void* Unregist( MapFileHandle hMapFile )
    {
        NW_UNUSED_VARIABLE( hMapFile );
        return NULL;
    }

    void UnregistAll() {}

    bool Exists() { return false; }


    bool QuerySymbol(
        u32 address,
        u8* strBuf,
        u32 strBufSize
    )
    {
        NW_UNUSED_VARIABLE( address );
        NW_UNUSED_VARIABLE( strBuf );
        NW_UNUSED_VARIABLE( strBufSize );
        return false;
    }
#endif // if defined(NW_MAP_ENABLE)

private:
    void AppendMapFile( MapFileBuf* pMapFile );

    void RemoveMapFile( MapFileBuf* pMapFile );

    bool QuerySymbolToSingleMapFile(
        MapFileBuf* pMapFile,
        u32         address,
        u8*         strBuf,
        u32         strBufSize
    );

    bool QuerySymbolToMapFile(
        u8*         buf,
        const void* moduleInfo,
        u32         address,
        u8*         strBuf,
        u32         strBufSize
    );

    u8* SearchNextLine( u8* buf, s32 lines );

    u8* SearchParagraph( u8* buf, const char* name );

    u8* SearchParam( u8* lineTop, u32 argNum, u8 splitter );

    u32 XStrToU32( const u8* str );

    u32 CopySymbol(
        const u8*   buf,
        u8*         str,
        u32         strLenMax,
        u8          splitter
    );

    u8 GetCharOnMem( const u8* buf );
    u8 GetCharOnDvd( const u8* buf );

    static const u32 MAP_BUFFER_SIZE = NW_UT_ROUNDUP( 512, PPC_IO_BUFFER_ALIGN );
    ALIGNVAR( PPC_IO_BUFFER_ALIGN ) static u8 s_MapBuf[ MAP_BUFFER_SIZE ];
    static s32 s_MapBufOffset;
    static u32 s_FileLength;
    static FSClient*     s_Client;
    static FSCmdBlock*   s_CmdBlock;
    static FSFileHandle  s_FileHandle;

    u8 (MapFile::*GetCharPtr)( const u8* buf );

    MapFileBuf* m_MapFileList;

    static nw::ut::Mutex s_Mutex;
    static bool s_MutexInitialized;
};

} // namespace dev
} // namespace nw

#endif  // NW_DEV_MAP_FILE_H_

