﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nw/ut/ut_Inlines.h>

namespace nw {
namespace ut {

class FileStream;

} // namespace ut
} // namespace nw

namespace nw {
namespace dev {

//---------------------------------------------------------------------------
//! @brief        指定ファイルにデバッグ出力を行うクラスです。
//!
//! @details
//! :category     デバッグ
//!
//! ログ出力を行う前に SetBuffer(), Attach() により、バッファと出力先ファイルを設定しておく必要があります。
//---------------------------------------------------------------------------
class FileLogger
{
private:
    struct BufferInfo
    {
        BufferInfo() : buffer(NULL), bufferedSize(0) { }

        char* buffer;
        u32   bufferedSize;
    };

public:
    //! @brief コンストラクタです。
    FileLogger();

    //! @brief デストラクタです。
    virtual ~FileLogger() {}

public:
    //---------------------------------------------------------------------------
    //! @brief        バッファ0 を取得します。
    //!
    //! @return       バッファ0 へのポインタを返します。
    //---------------------------------------------------------------------------
    char* GetBuffer0() const
    {
        return m_BufferInfo0.buffer;
    }

    //---------------------------------------------------------------------------
    //! @brief        バッファ1 を取得します。
    //!
    //! @return       バッファ1 へのポインタを返します。
    //---------------------------------------------------------------------------
    char* GetBuffer1() const
    {
        return m_BufferInfo1.buffer;
    }

    //---------------------------------------------------------------------------
    //! @brief        バッファサイズを取得します。
    //!
    //! @return       バッファサイズを返します。
    //---------------------------------------------------------------------------
    u32 GetBufferSize() const
    {
        return m_BufferSize;
    }

    //---------------------------------------------------------------------------
    //! @brief        文字列の書式化や文字列の一括出力に使用するバッファを指定します。
    //!
    //! @details      バッファは必ず設定する必要があります。
    //!
    //! @param[in]    buffer        バッファ0 へのポインタです。
    //! @param[in]    bufferSize    バッファのサイズです。
    //!                             プラットフォームのバッファアライメントサイズを加味したサイズを指定します。
    //---------------------------------------------------------------------------
    void SetBuffer(char* buffer, u32 bufferSize);

    //---------------------------------------------------------------------------
    //! @brief        文字列の書式化や文字列の一括出力に使用するバッファを指定します。
    //!
    //! @details      バッファは必ず設定する必要があります。
    //!               HW割り込み中のログ出力に対応するために文字列格納用と文字列出力用、２つのバッファを指定します。
    //!
    //! @param[in]    buffer0       バッファ0 へのポインタです。
    //! @param[in]    buffer1       バッファ1 へのポインタです。
    //! @param[in]    bufferSize    バッファ0、バッファ1 のサイズです。
    //!                             プラットフォームのバッファアライメントサイズを加味したサイズを指定します。
    //---------------------------------------------------------------------------
    void SetBuffer(char* buffer0, char* buffer1, u32 bufferSize);

    //---------------------------------------------------------------------------
    //! @brief        出力先ファイルを関連付けます。
    //!
    //! @param[in]    fileStream    出力先ファイルを指定します。
    //---------------------------------------------------------------------------
    void Attach(ut::FileStream& fileStream);

    //---------------------------------------------------------------------------
    //! @brief        出力先ファイルの関連付けを解除します。
    //!
    //! @return       直前まで関連付けられていた出力ファイルを返します。
    //---------------------------------------------------------------------------
    ut::FileStream* Detach();

    //---------------------------------------------------------------------------
    //! @brief        出力先ファイルが関連付けられているかどうかを調べます。
    //!
    //! @return       関連付けられている場合は true、関連付けられていない場合は false を返します。
    //---------------------------------------------------------------------------
    bool IsAttached() const
    {
        return m_pFileStream != NULL;
    }

    //---------------------------------------------------------------------------
    //! @brief        関連付けられている出力先ファイルを取得します。
    //!
    //! @return       関連付けられている出力ファイルを返します。
    //---------------------------------------------------------------------------
    ut::FileStream* GetFileStream() const
    {
        return m_pFileStream;
    }

    //---------------------------------------------------------------------------
    //! @brief        文字列を出力します。
    //!
    //! @details      HW割り込み中のログ出力には Printf() の代わりに PrintfToBuffer() を使用してください。
    //!
    //! @param[in]    format    出力文字列フォーマットです。
    //!
    //! @sa           PrintfToBuffer
    //---------------------------------------------------------------------------
    void Printf(const char* format, ...);

    //---------------------------------------------------------------------------
    //! @brief        文字列を出力します。
    //!
    //! @details      HW割り込み中のログ出力には VPrintf() の代わりに VPrintfToBuffer() を使用してください。
    //!
    //! @param[in]    format    出力文字列フォーマットです。
    //! @param[in]    vlist     可変引数リストです。
    //!
    //! @sa           VPrintfToBuffer
    //---------------------------------------------------------------------------
    void VPrintf(const char* format, std::va_list vlist);

    //---------------------------------------------------------------------------
    //! @brief        ファイルロガーが管理するバッファに文字列を出力します。
    //!
    //! @param[in]    format    出力文字列フォーマットです。
    //---------------------------------------------------------------------------
    void PrintfToBuffer(const char* format, ...);

    //---------------------------------------------------------------------------
    //! @brief        ファイルロガーが管理するバッファに文字列を出力します。
    //!
    //! @param[in]    format    出力文字列フォーマットです。
    //! @param[in]    vlist     可変引数リストです。
    //---------------------------------------------------------------------------
    void VPrintfToBuffer(const char* format, std::va_list vlist);

    //---------------------------------------------------------------------------
    //! @brief        ファイルロガーのバッファに格納された文字列をすべて出力します。
    //---------------------------------------------------------------------------
    void FlushBuffer();

protected:
    virtual u32 GetBufferAlignment() const
    {
        return 0;
    }

private:
    bool IsBufferSetuped() const;
    BufferInfo* GetIdleBufferInfo();

private:
    ut::FileStream* m_pFileStream;

    BufferInfo  m_BufferInfo0;
    BufferInfo  m_BufferInfo1;
    BufferInfo* m_pCurrentBufferInfo;
    u32 m_BufferSize;
};

} // namespace dev
} // namespace nw
