﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_DEV_FILE_DEVICE_MANAGER_H_
#define NW_DEV_FILE_DEVICE_MANAGER_H_

#include <nw/dev/dev_FileDevice.h>
#include <nw/dev/dev_Fs.h>

#define NW_DEV_FS_MAX_FULLPATH_SIZE nw::dev::FileDevice::MAX_FULLPATH_SIZE

namespace nw
{
namespace dev
{

//---------------------------------------------------------------------------
//! @brief        ファイルデバイスを管理するクラスです。
//!
//! @details :category     ファイルシステム
//---------------------------------------------------------------------------
class FileDeviceManager
{
private:
#if defined( NW_PLATFORM_WIN32 )
    typedef ::nw::internal::winext::FSClient FSClient;
    typedef ::nw::internal::winext::FSCmdBlock FSCmdBlock;
#endif

public:
    //! @brief Create に渡す生成パラメータです。
    struct CreateArg
    {
    private:
#if defined( NW_PLATFORM_WIN32 )
        typedef nw::internal::winext::FSClient FSClient;
        typedef nw::internal::winext::FSCmdBlock FSCmdBlock;
#endif

    public:
        nw::ut::IAllocator* allocator;      //!< アロケータです。
#if !defined( NW_USE_NINTENDO_SDK )
        FSClient*           client;         //!< ファイル操作に用いるファイルハンドルを指定します。コマンドブロックと共に NULL を指定すると Initialize で生成します。
        FSCmdBlock*         cmdBlock;       //!< ファイル操作に用いるコマンドブロックを指定します。ファイルハンドルと共に NULL を指定すると Initialize で生成します。
        bool                useMount;       //!< true を指定するとマウント可能ソースを検索し、ボリュームをマウントします。sd01 デバイスをマウントする場合は mountSd01 も true にしてください。
        bool                mountSd01;      //!< true を指定すると sd01 デバイスをマウントして使用します。 false を指定するとデフォルトで /vol/content にマウントされている hfio01 デバイスを使用します。
#endif

        //! @brief コンストラクタです。
        CreateArg()
            : allocator( NULL )
#if !defined( NW_USE_NINTENDO_SDK )
            , mountSd01( false )
            , client( NULL )
            , cmdBlock( NULL )
            , useMount( false )
#endif
        {}
    };

    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //---------------------------------------------------------------------------
    FileDeviceManager();

    //---------------------------------------------------------------------------
    //! @brief        デストラクタです。
    //---------------------------------------------------------------------------
    ~FileDeviceManager();


    //---------------------------------------------------------------------------
    //! @brief        初期化処理です。
    //!
    //! @param[in]    arg 初期化に用いるパラメータです。
    //---------------------------------------------------------------------------
    void Initialize( const CreateArg& arg );

    //---------------------------------------------------------------------------
    //! @brief        初期化が行われたかを取得します。
    //!
    //! @return       Initialize() が呼び出し済みであれば true を返します。
    //---------------------------------------------------------------------------
    bool IsInitialized() const { return m_Allocator != NULL; }

    //---------------------------------------------------------------------------
    //! @brief        終了処理です。
    //---------------------------------------------------------------------------
    void Finalize();


    //---------------------------------------------------------------------------
    //! @brief        唯一のインスタンスを取得します。
    //!
    //! @return       インスタンスを返します。
    //---------------------------------------------------------------------------
    static FileDeviceManager* GetInstance()
    {
        static FileDeviceManager s_Instance;
        return &s_Instance;
    }

    //---------------------------------------------------------------------------
    //! @brief        ファイルを開きます。
    //!
    //! @param[out]   handle    ファイルハンドルへのポインタです。
    //! @param[in]    filename  開くファイル名です。
    //! @param[in]    flag      ファイルの開き方を表すフラグです。
    //!
    //! @return       成功した場合、ファイルを開くのに用いたデバイスへのポインタを返します。
    //!               失敗した場合、 NULL を返します。
    //---------------------------------------------------------------------------
    FileDevice* Open(
            FileHandle* handle,
            const char* filename,
            FileDevice::FileOpenFlag flag = FileDevice::FILE_OPEN_FLAG_READ_ONLY
    );

    //---------------------------------------------------------------------------
    //! @brief        データをロードします。
    //!
    //! @param[in]    arg       読み込み設定です。
    //!
    //! @return       読み込んだデータへのポインタを返します。
    //---------------------------------------------------------------------------
    u8* Load(FileDevice::LoadArg& arg);

    //---------------------------------------------------------------------------
    //! @brief        データをロードします。
    //!
    //! @param[in]    path      ロードするファイルパスです。
    //! @param[in]    allocator ロードする際に用いるアロケーターです。
    //!
    //! @return       読み込んだデータへのポインタを返します。
    //---------------------------------------------------------------------------
    u8* Load(const char* path, nw::ut::IAllocator* allocator)
    {
        FileDevice::LoadArg arg;
        arg.path = path;
        arg.allocator = allocator;
        return Load( arg );
    }

    //---------------------------------------------------------------------------
    //! @brief        データをロードします。
    //!
    //!               ファイルが無い場合、 NULL を返します。
    //!
    //! @param[in]    arg       読み込み設定です。
    //!
    //! @return       読み込んだデータへのポインタを返します。
    //---------------------------------------------------------------------------
    u8* TryLoad(FileDevice::LoadArg& arg);

    //---------------------------------------------------------------------------
    //! @brief        データをロードします。
    //!
    //!               ファイルが無い場合、 NULL を返します。
    //!
    //! @param[in]    path      ロードするファイルパスです。
    //! @param[in]    allocator ロードする際に用いるアロケーターです。
    //!
    //! @return       読み込んだデータへのポインタを返します。
    //---------------------------------------------------------------------------
    u8* TryLoad(const char* path, nw::ut::IAllocator* allocator)
    {
        FileDevice::LoadArg arg;
        arg.path = path;
        arg.allocator = allocator;
        return TryLoad( arg );
    }

    //---------------------------------------------------------------------------
    //! @brief        読み込んだデータを解放します。
    //!
    //! @param[in]    arg       読み込み設定です。
    //! @param[in]    data      解放するデータです。
    //---------------------------------------------------------------------------
    void Unload(FileDevice::LoadArg& arg, u8* data);

    //---------------------------------------------------------------------------
    //! @brief        ディレクトリを開きます。
    //!
    //! @param[out]   handle    ディレクトリハンドルへのポインタです。
    //! @param[in]    dirName   開くディレクトリ名です。
    //!
    //! @return       成功した場合、ディレクトリを開くのに用いたデバイスへのポインタを返します。
    //!               失敗した場合、 NULL を返します。
    //---------------------------------------------------------------------------
    FileDevice* OpenDirectory(DirectoryHandle* handle, const char* dirName);

    //---------------------------------------------------------------------------
    //! @brief        ファイルデバイスを取得します。
    //!
    //! @return       ファイルデバイスを返します。
    //---------------------------------------------------------------------------
    FileDevice* GetFileDevice() const { return m_FileDevice; }

#if !defined(NW_USE_NINTENDO_SDK)
    //---------------------------------------------------------------------------
    //! @brief        FS のクライアントを取得します。
    //!
    //! @return       クライアントを返します。
    FSClient* GetFSClient() const { return m_FSClient; }

    //---------------------------------------------------------------------------
    //! @brief        FS のコマンドブロックを取得します。
    //!
    //! @return       コマンドブロックを返します。
    //---------------------------------------------------------------------------
    FSCmdBlock* GetFSCmdBlock() const { return m_FSCmdBlock; }
#endif

    //---------------------------------------------------------------------------
    //! @brief        FS のマウントパスを取得します。
    //!
    //! @return       マウントパスを返します。
    //---------------------------------------------------------------------------
#if defined(NW_USE_NINTENDO_SDK)
    const char* GetFSMountPath() const { return ""; }
#else
    const char* GetFSMountPath() const { return m_FSMountPath; }
#endif

    //---------------------------------------------------------------------------
    //! @brief        ファイルデバイスのカレントワークディレクトリを設定します。
    //!
    //! @param[in]    path      カレントワークディレクトリのパスです。
    //---------------------------------------------------------------------------
    void ChangeDir( const char* path )
    {
        NW_ASSERT_NOT_NULL( m_FileDevice );
        m_FileDevice->ChangeDir( path );
    }

    //---------------------------------------------------------------------------
    //! @brief        ファイルデバイスのカレントワークディレクトリを取得します。
    //!
    //! @return       カレントワークディレクトリを返します。
    //---------------------------------------------------------------------------
    const char* GetCWD() const
    {
        NW_ASSERT_NOT_NULL( m_FileDevice );
        return m_FileDevice->GetCWD();
    }

#if defined(NW_USE_NINTENDO_SDK)
    //---------------------------------------------------------------------------
    //! @briefprivate
    //! @brief        nn::fs のアロケータを設定します。
    //!
    //! @details
    //! nn::fs::SetAllocator() を呼び出し、 Initialize() で指定されたアロケータを nn::fs でも使うように設定します。
    //!
    //! 必ずしも本関数を使って nn::fs のアロケータを設定する必要はありません。
    //!
    //! nn::fs のアロケータの設定は一度しかできません。
    //!
    //! nn::fs が使用される間は FileDeviceManager のインスタンスが初期化済みの状態で存在している必要があります。
    //---------------------------------------------------------------------------
    void SetFsAllocator();
#endif

private:
#if defined(NW_USE_NINTENDO_SDK)
    static void* FsAllocateFunction(size_t size);
    static void FsDeallocateFunction(void* ptr, size_t size);
#endif

protected:
#if !defined(NW_USE_NINTENDO_SDK)
    static const char*  FS_SD01_MOUNT_PATH;    //!< デフォルトの FS の SDIO をマウントするパスです。
    static const char*  FS_HFIO01_MOUNT_PATH;  //!< デフォルトの FS の HFIO をマウントするパスです。
#endif

    FileDevice*         m_FileDevice;           //!< ファイルデバイスです。
#if !defined(NW_USE_NINTENDO_SDK)
    FSClient*           m_FSClient;             //!< FS のクライアントです。
    FSCmdBlock*         m_FSCmdBlock;           //!< FS のコマンドブロックです。
#endif
    char*               m_FSMountPath;          //!< FS のマウントパスです。
    bool                m_FSInitialized;        //!< FileDeviceManager によって FS が初期化されたかを表します。

    CreateArg                 m_Arg;            //!< 生成パラメータです。
    nw::ut::IAllocator*       m_Allocator;      //!< アロケーターです。
    static FileDeviceManager* s_pInstance;      //!< シングルトンオブジェクトへのポインタです。
};

} // namespace dev
} // namespace nw

#endif // NW_DEV_FILE_DEVICE_MANAGER_H_
