﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_DEV_FILE_DEVICE_H_
#define NW_DEV_FILE_DEVICE_H_

#include <nw/ut/ut_Memory.h>
#include <nw/ut/ut_SafeString.h>
#include <nw/dev/dev_Fs.h>

namespace nw
{
namespace dev
{

class FileDevice;

static const int HANDLE_BYTES_MAX = 32; //!< :private
#if defined(NW_USE_NINTENDO_SDK)
static const int DIR_NAME_MAX = nn::fs::EntryNameLengthMax + 1; //!< @briefprivate
#else
static const int DIR_NAME_MAX = 256;    //!< :private
#endif

//---------------------------------------------------------------------------
//! @brief        ディレクトリの中の要素です。
//!
//! @details :category     ファイルシステム
//---------------------------------------------------------------------------
struct DirectoryEntry
{
    //! @brief コンストラクタです。
    DirectoryEntry()
        : name( nw::ut::SafeString("") )
        , isDirectory(false)
    {}

    nw::ut::FixedSafeString<DIR_NAME_MAX> name;       //!< 要素の名前（ファイル名またはディレクトリ名）です。
    bool isDirectory;                                 //!< ディレクトリかどうかです。
};



//---------------------------------------------------------------------------
//! @brief        ファイルの各種操作に使うハンドルの基底クラスです。
//!
//! @details :category     ファイルシステム
//---------------------------------------------------------------------------
class HandleBase
{
    friend class FileDevice;
public:
    //! @brief コンストラクタです。
    HandleBase() : m_Device(NULL) {}

    //! @brief デストラクタです。
    virtual ~HandleBase() {}

    //! @brief デバイスを取得します。
    //! @return ファイルデバイスを返します。
    FileDevice* GetDevice() const { return m_Device; }

    //! @brief ファイルが開かれているかどうかを取得します。
    //! @return 開かれている場合は true を返します。
    bool IsOpen() const { return m_Device != NULL; }

protected:
    FileDevice* m_Device;                //!< デバイスを保持しているメンバです。
    u8 m_HandleBuffer[HANDLE_BYTES_MAX]; //!< ハンドル用バッファです。
};



//---------------------------------------------------------------------------
//! @brief        ファイルの各種操作に使うハンドルクラスです。
//!
//! @details :category     ファイルシステム
//---------------------------------------------------------------------------
class FileHandle : public HandleBase
{
public:
    //! @brief seek 時の基準位置です。
    enum SeekOrigin
    {
        SEEK_ORIGIN_BEGIN,      //!< ファイルの先頭
        SEEK_ORIGIN_CURRENT,    //!< ファイルの現在位置
        SEEK_ORIGIN_END         //!< ファイルの終端
    };


    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //---------------------------------------------------------------------------
    FileHandle() : HandleBase() {}

    //---------------------------------------------------------------------------
    //! @brief        デストラクタです。
    //---------------------------------------------------------------------------
    virtual ~FileHandle();


    //---------------------------------------------------------------------------
    //! @brief        ファイルを閉じます。
    //!
    //! @return       成功した場合、 true を返します。
    //---------------------------------------------------------------------------
    bool Close();

    //---------------------------------------------------------------------------
    //! @brief        ファイルの読み込みを行います。
    //!
    //!               失敗した場合、 ASSERT します。
    //!
    //! @param[out]   buf       読み込んだデータを受け取るバッファへのポインタです。
    //! @param[in]    size      読み込むサイズです。
    //!
    //! @return       実際に読み込まれたデータのサイズを返します。
    //---------------------------------------------------------------------------
    u32 Read(u8* buf, u32 size);

    //---------------------------------------------------------------------------
    //! @brief        ファイルの読み込みを行います。
    //!
    //!               失敗した場合、 false を返します。
    //!
    //! @param[out]   readSize  実際に読み込まれたデータのサイズを受け取る変数へのポインタです。
    //!                         サイズを受け取る必要がない場合は NULL を指定できます。
    //! @param[out]   buf       読み込んだデータを受け取るバッファへのポインタです。
    //! @param[in]    size      読み込むサイズです。
    //!
    //! @return       成功した場合、 true を返します。
    //---------------------------------------------------------------------------
    bool TryRead(u32* readSize, u8* buf, u32 size);

    //---------------------------------------------------------------------------
    //! @brief        読み書き位置を移動します。
    //!
    //!               失敗した場合、 ASSERT します。
    //!
    //! @param[in]    offset    移動量です。
    //! @param[in]    origin    基準位置です。
    //!
    //! @return       成功した場合、 true を返します。
    //---------------------------------------------------------------------------
    bool Seek(s32 offset, SeekOrigin origin);

    //---------------------------------------------------------------------------
    //! @brief        読み書き位置を移動します。
    //!
    //!               失敗した場合、 false を返します。
    //!
    //! @param[in]    offset    移動量です。
    //! @param[in]    origin    基準位置です。
    //!
    //! @return       成功した場合、 true を返します。
    //---------------------------------------------------------------------------
    bool TrySeek(s32 offset, SeekOrigin origin);

    //---------------------------------------------------------------------------
    //! @brief        読み書き位置を取得します。
    //!
    //!               失敗した場合、 ASSERT します。
    //!
    //! @return       読み書き位置を返します。
    //---------------------------------------------------------------------------
    u32 GetCurrentSeekPos();

    //---------------------------------------------------------------------------
    //! @brief        読み書き位置を取得します。
    //!
    //!               失敗した場合、 false を返します。
    //!
    //! @param[out]   pos       読み書き位置を受け取る変数へのポインタです。
    //!                         NULL は指定できません。
    //!
    //! @return       成功した場合、 true を返します。
    //---------------------------------------------------------------------------
    bool TryGetCurrentSeekPos(u32* pos);

    //---------------------------------------------------------------------------
    //! @brief        ファイルサイズを取得します。
    //!
    //!               失敗した場合、 ASSERT します。
    //!
    //! @return       ファイルサイズを返します。
    //---------------------------------------------------------------------------
    u32 GetFileSize();

    //---------------------------------------------------------------------------
    //! @brief        ファイルサイズを取得します。
    //!
    //!               失敗した場合、 false を返します。
    //!
    //! @param[out]   size      取得したサイズを受け取る変数へのポインタです。
    //!                         NULL は指定できません。
    //!
    //! @return       成功した場合、 true を返します。
    //---------------------------------------------------------------------------
    bool TryGetFileSize(u32* size);
};



//---------------------------------------------------------------------------
//! @brief        ディレクトリの各種操作に使うハンドルクラスです。
//!
//! @details :category     ファイルシステム
//---------------------------------------------------------------------------
class DirectoryHandle : public HandleBase
{
public:
    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //---------------------------------------------------------------------------
    DirectoryHandle() : HandleBase() {}

    //---------------------------------------------------------------------------
    //! @brief        デストラクタです。
    //---------------------------------------------------------------------------
    virtual ~DirectoryHandle() {}


    //---------------------------------------------------------------------------
    //! @brief        ディレクトリを閉じます。
    //!
    //! @return       成功した場合、 true を返します。
    //---------------------------------------------------------------------------
    bool Close();

    //---------------------------------------------------------------------------
    //! @brief        ディレクトリ情報の読み込みを行います。
    //!
    //!               失敗した場合、 ASSERT します。
    //!
    //! @param[out]   entry     読み込んだディレクトリ情報を入れる場所です。
    //! @param[in]    num       読み込むディレクトリ情報の個数です。
    //!
    //! @return       実際に読み込まれたディレクトリ情報の数を返します。
    //---------------------------------------------------------------------------
    u32 Read(DirectoryEntry entry[], u32 num);

    //---------------------------------------------------------------------------
    //! @brief        ディレクトリ情報の読み込みを行います。
    //!
    //!               失敗した場合、 false を返します。
    //!
    //! @param[out]   readNum   実際に読み込んだディレクトリ情報の個数を受け取る変数へのポインタです。
    //!                         個数を受け取る必要がない場合は NULL を指定できます。
    //! @param[out]   entry     読み込んだディレクトリ情報を入れる場所です。
    //! @param[in]    num       読み込むディレクトリ情報の個数です。
    //!
    //! @return       成功した場合、 true を返します。
    //---------------------------------------------------------------------------
    bool TryRead(u32* readNum, DirectoryEntry entry[], u32 num);
};



//---------------------------------------------------------------------------
//! @brief        ファイルデバイスを扱うクラスです。
//!
//! @details :category     ファイルシステム
//---------------------------------------------------------------------------
// TODO: 書き込みを追加する。
class FileDevice
{
public:
    //! @brief ファイルの開き方を表す列挙型です。
    enum FileOpenFlag
    {
        FILE_OPEN_FLAG_READ_ONLY,   //!< 読み込みのみ
        FILE_OPEN_FLAG_WRITE_ONLY,  //!< （未実装）書き込みのみ（ファイルがあっても上書き。ない場合は FILE_OPEN_FLAG_CREATE と同じ動作）
        FILE_OPEN_FLAG_READ_WRITE,  //!< （未実装）読み書き
        FILE_OPEN_FLAG_CREATE       //!< （未実装）書き込み（すでにファイルがあるとエラー）
    };

    //! 格納可能なフルパス長です。
#if defined(NW_USE_NINTENDO_SDK)
    static const int MAX_FULLPATH_SIZE = 4096; // 適当
#else
    static const int MAX_FULLPATH_SIZE = FS_MAX_FULLPATH_SIZE;
#endif
    //! ファイルアクセスを行うときの最小のアラインメント。プラットフォームによって異なる。
#if defined(NW_PLATFORM_CAFE)
    static const int BUFFER_MIN_ALIGNMENT = 64;
#else
    static const int BUFFER_MIN_ALIGNMENT = 32;
#endif

    //! @brief Load 用の引数構造体です。
    struct LoadArg
    {
        /// コンストラクタ
        LoadArg()
            : path( NULL )
            , buffer( NULL )
            , bufferSize( 0 )
            , allocator( NULL )
            , alignment( 0 )
            , readSize( 0 )
            , roundupSize( 0 )
            , needUnload( false )
        {}

        //----------------------------------------
        //! @name 入力
        //  @{
        const char*         path;           //!< 読み込むファイルへのパスです。必ず指定します。
        u8*                 buffer;         //!< 読み込んだデータを格納するバッファです。指定しない場合、内部でバッファを確保します。
        u32                 bufferSize;     //!< buffer に指定したバッファのサイズ、または内部で確保するバッファのサイズです。
        nw::ut::IAllocator* allocator;      //!< 内部でバッファを確保する場合に使用するアロケーターです。
        s32                 alignment;      //!< 内部でバッファを確保する場合に指定するアライメントです。
        //  @}

        //----------------------------------------
        //! @name 出力
        //  @{
        u32                 readSize;       //!< 読み込んだファイルのサイズです。
        u32                 roundupSize;    //!< ラウンドアップ後のサイズです。bufferSize が指定されている場合は、bufferSize を返します。
        bool                needUnload;     //!< 読み込まれたデータを Unload すべきかを表します。
        //  @}
    };


    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //---------------------------------------------------------------------------
    FileDevice();

    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //!
    //! @param[in]    CWDPath   カレントワークディレクトリです。
    //---------------------------------------------------------------------------
    /* ctor */ explicit FileDevice( const char* CWDPath );

    //---------------------------------------------------------------------------
    //! @brief        デストラクタです。
    //---------------------------------------------------------------------------
    virtual ~FileDevice() {}


    //---------------------------------------------------------------------------
    //! @brief        カレントワークディレクトリを設定します。
    //!
    //! @param[in]    path      カレントワークディレクトリのパスです。
    //---------------------------------------------------------------------------
    void ChangeDir( const char* path )
    {
        m_CWDPath = nw::ut::SafeString(path);
    }

    //---------------------------------------------------------------------------
    //! @brief        カレントワークディレクトリを取得します。
    //!
    //! @return       カレントワークディレクトリを返します。
    //---------------------------------------------------------------------------
    const char* GetCWD()
    {
        return m_CWDPath.c_str();
    }

    //---------------------------------------------------------------------------
    //! @brief        ファイルをロードします。
    //!
    //!               ファイルのオープン、バッファの確保、読み込み、クローズをまとめて行います。
    //!               ファイルが無い場合、 ASSERT します。
    //!
    //! @param[in]    arg       読み込み設定です。
    //!
    //! @return       読み込んだデータへのポインタを返します。
    //---------------------------------------------------------------------------
    u8* Load(LoadArg& arg)
    {
        u8* result = TryLoad(arg);
        NW_ASSERTMSG( result, "Load failed. [%s]", arg.path );
        return result;
    }

    //---------------------------------------------------------------------------
    //! @brief        ファイルをロードします。
    //!
    //!               ファイルのオープン、バッファの確保、読み込み、クローズをまとめて行います。
    //!               ファイルが無い場合、 NULL を返します。
    //!
    //! @param[in]    arg       読み込み設定です。
    //!
    //! @return       読み込んだデータへのポインタを返します。失敗した場合、 NULL を返します。
    //---------------------------------------------------------------------------
    u8* TryLoad(LoadArg& arg);

    //---------------------------------------------------------------------------
    //! @brief        ファイルを開きます。
    //!
    //!               ファイルが無い場合、 ASSERT します。
    //!
    //! @param[out]   handle    ファイルハンドルを受け取る変数へのポインタです。
    //! @param[in]    fileName  開くファイル名です。
    //! @param[in]    flag      ファイルの開き方を表すフラグです。
    //!
    //! @return       成功した場合、ファイルを開くのに用いたデバイスへのポインタを返します。
    //---------------------------------------------------------------------------
    FileDevice* Open(
            FileHandle* handle,
            const char* fileName,
            FileOpenFlag flag = FILE_OPEN_FLAG_READ_ONLY)
    {
        FileDevice* result = TryOpen(handle, fileName, flag);
        NW_ASSERTMSG( result, "Open failed. [%s]", fileName );
        return result;
    }

    //---------------------------------------------------------------------------
    //! @brief        ファイルを開きます。
    //!
    //!               ファイルが無い場合、 NULL を返します。
    //!
    //! @param[out]   handle    ファイルハンドルを受け取る変数へのポインタです。
    //! @param[in]    fileName  開くファイル名です。
    //! @param[in]    flag      ファイルの開き方を表すフラグです。
    //!
    //! @return       成功した場合、ファイルを開くのに用いたデバイスへのポインタを返します。
    //!               失敗した場合、 NULL を返します。
    //---------------------------------------------------------------------------
    FileDevice* TryOpen(
        FileHandle* handle,
        const char* fileName,
        FileOpenFlag flag
    );

    //---------------------------------------------------------------------------
    //! @brief        ファイルを閉じます。
    //!
    //! @param[in]    handle    ファイルハンドルです。
    //!
    //! @return       成功した場合、true を返します。
    //---------------------------------------------------------------------------
    bool Close(FileHandle* handle);

    //---------------------------------------------------------------------------
    //! @brief        ファイルを読み込みます。
    //!
    //!               失敗した場合、 ASSERT します。
    //!
    //! @param[in]    handle    ファイルハンドルです。
    //! @param[out]   buf       読み込んだデータを受け取るバッファへのポインタです。
    //! @param[in]    size      読み込むサイズです。
    //!
    //! @return       実際に読み込まれたデータのサイズを返します。
    //---------------------------------------------------------------------------
    u32 Read(FileHandle* handle, u8* buf, u32 size)
    {
        u32 readSize = 0;
        bool success = TryRead(&readSize, handle, buf, size);
        NW_ASSERTMSG(success, "file Read error");
        return readSize;
    }

    //---------------------------------------------------------------------------
    //! @brief        ファイルを読み込みます。
    //!
    //!               失敗した場合、 false を返します。
    //!
    //! @param[out]   readSize  実際に読み込まれたデータのサイズを受け取る変数へのポインタです。
    //!                         サイズを受け取る必要がない場合は NULL を指定できます。
    //! @param[in]    handle    ファイルハンドルです。
    //! @param[out]   buf       読み込んだデータを受け取るバッファへのポインタです。
    //! @param[in]    size      読み込むサイズです。
    //!
    //! @return       成功した場合、 true を返します。
    //---------------------------------------------------------------------------
    bool TryRead(u32* readSize, FileHandle* handle, u8* buf, u32 size);

    //---------------------------------------------------------------------------
    //! @brief        読み書き位置を移動します。
    //!
    //!               失敗した場合、 ASSERT します。
    //!
    //! @param[in]    handle    ファイルハンドルです。
    //! @param[in]    offset    移動量です。
    //! @param[in]    origin    基準位置です。
    //!
    //! @return       成功した場合、 true を返します。
    //---------------------------------------------------------------------------
    bool Seek(FileHandle* handle, s32 offset, FileHandle::SeekOrigin origin)
    {
        bool success = TrySeek(handle, offset, origin);
        NW_ASSERTMSG(success, "file Seek error");
        return success;
    }

    //---------------------------------------------------------------------------
    //! @brief        読み書き位置を移動します。
    //!
    //!               失敗した場合、 false を返します。
    //!
    //! @param[in]    handle    ファイルハンドルです。
    //! @param[in]    offset    移動量です。
    //! @param[in]    origin    基準位置です。
    //!
    //! @return       成功した場合、 true を返します。
    //---------------------------------------------------------------------------
    bool TrySeek(FileHandle* handle, s32 offset, FileHandle::SeekOrigin origin);

    //---------------------------------------------------------------------------
    //! @brief        読み書き位置を取得します。
    //!
    //!               失敗した場合、 ASSERT します。
    //!
    //! @param[in]    handle    ファイルハンドルです。
    //!
    //! @return       読み書き位置を返します。
    //---------------------------------------------------------------------------
    u32 GetCurrentSeekPos(FileHandle* handle)
    {
        u32 pos = 0;
        bool success = TryGetCurrentSeekPos(&pos, handle);
        NW_ASSERTMSG(success, "GetCurrentSeekPos error");
        return pos;
    }

    //---------------------------------------------------------------------------
    //! @brief        読み書き位置を取得します。
    //!
    //!               失敗した場合、 false を返します。
    //!
    //! @param[out]   pos       読み書き位置を受け取る変数へのポインタです。
    //!                         NULL は指定できません。
    //! @param[in]    handle    ファイルハンドルです。
    //!
    //! @return       成功した場合、 true を返します。
    //---------------------------------------------------------------------------
    bool TryGetCurrentSeekPos(u32* pos, FileHandle* handle);

    //---------------------------------------------------------------------------
    //! @brief        ファイルサイズを取得します。
    //!
    //!               失敗した場合、 ASSERT します。
    //!
    //! @param[in]    handle    ファイルハンドルです。
    //!
    //! @return       ファイルサイズを返します。
    //---------------------------------------------------------------------------
    u32 GetFileSize(FileHandle* handle)
    {
        u32 size = 0;
        bool success = TryGetFileSize(&size, handle);
        NW_ASSERTMSG(success, "GetFileSize error");
        return size;
    }

    //---------------------------------------------------------------------------
    //! @brief        ファイルサイズを取得します。
    //!
    //!               失敗した場合、 false を返します。
    //!
    //! @param[out]   size      取得したサイズを受け取る変数へのポインタです。
    //!                         NULL は指定できません。
    //! @param[in]    handle    ファイルハンドルです。
    //!
    //! @return       成功した場合、 true を返します。
    //---------------------------------------------------------------------------
    bool TryGetFileSize(u32* size, FileHandle* handle);

    //---------------------------------------------------------------------------
    //! @brief        ディレクトリを開きます。
    //!
    //!               ディレクトリが無い場合、 ASSERT します。
    //!
    //! @param[out]   handle    ディレクトリハンドルを受け取る変数へのポインタです。
    //! @param[in]    dirName   開くディレクトリ名です。
    //!
    //! @return       成功した場合、ディレクトリを開くのに用いたデバイスへのポインタを返します。
    //---------------------------------------------------------------------------
    FileDevice* OpenDirectory(DirectoryHandle* handle, const char* dirName)
    {
        FileDevice* result = TryOpenDirectory(handle, dirName);
        NW_ASSERTMSG(result, "open directory failed. [%s]", dirName);
        return result;
    }

    //---------------------------------------------------------------------------
    //! @brief        ディレクトリを開きます。
    //!
    //!               ディレクトリが無い場合、 NULL を返します。
    //!
    //! @param[out]   handle    ディレクトリハンドルを受け取る変数へのポインタです。
    //! @param[in]    dirName   開くディレクトリ名です。
    //!
    //! @return       成功した場合、ディレクトリを開くのに用いたデバイスへのポインタを返します。
    //!               失敗した場合、 NULL を返します。
    //---------------------------------------------------------------------------
    FileDevice* TryOpenDirectory(DirectoryHandle* handle, const char* dirName);

    //---------------------------------------------------------------------------
    //! @brief        ディレクトリを閉じます。
    //!
    //! @param[in]    handle    ディレクトリハンドルです。
    //!
    //! @return       成功した場合、true を返します。
    //---------------------------------------------------------------------------
    bool CloseDirectory(DirectoryHandle* handle);

    //---------------------------------------------------------------------------
    //! @brief        ディレクトリ情報を読み込みます。
    //!
    //!               失敗した場合、 ASSERT します。
    //!
    //! @param[in]    handle    ディレクトリハンドルです。
    //! @param[out]   entry     読み込んだディレクトリ情報を入れる場所です。
    //! @param[in]    num       読み込むディレクトリ情報の個数です。
    //!
    //! @return       実際に読み込まれたディレクトリ情報の個数を返します。
    //---------------------------------------------------------------------------
    u32 ReadDirectory(DirectoryHandle* handle, DirectoryEntry entry[], u32 num)
    {
        u32 readNum = 0;
        bool success = TryReadDirectory(&readNum, handle, entry, num);
        NW_ASSERTMSG(success, "read directory failed");
        return readNum;
    }

    //---------------------------------------------------------------------------
    //! @brief        ディレクトリ情報を読み込みます。
    //!
    //!               失敗した場合、 false を返します。
    //!
    //! @param[out]   readNum   実際に読み込まれたディレクトリ情報の個数を受け取る変数へのポインタです。
    //!                         サイズを受け取る必要がない場合は NULL を指定できます。
    //! @param[in]    handle    ディレクトリハンドルです。
    //! @param[out]   entry     読み込んだディレクトリ情報を入れる場所です。
    //! @param[in]    num       読み込むディレクトリ情報の個数です。
    //!
    //! @return       成功した場合、 true を返します。
    //---------------------------------------------------------------------------
    bool TryReadDirectory(u32* readNum, DirectoryHandle* handle, DirectoryEntry entry[], u32 num);

    //---------------------------------------------------------------------------
    //! @brief        カレントディレクトリを変更します。
    //!
    //! @param[out]   path      新しいカレントディレクトリです。
    //!
    //! @return       成功した場合、 true を返します。
    //---------------------------------------------------------------------------
    static bool SetCurrentDir(const char* path);


protected:
    struct FileHandleInner;
    struct DirHandleInner;

    //! @brief        ファイルをロードします。
    //! @param[in]    arg 読み込み設定です。
    //! @return       読み込んだデータへのポインタを返します。
    u8* DoLoad(LoadArg& arg);

    //! @brief        ファイルを開きます。
    //! @param[in]    handle ファイルハンドルです。
    //! @param[in]    fileName ファイル名です。
    //! @param[in]    flag ファイルの開き方を示すフラグです。
    //! @return       ファイルデバイスを返します。
    FileDevice* DoOpen(FileHandle* handle, const char* fileName, FileOpenFlag flag);

    //! @brief        ファイルを閉じます。
    //! @param[in]    handle ファイルハンドルです。
    //! @return       ファイルを閉じるのに成功した場合は true を返します。
    bool DoClose(FileHandle* handle);

    //! @brief        ファイルを読み込みます。
    //! @param[out]   readSize 読み込みに成功した要素数です。
    //! @param[in]    handle ファイルハンドルです。
    //! @param[out]   buf 読み込み先になるデータ配列へのポインタです。
    //! @param[in]    size 読み込むデータ配列の数です。
    //! @return       ファイルの読み込みに成功した場合は true を返します。
    bool DoRead(u32* readSize, FileHandle* handle, u8* buf, u32 size);

    //! @brief        読み書き位置を移動します。
    //! @param[in]    handle ファイルハンドルです。
    //! @param[in]    offset 移動量です。
    //! @param[in]    origin 移動の基準となる位置です。
    //! @return       移動に成功した場合は true を返します。
    bool DoSeek(FileHandle* handle, s32 offset, FileHandle::SeekOrigin origin);

    //! @brief        読み書き位置を取得します。
    //! @param[out]   pos 取得した位置を出力します。
    //! @param[in]    handle ファイルハンドルです。
    //! @return       常に true を返します。
    bool DoGetCurrentSeekPos(u32* pos, FileHandle* handle);

    //! @brief        ファイルサイズを取得します。
    //! @param[out]   size 取得したファイルサイズを出力します。
    //! @param[in]    handle ファイルハンドルです。
    //! @return       ファイルサイズの取得に成功した場合は true を返します。
    bool DoGetFileSize(u32* size, FileHandle* handle);

    //! @brief        ディレクトリを開きます。
    //! @param[in]    handle ディレクトリハンドルです。
    //! @param[in]    dirName ディレクトリ名です。
    //! @return       ファイルデバイスです。
    FileDevice* DoOpenDirectory(DirectoryHandle* handle, const char* dirName);

    //! @brief        ディレクトリを閉じます。
    //! @param[in]    handle ディレクトリハンドルです。
    //! @return       ディレクトリを閉じるのに成功した場合は true を返します。
    bool DoCloseDirectory(DirectoryHandle* handle);

    //! @brief        ディレクトリ情報を読み込みます。
    //! @param[out]    readNum 読み込んだディレクトリエントリの数を格納するポインタです。。
    //! @param[in]     handle ディレクトリハンドルです。
    //! @param[out]    entry ディレクトリの中の要素です。
    //! @param[in,out] num 読み込むディレクトリエントリの数です。
    //! @return        ディレクトリ情報の読み込みに成功した場合は true を返します。
    bool DoReadDirectory(u32* readNum, DirectoryHandle* handle, DirectoryEntry entry[], u32 num);


    //! @briefprivate
    //! @details       ハンドルバッファを取得します。
    //! @param[in]     handle ハンドルです。
    //! @return        ハンドルバッファを返します。
    u8* GetHandleBaseHandleBuffer(HandleBase* handle) const
    {
        return handle->m_HandleBuffer;
    }

    //! @brief        デバイスを設定します。
    //! @param[in]    handle ハンドルです。
    //! @param[in]    device ファイルデバイスです。
    void SetHandleBaseFileDevice(HandleBase* handle, FileDevice* device) const
    {
        handle->m_Device = device;
    }

    //! @brief        ファイルハンドルから、FS のファイルハンドルを扱う FileHandleInner のポインタを取り出します。
    //! @param[in]    handle ファイルハンドルです。
    //! @return       FileHandleInner のポインタを返します。
    FileHandleInner* GetFileHandleInner(FileHandle* handle);

    //! @brief        ディレクトリハンドルから FS のディレクトリハンドルを扱う DirHandleInner のポインタを取り出します。
    //! @param[in]    handle ディレクトリハンドルです。
    //! @return       DirHandleInner のポインタを返します。
    DirHandleInner* GetDirHandleInner(DirectoryHandle* handle);

    nw::ut::FixedSafeString<MAX_FULLPATH_SIZE> m_CWDPath;       //!< カレントワークディレクトリのパスです。
    u32 m_LastRawError;                                         //!< 直前に発生したエラーコードです。
};

} // namespace dev
} // namespace nw

#endif // NW_DEV_FILE_DEVICE_H_
