﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_DEV_W_PAD_DEVICE_CAFE_H_
#define NW_DEV_W_PAD_DEVICE_CAFE_H_

#include <nw/ut/ut_Memory.h>
#if defined(CAFE_SDK_1_6)
#include <cafe/controllers/kpad/kpad.h>
#else
#include <cafe/pads/kpad/kpad.h>
#endif

namespace nw
{
namespace dev
{

//---------------------------------------------------------------------------
//! @brief        Cafe WPAD デバイス情報管理クラスです。
//!
//! @details :category     入力デバイス
//---------------------------------------------------------------------------
class WPadDeviceCafe
{
public:
    //! @brief KPAD 情報を表す構造体です。
    struct KPadInfo
    {
        KPADStatus status[KPAD_MAX_READ_BUFS];
        s32 lastReadLength;
        s32 lastReadError;
        u32 deviceType;

        //! @brief コンストラクタです。
        KPadInfo()
            : lastReadLength( 0 )
            , lastReadError( KPAD_READ_ERR_NO_CONTROLLER )
            , deviceType( WPAD_DEV_NOT_FOUND )
        {}
    };


#if defined(NW_DEV_PAD_ENABLE)
    //---------------------------------------------------------------------------
    //! @brief        インスタンスを取得します。
    //!
    //! @return       シングルトンのインスタンスを返します。
    //---------------------------------------------------------------------------
    static WPadDeviceCafe* GetInstance()
    {
        static WPadDeviceCafe s_Instance;
        return &s_Instance;
    }


    //---------------------------------------------------------------------------
    //! @brief        初期化処理です。
    //---------------------------------------------------------------------------
    void Initialize( nw::ut::IAllocator* allocator );

    //---------------------------------------------------------------------------
    //! @brief        終了処理です。
    //---------------------------------------------------------------------------
    void Finalize();

    //---------------------------------------------------------------------------
    //! @brief        デバイスを更新します。
    //---------------------------------------------------------------------------
    void Update();

    //---------------------------------------------------------------------------
    //! @brief        PADライブラリで読み込んだ値の情報を取得します。
    //!
    //! @param[in]    channel   取得するチャンネルです。
    //!
    //! @return       パッドの状態を返します。
    //---------------------------------------------------------------------------
    const KPadInfo& GetKPadInfo(s32 channel) const { return m_KPadInfos[channel]; }


protected:
    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //---------------------------------------------------------------------------
    WPadDeviceCafe()
        : m_InitializedCount( 0 )
    {
        if (s_pInstance != NULL)
        {
            NW_FATAL_ERROR("nw::dev::WPadDeviceCafe instance already exists.");
        }
        s_pInstance = this;
    }
#else
    static WPadDeviceCafe* GetInstance() { return NULL; }

    void Initialize( nw::ut::IAllocator* /* allocator */) {}

    void Finalize() {}

    void Update() {}

    const KPadInfo& GetKPadInfo(s32) const { return m_KPadInfos[0]; }

protected:
    WPadDeviceCafe() {}
#endif // NW_DEV_PAD_ENABLE

    KPADUnifiedWpadStatus m_UniRingBufs[WPAD_MAX_CONTROLLERS * KPAD_MAX_READ_BUFS];
    KPadInfo m_KPadInfos[WPAD_MAX_CONTROLLERS];

    s32 m_InitializedCount;

    static WPadDeviceCafe* s_pInstance;
};

} // namespace dev
} // namespace nw

#endif // NW_DEV_W_PAD_DEVICE_CAFE_H_
