﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_DEV_PRIMITIVE_RENDERER_CAFE_H_
#define NW_DEV_PRIMITIVE_RENDERER_CAFE_H_

#include <nw/dev/dev_PrimitiveRendererBase.h>
#include <nw/gfnd/cafe/gfnd_TextureCafeGX2.h>
#include <cafe/gx2.h>

namespace nw
{
namespace dev
{

//---------------------------------------------------------------------------
//! @brief        Cafe 用の PrimitiveRenderer の具体的な実装クラスです。
//---------------------------------------------------------------------------
class PrimitiveRendererCafe : public PrimitiveRendererBase
{
public:
    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //!
    //! @param[in]    allocator   アロケーターです。
    //---------------------------------------------------------------------------
    /* ctor */ explicit PrimitiveRendererCafe( nw::ut::IAllocator* allocator );

    //---------------------------------------------------------------------------
    //! @brief        デストラクタです。
    //---------------------------------------------------------------------------
    virtual ~PrimitiveRendererCafe();

    //---------------------------------------------------------------------------
    //! @brief        描画のための準備を行います。
    //!
    //! @param[in]    allocator   アロケーターです。
    //! @param[in]    binaryData  シェーダバイナリへのポインタです。
    //! @param[in]    binarySize  シェーダバイナリのサイズです。
    //---------------------------------------------------------------------------
    virtual void InitializeFromBinaryImpl( nw::ut::IAllocator* allocator, const void* binaryData, u32 binarySize );

    //---------------------------------------------------------------------------
    //! @brief        描画のための準備を行います。
    //!
    //! @param[in]    allocator   アロケーターです。
    //! @param[in]    shaderPath  シェーダパスです。
    //---------------------------------------------------------------------------
    virtual void InitializeImpl( nw::ut::IAllocator* allocator, const char* shaderPath );

    //---------------------------------------------------------------------------
    //! @brief        頂点バッファをバインドします。
    //!
    //! @param[in]    buf         バッファの構造体です。
    //---------------------------------------------------------------------------
    virtual void CreateVertexBufferImpl( DrawBuffer& buf ) {}


    //----------------------------------------
    //! @name 行列設定
    //  @{

    //---------------------------------------------------------------------------
    //! @brief        PrimitiveRenderer で使うビュー行列を設定します。
    //!
    //! @param[in]    viewMtx     設定するビュー行列です。
    //---------------------------------------------------------------------------
    virtual void SetViewMtxImpl( const nw::math::MTX34* viewMtx ) { m_ViewMtx = viewMtx; }

    //---------------------------------------------------------------------------
    //! @brief        PrimitiveRenderer で使うプロジェクション行列を設定します。
    //!
    //! @param[in]    projectionMtx   設定するプロジェクション行列です。
    //---------------------------------------------------------------------------
    virtual void SetProjectionMtxImpl( const nw::math::MTX44* projectionMtx ) { m_ProjMtx = projectionMtx; }

    //---------------------------------------------------------------------------
    //! @brief        PrimitiveRenderer で使うビューポートを設定します。
    //!
    //! @param[in]    x           設定する x 座標です。
    //! @param[in]    y           設定する y 座標です。
    //! @param[in]    width       設定する幅です。
    //! @param[in]    height      設定する高さです。
    //---------------------------------------------------------------------------
    virtual void SetViewportImpl( const s32 x, const s32 y, const s32 width, const s32 height );

    //  @}


    //----------------------------------------
    //! @name 描画
    //  @{

    //---------------------------------------------------------------------------
    //! @brief        描画に必要な設定を行います。
    //---------------------------------------------------------------------------
    virtual void BeginImpl();

    //---------------------------------------------------------------------------
    //! @brief        描画を終了します。
    //---------------------------------------------------------------------------
    virtual void EndImpl();

    //---------------------------------------------------------------------------
    //! @brief        平面の三角形を描画します。
    //!
    //! @param[in]    modelMtx    モデル行列です。
    //! @param[in]    color       頂点色です。
    //---------------------------------------------------------------------------
    virtual void DrawTriangleImpl(
        const nw::math::MTX34& modelMtx,
        const nw::ut::Color4u8& color
    );

    //---------------------------------------------------------------------------
    //! @brief        バッファを指定して三角形を描画します。
    //!
    //! @param[in]    modelMtx    モデル行列です。
    //! @param[in]    color       頂点色です。
    //! @param[in]    buf         バッファの構造体です。
    //---------------------------------------------------------------------------
    virtual void DrawTriangleImpl(
        const nw::math::MTX34& modelMtx,
        const nw::ut::Color4u8& color,
        const DrawBuffer& buf
    );

    //---------------------------------------------------------------------------
    //! @brief        平面の四角形を描画します。
    //!
    //! @param[in]    modelMtx    モデル行列です。
    //! @param[in]    colorLeft   左側の点の頂点色です。
    //! @param[in]    colorRight  右側の点の頂点色です。
    //---------------------------------------------------------------------------
    virtual void DrawQuadImpl(
        const nw::math::MTX34& modelMtx,
        const nw::ut::Color4u8& colorLeft,
        const nw::ut::Color4u8& colorRight
    );

    //--------------------------------------------------------------------------
    //! @brief        テクスチャを張り込んだ平面の四角形を描画します。
    //!
    //! @param[in]    modelMtx    モデル行列です。
    //! @param[in]    texture     貼り付けるテクスチャです。
    //! @param[in]    colorLeft   左側の点の頂点色です。
    //! @param[in]    colorRight  右側の点の頂点色です。
    //! @param[in]    uvSrc       左下の点の UV 値です。
    //! @param[in]    uvSize      UV 値の範囲です(右上の点の UV 値 - 左下の点の UV 値)。
    //--------------------------------------------------------------------------
    virtual void DrawQuadImpl(
        const nw::math::MTX34& modelMtx,
        const nw::gfnd::Texture& texture,
        const nw::ut::Color4u8& colorLeft,
        const nw::ut::Color4u8& colorRight,
        const nw::math::VEC2& uvSrc,
        const nw::math::VEC2& uvSize
    );

    //---------------------------------------------------------------------------
    //! @brief        四角形の枠線を描画します。
    //!
    //! @param[in]    modelMtx    モデル行列です。
    //! @param[in]    colorLeft   左側の点の頂点色です。
    //! @param[in]    colorRight  右側の点の頂点色です。
    //---------------------------------------------------------------------------
    virtual void DrawBoxImpl(
        const nw::math::MTX34& modelMtx,
        const nw::ut::Color4u8& colorLeft,
        const nw::ut::Color4u8& colorRight
    );

    //---------------------------------------------------------------------------
    //! @brief        直方体を描画します。
    //!
    //! @param[in]    modelMtx    モデル行列です。
    //! @param[in]    color0      手前左上の頂点色です。
    //! @param[in]    color1      奥右下の頂点色です。
    //---------------------------------------------------------------------------
    virtual void DrawCubeImpl(
        const nw::math::MTX34& modelMtx,
        const nw::ut::Color4u8& color0,
        const nw::ut::Color4u8& color1
    );

    //--------------------------------------------------------------------------
    //! @brief        ワイヤーフレームの直方体を描画します。
    //!
    //! @param[in]    modelMtx    モデル行列です。
    //! @param[in]    color0      手前左上の頂点色です。
    //! @param[in]    color1      奥右下の頂点色です。
    //---------------------------------------------------------------------------
    virtual void DrawWireCubeImpl(
        const nw::math::MTX34& modelMtx,
        const nw::ut::Color4u8& color0,
        const nw::ut::Color4u8& color1
    );

    //---------------------------------------------------------------------------
    //! @brief        線分を描画します。
    //!
    //! @param[in]    modelMtx    モデル行列です。
    //! @param[in]    color0      1 つ目の頂点色です。
    //! @param[in]    color1      2 つ目の頂点色です。
    //! @param[in]    lineWidth   線分の幅です。
    //---------------------------------------------------------------------------
    virtual void DrawLineImpl(
        const nw::math::MTX34& modelMtx,
        const nw::ut::Color4u8& color0,
        const nw::ut::Color4u8& color1,
        const f32 lineWidth = 1.f
    );

    //---------------------------------------------------------------------------
    //! @brief        バッファを指定して線分を描画します。
    //!
    //! @param[in]    modelMtx    モデル行列です。
    //! @param[in]    color       頂点色です。
    //! @param[in]    buf         バッファの構造体です。
    //! @param[in]    lineWidth   線分の幅です。
    //---------------------------------------------------------------------------
    virtual void DrawLineImpl(
        const nw::math::MTX34& modelMtx,
        const nw::ut::Color4u8& color,
        const DrawBuffer& buf,
        const f32 lineWidth = 1.f
    );

    //---------------------------------------------------------------------------
    //! @brief        4 x 8 分割、直径 1 の球を描画します。
    //!
    //! @param[in]    modelMtx    モデル行列です。
    //! @param[in]    north       北極(y が + 側)の頂点色です。
    //! @param[in]    south       南極(y が - 側)の頂点色です。
    //---------------------------------------------------------------------------
    virtual void DrawSphere4x8Impl(
        const nw::math::MTX34& modelMtx,
        const nw::ut::Color4u8& north,
        const nw::ut::Color4u8& south
    );

    //---------------------------------------------------------------------------
    //! @brief        8 x 16 分割、直径 1 の球を描画します。
    //!
    //! @param[in]    modelMtx    モデル行列です。
    //! @param[in]    north       北極(y が + 側)の頂点色です。
    //! @param[in]    south       南極(y が - 側)の頂点色です。
    //---------------------------------------------------------------------------
    virtual void DrawSphere8x16Impl(
        const nw::math::MTX34& modelMtx,
        const nw::ut::Color4u8& north,
        const nw::ut::Color4u8& south
    );

    //---------------------------------------------------------------------------
    //! @brief        16 分割、直径 1 の円を描画します。
    //!
    //! @param[in]    modelMtx    モデル行列です。
    //! @param[in]    edge        円の色です。
    //---------------------------------------------------------------------------
    virtual void DrawCircle16Impl( const nw::math::MTX34& modelMtx, const nw::ut::Color4u8& edge );

    //---------------------------------------------------------------------------
    //! @brief        32 分割、直径 1 の円を描画します。
    //!
    //! @param[in]    modelMtx    モデル行列です。
    //! @param[in]    edge        円の色です。
    //---------------------------------------------------------------------------
    virtual void DrawCircle32Impl( const nw::math::MTX34& modelMtx, const nw::ut::Color4u8& edge );

    //  @}

private:
    void DrawTriangles(
        const nw::math::MTX34& modelMtx,
        const nw::ut::Color4u8& color0,
        const nw::ut::Color4u8& color1,
        Vertex* vertex,
        u32 vertexNum,
        u16* index,
        u32 indexNum,
        const GX2Texture* texture = NULL
    );

    void DrawLines(
        const nw::math::MTX34& modelMtx,
        const nw::ut::Color4u8& color0,
        const nw::ut::Color4u8& color1,
        GX2PrimitiveType lineType,
        Vertex* vertex,
        u32 vertexNum,
        u16* index,
        u32 indexNum,
        f32 lineWidth = 1.f
    );

    bool ReadVertexShader(
        nw::ut::IAllocator* allocator,
        GX2VertexShader**   ppShader,
        u32                 index,
        const void*         pData
    );

    bool ReadPixelShader(
        nw::ut::IAllocator* allocator,
        GX2PixelShader**    ppShader,
        u32                 index,
        const void*         pData
    );

    enum
    {
        ATTRIBUTE_NUM = 3
    };

    GX2VertexShader* m_VertexShader;    //!< 頂点シェーダーへのポインタ
    GX2PixelShader* m_PixelShader;      //!< ピクセルシェーダーへのポインタ
    void* m_VertexProgram;              //!< 頂点シェーダーのシェーダプログラムへのポインタ
    void* m_PixelProgram;               //!< ピクセルシェーダーのシェーダプログラムへのポインタ
    GX2FetchShader m_FetchShader;       //!< フェッチシェーダー
    void* m_FetchShaderBufPtr;
    GX2Sampler m_DrawQuadSampler;

    const nw::math::MTX34* m_ViewMtx;   //!< ビュー行列
    const nw::math::MTX44* m_ProjMtx;   //!< プロジェクション行列

    nw::ut::IAllocator* m_Allocator;    //!< アロケーター

    //----------------------------------------
    //! @name ユニフォーム／頂点属性位置
    //  @{

    u32 m_ParamWVPOffset;
    u32 m_ParamUserOffset;
    u32 m_ParamRateOffset;
    u32 m_ParamColor0Offset;
    u32 m_ParamColor1Offset;
    u32 m_ParamUVSrcOffset;
    u32 m_ParamUVSizeOffset;
    u32 m_ParamTexLocation;
    u32 m_AttrVertexLocation;
    u32 m_AttrTexCoord0Location;
    u32 m_AttrColorRateLocation;
    GX2AttribStream m_Attributes[ ATTRIBUTE_NUM ];

    //  @}

    //----------------------------------------
    //! @name バッファ
    //  @{

    u16* m_TriangleIndexBuf;
    Vertex* m_QuadVertexBuf;
    u16* m_QuadIndexBuf;
    u16* m_BoxIndexBuf;
    Vertex* m_LineVertexBuf;
    u16* m_LineIndexBuf;
    Vertex* m_CubeVertexBuf;
    u16* m_CubeIndexBuf;
    Vertex* m_WireCubeVertexBuf;
    u16* m_WireCubeIndexBuf;
    Vertex* m_SphereSVertexBuf;
    u16* m_SphereSIndexBuf;
    Vertex* m_SphereLVertexBuf;
    u16* m_SphereLIndexBuf;
    Vertex* m_DiskSVertexBuf;
    Vertex* m_DiskLVertexBuf;
    u16* m_CircleSIndexBuf;
    u16* m_CircleLIndexBuf;

    //  @}
};

} // namespace dev
} // namespace nw

#endif  // NW_DEV_PRIMITIVE_RENDERER_CAFE_H_

