﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_CTRL_BUTTON_BASE_H_
#define NW_CTRL_BUTTON_BASE_H_

#include <nw/ut/ut_BitFlag.h>

namespace nw
{
namespace ctrl
{

//---------------------------------------------------------------------------
//! @brief ボタンの状態遷移を抽象化したクラスです。
//!
//---------------------------------------------------------------------------
class ButtonBase
{
public:
    //! ステートを示します。
    enum State {
        STATE_OFF_IDLE,     //!< OFF状態で止まっている
        STATE_ON,           //!< ONアニメ再生中
        STATE_OFF,          //!< OFFアニメ再生中
        STATE_ON_IDLE,      //!< ON状態で止まっている
        STATE_DOWN,         //!< DOWNアニメ再生中
        STATE_DOWN_IDLE,    //!< DOWN状態で止まっている
        STATE_CANCEL        //!< CANCELアニメ再生中
    };

    /// ボタンへの操作を示します。
    enum Action {
        ACTION_ON,      //!< カーソルがボタンに乗ったときのアクション
        ACTION_OFF,     //!< カーソルがボタンから離れたときのアクション
        ACTION_DOWN,    //!< カーソルがボタンに乗った状態で決定ボタンを押したときのアクション
        ACTION_CANCEL   //!< 押されたままの状態になっているボタンを押されていない状態に戻すときのアクション
    };

    //----------------------------------------
    //! @name 構築／破棄
    //@{
    //! @brief コンストラクタです。
    //!
    //! @details
    //! 状態の初期化を行います。
    //!
    ButtonBase();

    //! @brief デストラクタです。デフォルトでは何もしません。
    //!
    virtual ~ButtonBase();
    //@}

    //----------------------------------------
    //! @name アクションの発行
    //@{
    //! @brief onアクションを発行します。
    //!
    //! @details
    //! 無効状態では、この関数は何もしません。
    //!
    virtual void On();

    //! @brief offアクションを発行します。
    //!
    //! @details
    //! 無効状態でも、このアクションは受け付けます。
    //!
    virtual void Off();

    //! @brief downアクションを発行します。
    //!
    //! @details
    //! 無効状態では、この関数は何もしません。
    //!
    virtual void Down();

    //! @brief cancelアクションを発行します。
    //!
    //! @details
    //! 例えばラジオボタンのように「複数あるうちの一つしか有効にならない」挙動の場合に、
    //! ボタンが押されたらそれ以外の全てのボタンにcancelを発行するというような用途で使用します。
    //!
    virtual void Cancel();
    //@}

    //----------------------------------------
    //! @name ステートの強制的な変更
    //@{
    //! @brief ボタンのステートを強制的にSTATE_OFF_IDLEに変更します。
    //!
    //! @details
    //! アクションキューは何も入っていない状態になります。
    //! ステートの強制的な変更を行うと、アニメが繋がらなくなる恐れがありますので、注意してご使用ください。
    //!
    //! なお、このメソッドはボタンの有効/無効に関わらずステートを変更することができます。
    //!
    virtual void ForceOff();

    //! @brief ボタンのステートを強制的にSTATE_ON_IDLEに変更します。
    //!
    //! @details
    //! その他の注意点等は@ref forceOffと同じです。
    //!
    virtual void ForceOn();

    //! @brief ボタンのステートを強制的にSTATE_DOWN_IDLEに変更します。
    //!
    //! @details
    //! その他の注意点等は@ref forceOffと同じです。
    //!
    virtual void ForceDown();
    //@}

    //----------------------------------------
    //! @name 状態の更新
    //@{
    //! @brief 状態の更新を行います。
    //!
    //! @details
    //! ゲームフレームを進めるたびに1回呼び出してください。
    //!
    virtual void Update();
    //@}

    //----------------------------------------
    //! @name 状態の設定／取得
    //@{
    //! @brief 現在の状態を取得します。
    //!
    //! @return 現在の状態
    //!
    State GetState() const { return m_State; }

    //! @brief 現在有効な状態か否かを取得します。
    //!
    //! @retval true    有効な状態。全てのアクションを受け付ける。
    //! @retval false   無効な状態。offアクションとcancelアクションを受け付ける。
    //!
    bool IsActive() const { return m_Flag.IsMaskOn(MASK_ACTIVE); }

    //! @brief 現在有効な状態か否かを設定します。
    //!
    //! @details
    //! ボタンが無効な状態とは、「ONアクション及びDOWNアクションを受け付けない状態」を示します。
    //! つまり、無効な状態でもOFFアクションは受け付けます。これはボタンにカーソルが当たった状態で
    //! 無効になったときに、その後自然にボタンをOFF状態に移行させるためです。
    //!
    //! 一切のアクションを受け付けないようにするには、SetAcceptActionAll()を使用してください。
    //!
    //! また、無効な状態でも、ForceOn()等の強制的な変更は受け付けます。
    //!
    //! @param[in] active   有効にするか否か
    //!
    virtual void SetActive(bool active);

    //! @brief 現在指定したアクションを受け付けるか否かを取得します。
    //!
    //! @details
    //! この設定はIsEnable/SetEnableで扱う状態とは独立であることに注意してください。
    //! つまり、ONアクションが受け付けられるには、(SetEnable() && IsAcceptAction(ACTION_ON))である必要があります。
    //!
    //! デフォルトでは全てのアクションを受け付けます。
    //!
    //! ただしForceOn()等の強制的な変更はこの設定に関係なく受け付けます。
    //!
    //! @param[in] action   受け付けるか調べたいアクション
    //!
    bool IsAcceptAction(Action action) const { return m_Flag.IsBitOn(action); }

    //! @brief 現在指定したアクションを受け付けるか否かを設定します。
    //!
    //! @details
    //! このメソッドを使用して部分的にアクションを受け付ける設定を変更すると、
    //! ボタンが不自然な挙動を示すことがあります。十分注意してご使用ください。
    //!
    //! @param[in] action   受け付けるかを指定したいアクション
    //! @param[in] accept   受け付けるか否か
    //!
    void SetAcceptAction(Action action, bool accept) { m_Flag.ChangeBit(action, accept); }

    //! @brief アクションを受け付けるか否かの設定を、全てのアクションに対して一度に設定します。
    //!
    //! @param[in] accept   アクションを受け付けるか否か
    //!
    void SetAcceptActionAll(bool accept)
    {
        m_Flag.ChangeMask(MASK_ACCEPT_ON | MASK_ACCEPT_OFF | MASK_ACCEPT_DOWN | MASK_ACCEPT_CANCEL, accept);
    }

    //! @brief ボタンのステートがSTATE_DOWNまたはSTATE_DOWN_IDLEであるか、またはアクションキューにDOWNアクションが入っているかを取得します。
    //!
    //! @details
    //! この関数がtrueを返すということは、ForceOff等で強制的にステートが変更されない限り、そのボタンは将来的に押された状態になる
    //! ということを示します。
    //!
    bool IsDowning() const;
    //@}

protected:
    enum {
        FLAG_BIT_MAX = 4    //!< m_Flagで使用しているビットの最大値
    };

    enum FlagMask {
        MASK_ACCEPT_ON = 1 << 0,
        MASK_ACCEPT_OFF = 1 << 1,
        MASK_ACCEPT_DOWN = 1 << 2,
        MASK_ACCEPT_CANCEL = 1 << 3,
        MASK_ACTIVE = 1 << 4,
    };

    //! ボタンのアクションを管理するためのキューです。
    class ActionQueue
    {
    public:
        enum {
            QUEUE_SIZE = 4
        };

        //! コンストラクタです。要素0個で初期化します。
        ActionQueue() : m_ActionNum(0) {}

        //! キューを空にします。
        void Clear()
        {
            m_ActionNum = 0;
        }

        //! キューの中にDOWNアクションがあるか調査します。
        bool IsDownExist() const;

        //! キューの先頭の要素を取得します。
        Action Peek() const
        {
            NW_ASSERT( ! IsEmpty());
            return m_Actions[0];
        }

        //! キューが空か否かを取得します。
        bool IsEmpty() const
        {
            return (m_ActionNum == 0);
        }

        //! キューがいっぱいか否かを取得します。
        bool IsFull() const
        {
            return (m_ActionNum >= QUEUE_SIZE);
        }

        //! キューの最後尾に要素を追加し、処理しても処理しなくても同じことになるアクションを削除します。
        void PushWithOmit(Action action);

        //! キューの先頭から要素を削除します。
        void Pop();

    protected:
        Action m_Actions[QUEUE_SIZE];
        int m_ActionNum;

    };

    //! Onアクションに対する処理を記述します。trueを返したときは、処理したアクションをキューからpopすることを示します。
    virtual bool ProcessOn();
    //! Offアクションに対する処理を記述します。trueを返したときは、処理したアクションをキューからpopすることを示します。
    virtual bool ProcessOff();
    //! Downアクションに対する処理を記述します。trueを返したときは、処理したアクションをキューからpopすることを示します。
    virtual bool ProcessDown();
    //! Cancelアクションに対する処理を記述します。trueを返したときは、処理したアクションをキューからpopすることを示します。
    virtual bool ProcessCancel();

    //! STATE_ONの状態を進めます。trueを返したときは、STATE_ON状態を終了し、STATE_ON_IDLEに移ります。
    virtual bool UpdateOn();
    //! STATE_OFFの状態を進めます。trueを返したときは、STATE_OFF状態を終了し、STATE_OFF_IDLEに移ります。
    virtual bool UpdateOff();
    //! STATE_DOWNの状態を進めます。trueを返したときは、STATE_DOWN状態を終了し、STATE_DOWN_IDLEに移ります。
    virtual bool UpdateDown();
    //! STATE_CANCELの状態を進めます。trueを返したときは、STATE_CANCEL状態を終了し、STATE_OFF_IDLEに移ります。
    virtual bool UpdateCancel();

    //! STATE_ONが始まったときに呼び出されるメソッドです。デフォルトでは何も行いません。
    virtual void StartOn();
    //! STATE_OFFが始まったときに呼び出されるメソッドです。デフォルトでは何も行いません。
    virtual void StartOff();
    //! STATE_DOWNが始まったときに呼び出されるメソッドです。デフォルトでは何も行いません。
    virtual void StartDown();
    //! STATE_CANCELが始まったときに呼び出されるメソッドです。デフォルトでは何も行いません。
    virtual void StartCancel();

    //! STATE_ONが終わったときに呼び出されるメソッドです。デフォルトではステートをSTATE_ON_IDLEに移行します。
    virtual void FinishOn();
    //! STATE_OFFが終わったときに呼び出されるメソッドです。デフォルトではステートをSTATE_OFF_IDLEに移行します。
    virtual void FinishOff();
    //! STATE_DOWNが終わったときに呼び出されるメソッドです。デフォルトではステートをSTATE_DOWN_IDLEに移行します。
    virtual void FinishDown();
    //! STATE_CANCELが終わったときに呼び出されるメソッドです。デフォルトではステートをSTATE_OFF_IDLEに移行します。
    virtual void FinishCancel();

    //! ステートを変更します。
    virtual void ChangeState(State state);
    //! ステートを強制的に変更した場合は、ChangeStateではなくこちらが呼ばれます。
    virtual void ForceChangeState(State state);

    //! アクションキューから要素を一つ取出し、処理します。キューにアクションがない場合は何もしません。
    void ProcessActionFromQueue();

    State m_State;
    ut::BitFlag32 m_Flag;
    ActionQueue m_ActionQueue;

};

} // namespace nw::ctrl
} // namespace nw

#endif // NW_CTRL_BUTTON_BASE_H_
