﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_CTRL_ANIM_BUTTON_H_
#define NW_CTRL_ANIM_BUTTON_H_

#include <nw/ctrl/ctrl_ButtonBase.h>
#include <nw/ut/ut_LinkList.h>
#include <nw/math/math_Types.h>
#include <nw/ut/ut_RuntimeTypeInfo.h>

namespace nw
{

namespace lyt
{
class Animator;
class ControlSrc;
class Layout;
class Pane;
}

namespace ctrl
{

//---------------------------------------------------------------------------
//! @brief ButtonBaseに、アニメーションやあたり判定、ButtonGroupとの連携等の機能を追加したクラスです。
//!
//! @details
//! ctrlモジュールのボタンの基底としては、ButtonBaseではなくこちらを使用してください。
//!
//---------------------------------------------------------------------------
class AnimButton : public ButtonBase
{
public:
    //! 実行時型情報です。
    NW_UT_RUNTIME_TYPEINFO_ROOT();

    //! @brief コンストラクタです。
    //!
    AnimButton();

    //! @brief ボタンの構築を行います。
    //!
    //! @param[in] controlSrc   コントロールの情報
    //! @param[in,out] layout   所属するレイアウト
    //!
    void Build(const lyt::ControlSrc& controlSrc, lyt::Layout* layout);

    //! @brief ボタン名の取得
    //!
    //! @details
    //! ボタンの名前は、ボタンが所属するレイアウトが部品レイアウトの場合は
    //! 部品ペイン名、部品レイアウトでない場合はレイアウト名に等しくなります。
    //!
    const char* GetName() const { return m_Name; }

    //----------------------------------------
    //! @name 当たり判定
    //@{
    //! @brief 当たり判定の範囲を更新します。
    //!
    //! @details
    //! この処理はcalcでは行われないため、明示的に呼び出す必要があります。
    //! ただし、通常はButtonGroupのcalcで呼ばれますので、個別に行う必要はありません。
    //! ButtonGroupのオプションで、更新を行うか否かを設定できるようになっています。
    //!
    //! 大きさが大きく変わらないボタンであれば、当たりの範囲を更新しないことにより、
    //! 処理を節約することができます。
    //!
    //! この処理はボタンの当たりを示すペインのGlobalMatrixを使用するため、
    //! 一回以上LayoutのCalculateMtxを行ってから呼び出すようにしてください。
    //!
    //! なお、このクラスでは、回転しているペインの当たり判定には対応していません。
    //! 回転しているペインの当たり判定に対応するには、このメソッド及びHitTest
    //! メソッドをオーバーライドする必要があります。
    //!
    virtual void UpdateHitBox();

    //! @brief 指定した位置にボタンが当たっているか否かを取得します。
    //!
    virtual bool HitTest(const math::VEC2& vec2) const;

    //! @brief 当たりの大きさを示すペインを取得します。
    //!
    lyt::Pane* GetHitPane() const { return m_HitPane; }

    //! @brief 当たりの範囲の左下の位置（最小点）を取得します。
    //!
    const math::VEC2& GetHitBoxBottomLeft() const { return m_HitBoxBottomLeft; }

    //! @brief 当たりの範囲の右上の位置（最大点）を取得します。
    //!
    const math::VEC2& GetHitBoxTopRight() const { return m_HitBoxTopRight; }
    //@}

    //----------------------------------------
    //! @name タグ
    //@{
    //! @brief タグの値を取得します。
    //!
    int GetTag() const { return m_Tag; }

    //! @brief タグの値を取得します。
    //!
    //! @details
    //! タグとは、各ボタンに一つ設定できる整数値です。
    //! タグはボタンの内部では一切使用されませんので、ボタンを使う側が使い方を決めることができます。
    //!
    //! 主にボタンの判別を行いやすくする用途を想定しています。
    //!
    //! 初期値は0になっています。
    //!
    void SetTag(int tag) { m_Tag = tag; }
    //@}

    //----------------------------------------
    //! @name ボタンを押したときの排他処理
    //@{
    //! @brief ボタンが押されたときに、同じボタングループに属するボタンが押されることを防ぐか否かを取得します。
    //!
    //! @details
    //! この設定は、AnimButton内では使用せず、ButtonGroupから使用されます。
    //!
    //! 押した後に画面が切り替わるボタンなど、同時に複数のボタンが押されて欲しくない場合に使用する機能です。
    //!
    //! @return
    //! trueのときは、同じボタングループに属するボタンが押されることを防ぎます。
    //!
    bool IsExcludeDown() const { return m_Flag.IsMaskOn(MASK_EXCLUDE_DOWN); }

    //! @brief ボタンが押されたときに、同じボタングループに属するボタンが押されることを防ぐか否かを設定します。
    //!
    //! @details
    //! デフォルトではfalse（同じボタングループに属するボタンが押されることを防がない）です。
    //!
    //! @param[in] excludeDown  trueを与えると、同じボタングループに属するボタンが押されることを防ぎます。
    //!
    void SetExcludeDown(bool excludeDown) { m_Flag.ChangeMask(MASK_EXCLUDE_DOWN, excludeDown); }
    //@}

    //----------------------------------------
    //! @name ドラッグに関する機能
    //@{
    //! ドラッグのモードです。
    enum DragMode {
        DRAG_MODE_DISABLE,  //!< ドラッグは無効です。
        DRAG_MODE_ON_DOWN,  //!< ボタンが押されたときにドラッグ状態になります。
        DRAG_MODE_ON_HIT    //!< ボタンに当たったときにドラッグ状態になります。
    };

    //! @brief ドラッグのモードを取得します。
    //!
    //! @details
    //! この設定は、AnimButton内では使用せず、ButtonGroupから使用されます。
    //!
    //! @return
    //! ドラッグのモード
    //!
    DragMode GetDragMode() const { return static_cast<DragMode>((m_Flag.GetDirect() >> BIT_DRAG_MODE_START) & 0x3); }

    //! @brief ドラッグのモードを設定します。
    //!
    //! @details
    //! この設定は、AnimButton内では使用せず、ButtonGroupから使用されます。
    //!
    //! デフォルトではDRAG_MODE_DISABLE（ドラッグが無効）です。
    //!
    //! @param[in] mode ドラッグのモード
    //!
    void SetDragMode(DragMode mode) { m_Flag.SetDirect((m_Flag.GetDirect() & ~MASK_DRAG_MODE) | (mode << BIT_DRAG_MODE_START)) ; }

    //! @brief ドラッグ時の位置を初期化します。
    //!
    //! @details
    //! このクラスでは空の実装となっています。ドラッグを使用する場合は、継承先でオーバーライド
    //! する必要があります。
    //!
    //! @param[in] pos  現在ドラッグしている位置です。
    //!
    virtual void InitDragPosition(const math::VEC2& pos);

    //! @brief ドラッグ時の位置を更新します。
    //!
    //! @details
    //! このクラスでは空の実装となっています。ドラッグを使用する場合は、継承先でオーバーライド
    //! する必要があります。
    //!
    //! @param[in] pos  現在ドラッグしている位置です。ポインタが画面から外れている場合はNULLとなります。
    //!
    virtual void UpdateDragPosition(const math::VEC2* pos);
    //@}

    //----------------------------------------
    //! @name コールバックの設定
    //@{
    //! @brief ボタンのステートが変化したときのコールバックです。
    //!
    //! @detail
    //! このコールバック内では、ボタンのステートの変更(ForceXXX)は行わないでください。
    //! ステートを変更しても、その後の処理でステートが移ってしまう可能性があるためです。
    //!
    //! @param[in,out] button   押されたボタン
    //! @param[in] prevState    変化前のステート
    //! @param[in] nextState    変化後のステート
    //! @param[in,out] param    ボタンが押されたときに設定したパラメータ
    //!
    typedef void (*StateChangeCallback)(AnimButton* button, State prevState, State nextState, void* param);

    //! @brief ボタンのステートが変化したときのコールバックを設定します。
    //!
    //! @detail
    //! なお、このコールバックは、ForceXXXメソッドで強制的にステートを変更した場合には呼び出されません。
    //!
    //! コールバックを解除するには、callbackにNULLを与えてください。
    //!
    //! @param[in] callback 呼び出したいコールバック
    //! @param[in] param    コールバックに与えるパラメータ
    //!
    void SetStateChangeCallback(StateChangeCallback callback, void* param);
    //@}

    //----------------------------------------
    //! @name アニメの管理
    //@{
    //! @brief 無効アニメを再生します。
    //!
    //! @details
    //! 無効アニメを再生します。無効アニメの再生は、ボタンの有効・無効状態とは独立であることに
    //! ご注意ください。ボタンを無効にすると同時に無効アニメを再生したい場合は、SetActive(false)
    //! とPlayDisableAnim(true)を両方呼び出してください。
    //!
    //! アニメの再生は、内部的にPlayFromCurrentで行われているため、既に無効アニメ状態にした後に
    //! PlayDisableAnim(true)を呼び出しても、状態は変わらないことにご注意ください。
    //!
    //! @param[in] is_disable   trueの場合はDisableアニメが順再生されて無効アニメ状態になり、
    //!                         falseの場合は逆再生されて有効アニメ状態になります。
    //!
    virtual void PlayDisableAnim(bool is_disable);

    //! @brief ボタンのOn/Offアニメを取得します。
    //!
    //! @return アニメ
    //!
    lyt::Animator* GetOnOffAnimator() const { return m_OnOffAnimator; }

    //! @brief ボタンのDownアニメを取得します。
    //!
    //! @return アニメ
    //!
    lyt::Animator* GetDownAnimator() const { return m_DownAnimator; }

    //! @brief ボタンのCancelアニメを取得します。
    //!
    //! @return アニメ
    //!
    lyt::Animator* GetCancelAnimator() const { return m_CancelAnimator; }

    //! @brief ボタンのDisableアニメを取得します。
    //!
    //! @return アニメ
    //!
    lyt::Animator* GetDisableAnimator() const { return m_DisableAnimator; }

    //! @brief ボタンのOn/Offアニメを設定します。
    //!
    //! @param[in] animator 設定するアニメ
    //!
    void SetOnOffAnimator(lyt::Animator* animator) { m_OnOffAnimator = animator; }

    //! @brief ボタンのDownアニメを設定します。
    //!
    //! @param[in] animator 設定するアニメ
    //!
    void SetDownAnimator(lyt::Animator* animator) { m_DownAnimator = animator; }

    //! @brief ボタンのCancelアニメを設定します。
    //!
    //! @param[in] animator 設定するアニメ
    //!
    void SetCancelAnimator(lyt::Animator* animator) { m_CancelAnimator = animator; }

    //! @brief ボタンのDisableアニメを設定します。
    //!
    //! @param[in] animator 設定するアニメ
    //!
    void SetDisableAnimator(lyt::Animator* animator) { m_DisableAnimator = animator; }

    //! @brief ボタンが保持している全てのアニメを無効にします。
    //!
    //! @details
    //! これは、GetXXXAnimatorで全てのアニメを取得し、それぞれにSetEnable(false)を呼ぶのと同じ動作を行います。
    //! 通常は使う必要はありませんが、ボタンの通常の動作を止め、別なアニメを再生したいとき等に有効です。
    //! なお、ボタンのアニメは必要になったら自動的に有効化されますので、状態を戻す必要はありません。
    //!
    virtual void SetAllAnimatorDisable();
    //@}

    // 強制的にステートを変更したときにアニメの状態を変更するためにオーバーライドしています。
    virtual void ForceOff();
    virtual void ForceOn();
    virtual void ForceDown();

    //! @details :private
    // ButtonGroupが管理する際に用いるリンクです。
    ut::LinkListNode m_Link;

protected:
    enum {
        FLAG_BIT_MAX = 6    //!< m_Flagで使用しているビットの最大値
    };

    enum FlagMask {
        //! このボタンが押されたときに、同じボタングループ内の他のボタンが押されるのを妨げるか否か
        MASK_EXCLUDE_DOWN = 1 << (ButtonBase::FLAG_BIT_MAX + 1),
        //! このボタンのドラッグモードを保持するビットの開始点
        BIT_DRAG_MODE_START = ButtonBase::FLAG_BIT_MAX + 2,
        //! このボタンのドラッグモードを保持するビットのマスク
        MASK_DRAG_MODE = 0x3 << BIT_DRAG_MODE_START
    };

    StateChangeCallback m_StateChangeCallback;
    void* m_StateChangeCallbackParam;
    lyt::Animator* m_OnOffAnimator;
    lyt::Animator* m_DownAnimator;
    lyt::Animator* m_CancelAnimator;
    lyt::Animator* m_DisableAnimator;
    lyt::Pane* m_HitPane;
    math::VEC2 m_HitBoxBottomLeft;
    math::VEC2 m_HitBoxTopRight;
    int m_Tag;
    const char* m_Name;

    // AnimButtonの機能を実装するためのオーバーライドです。
    virtual void StartOn();
    virtual void StartOff();
    virtual void StartDown();
    virtual void StartCancel();

    virtual bool UpdateOn();
    virtual bool UpdateOff();
    virtual bool UpdateDown();
    virtual bool UpdateCancel();

    virtual bool ProcessCancel();

    virtual void ChangeState(State state);

    ///! 指定したアニメを有効化し、それ以外を無効化します。
    void EnableAnim(lyt::Animator* animator);

};

} // namespace nw::ctrl
} // namespace nw

#endif // NW_CTRL_ANIM_BUTTON_H_
