﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief ウェブページを開くのに利用される引数の宣言
 */

#pragma once

#include <nn/nn_Result.h>
#include <nn/applet/applet.h>
#include <nn/web/web_ShowArg.h>
#include <nn/web/web_Types.h>

namespace nn { namespace web {

/**
 * @brief ニュースページを開くための情報を構築するクラスです。
 */
class ShowNewsPageArg
{
public:
    /**
     * @brief NULL 終端を含むリクエストできるニュースページ用 URL の最大バイト数です。
     */
    static const size_t RequestUrlMaxLength = 3072;

    /**
     * @brief NULL 終端を含むコールバックとして指定できるニュースページ用 URL の最大バイト数です。
     */
    static const size_t CallbackUrlMaxLength = 384;

    /**
     * @brief NULL 終端を含むコールバック可能として指定できるニュースページ用 URL の最大バイト数です。
     */
    static const size_t CallbackableUrlMaxLength = 384;

    /**
     * @brief NULL 終端を含むホワイトリストの最大バイト数です。
     */
    static const size_t WhitelistMaxLength = 4096;

    /**
     * @brief ニュースページを開くための最低限の情報を持ったクラスを作成します。
     *
     * @param[in] pRequestUrl    開くページの URL を指定してください。
     *
     * @details 指定する値は NULL 終端されている必要があります。
     *          RequestUrlMaxLength -1 文字以下にする必要があります。
     */
    explicit ShowNewsPageArg(const char* pRequestUrl) NN_NOEXCEPT;

    /**
     * @brief 動画専用起動とするかどうかを指定します。
     *
     * @param[in]   bootAsMediaPlayer  動画専用起動とするかどうか
     *
     * @details 読み込み先が動画ファイルであり、ウェブページを介さずに
     *          直接動画再生を開始したい場合に true を指定してください。
     */
    void SetBootAsMediaPlayer(bool bootAsMediaPlayer) NN_NOEXCEPT;

    /**
     * @brief コールバック URL を設定します。
     *
     * @param[in]   pCallbackUrl    コールバックURLを指定してください
     *
     * @details 指定する値は NULL 終端されている必要があります。
     *          CallbackNewsPageUrlMaxLength -1 文字以下にする必要があります。
     *          二度呼び出した場合、値は上書きされます。
     */
    void SetCallbackUrl(const char* pCallbackUrl) NN_NOEXCEPT;

    /**
     * @brief コールバック可能 URL を設定します。
     *
     * @param[in]   pCallbackableUrl    コールバック可能 URL を指定してください。
     *
     * @details 指定する値は NULL 終端されている必要があります。
     *          CallbackableNewsPageUrlMaxLength -1 文字以下にする必要があります。
     *          二度呼び出した場合、値は上書きされます。
     */
    void SetCallbackableUrl(const char* pCallbackableUrl) NN_NOEXCEPT;

    /**
     * @brief ホワイトリストを設定します。
     *
     * @param[in]   pWhitelist  ホワイトリストを指定してください。
     *
     * @details 指定する値は NULL 終端されている必要があります。
     *          WhitelistMaxLength -1 文字以下にする必要があります。
     *          二度呼び出した場合、値は上書きされます。
     */
    void SetWhitelist(const char* pWhitelist) NN_NOEXCEPT;

    /**
     * @brief 左スティックモードをポインターに切り替える事を許可するかどうかを設定します。
     *
     * @param[in]   pointerEnabled  左スティックモードをポインターに切り替える事を許可する場合は true を指定してください。
     *
     * @details デフォルト値は true です。
     *          二度呼び出した場合、値は上書きされます。
     */
    void SetPointerEnabled(const bool pointerEnabled) NN_NOEXCEPT;

    /**
     * @brief 左スティックのモードを設定します。
     *
     * @param[in]   mode    左スティックのモードを指定してください。
     *
     * @details 二度呼び出した場合、値は上書きされます。
     */
    void SetLeftStickMode(const WebLeftStickMode mode) NN_NOEXCEPT;

    /**
     * @brief Uid を設定します。
     *
     * @param[in]   uid アカウントの Uid を指定してください。
     *
     * @details 指定する値は、アカウントシステムに登録されているユーザーの Uid または
     *          nn::account::InvalidUid である必要があります。
     *          呼び出さなかった場合または nn::account::InvalidUid を指定した場合は
     *          アノニマスモードになります。
     *          二度呼び出した場合、値は上書きされます。
     */
    void SetUid(const nn::account::Uid& uid) NN_NOEXCEPT;

    /**
     * @brief (廃止予定) フッターを利用するかどうかを設定します。
     * @deprecated 「SetFooterEnabled」は廃止予定です。
     */
    NN_DEPRECATED void SetFooterEnabled(const bool footerEnabled) NN_NOEXCEPT;

    /**
     * @brief ページの背景を設定します。
     *
     * @param[in]   backgroundKind  背景を指定してください。
     *
     * @details 二度呼び出した場合、値は上書きされます。
     */
    void SetBackgroundKind(const WebBackgroundKind backgroundKind) NN_NOEXCEPT;

    /**
     * @brief 起動画面種類を指定します。
     *
     * @param[in]   bootDisplayKind  起動画面種類を指定してください。
     *
     * @details 二度呼び出した場合、値は上書きされます。
     */
    void SetBootDisplayKind(const WebBootDisplayKind bootDisplayKind) NN_NOEXCEPT;

    /**
     * @brief スクリーンショット撮影を許可するかどうかを設定します。
     *
     * @param[in]   screenShotEnabled   撮影を許可する場合は true を指定してください。
     *
     * @details true を指定した場合はスクリーンショット撮影が許可されます。
     *          デフォルト値は false です。
     *          二度呼び出した場合、値は上書きされます。
     */
    void SetScreenShotEnabled(const bool screenShotEnabled) NN_NOEXCEPT;

    /**
     * @internal
     *
     * @brief   ニュースページを開く際に必要なデータを一つのストレージにまとめ、ライブラリアプレットの入力チャンネルにプッシュします。
     *
     * @param[in]   libraryAppletHandle     対象のライブラリアプレットのハンドルを指定します。
     */
    void PushToInChannel(nn::applet::LibraryAppletHandle handle) const NN_NOEXCEPT;

protected:
    /**
    * @brief 情報を格納するために使用するデータのサイズ(バイト)です。
    */
    static const size_t DataSize = 8192;

    NN_ALIGNAS(4) uint8_t m_Data[DataSize];

private:
    /**
     * @internal
     */
    ShowNewsPageArg() NN_NOEXCEPT;
};

//--------------------------------------------------------------------------
/**
 * @brief ウェブページを開くための情報を設定する為のクラスです。
 */
class ShowWebPageArgPrivate
    : protected ShowWebPageArg
{
public:
    /**
     * @brief スクリーンショット撮影を許可するかどうかを設定します。
     *
     * @param[in]   screenShotEnabled   撮影を許可する場合 true を指定してください。
     *
     * @details true を指定した場合はスクリーンショット撮影が許可されます。
     *          デフォルト値は false です。
     *          二度呼び出した場合、値は上書きされます。
     */
    void SetScreenShotEnabled(const bool screenShotEnabled) NN_NOEXCEPT;
};

}} // namespace nn::web
