﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief ログインぺージを開くのに利用される引数の宣言(非公開)
 */

#pragma once

#include <nn/applet/applet.h>
#include <nn/account/account_Types.h>
#include <nn/web/web_LoginTypesPrivate.h>

namespace nn { namespace web {

/**
 * @brief ログインページを表示するための情報を構築するクラスです。
 */
class ShowLoginPageArg
{
public:
    /**
     * @brief NULL 終端を含めたリクエストできる URL の最大バイト数です。
     */
    static const size_t RequestUrlMaxLength = 3072;

    /**
     * @brief NULL 終端を含めたコールバックとして指定できる URL の最大バイト数です。
     */
    static const size_t CallbackUrlMaxLength = 1024;

    /**
     * @brief NULL 終端を含めたコールバック可能 URL として指定できる URL の最大バイト数です。
     */
    static const size_t CallbackableUrlMaxLength = 1024;

    /**
     * @brief ログインページを表示する為の最低限の情報を持つクラスを作成します。
     */
    explicit ShowLoginPageArg() NN_NOEXCEPT;

    /**
     * @brief リクエスト URL を設定します。
     *
     * @param[in]   pRequestUrl 開きたいページの URL を指定してください。
     *
     * @details 指定する値は NULL 終端されている必要があります。
     *          RequestUrlMaxLength -1 文字以下にする必要があります。
     *          二度呼び出した場合、値は上書きされます。
     *
     * @note 本 API は SSPGBOX-84 を受け追加されました。
     *       今後変更、削除される可能性があります。
     */
    void SetRequestUrl(const char* pRequestUrl) NN_NOEXCEPT;

    /**
     * @brief コールバック URL を設定します。
     *
     * @param[in]   pCallbackUrl コールバック URL を指定してください。
     *
     * @details 指定する値は NULL 終端されている必要があります。
     *          CallbackUrlMaxLength -1 文字以下にする必要があります。
     *          二度呼び出した場合、値は上書きされます。
     */
    void SetCallbackUrl(const char* pCallbackUrl) NN_NOEXCEPT;

    /**
     * @brief コールバック可能 URL を設定します。
     *
     * @param[in]   pCallbackableUrl コールバック可能 URL を指定してください。
     *
     * @details 指定する値は NULL 終端されている必要があります。
     *          CallbackableUrlMaxLength -1 文字以下にする必要があります。
     *          二度呼び出した場合、値は上書きされます。
     */
    void SetCallbackableUrl(const char* pCallbackableUrl) NN_NOEXCEPT;

    /**
     * @brief Uid を設定します。
     *
     * @param[in] uid   アカウントの Uid を指定してください。
     *
     * @details 指定する値は、アカウントシステムに登録されているユーザーの Uid または
     *          nn::account::InvalidUid である必要があります。
     *          呼び出さなかった場合または nn::account::InvalidUid を指定した場合は
     *          アノニマスモードになります。
     *          二度呼び出した場合、値は上書きされます。
     */
    void SetUid(const nn::account::Uid& uid) NN_NOEXCEPT;

    /**
     * @brief Cookie と LocalStorage の初期化を行うか設定します。
     *
     * @param[in] clearCookieAndLocalStorage Cookie と LocalStorage の初期化を行う場合は true を指定してください。
     *
     * @details 二度呼び出した場合、値は上書きされます。
     */
    void SetClearCookieAndLocalStorage(const bool clearCookieAndLocalStorage) NN_NOEXCEPT;

    /**
     * @brief NA 登録サイト (acccounts.nintendo.com) 以外の Cookie と LocalStorage の初期化を行うか設定します。
     *
     * @param[in] clearCookieAndLocalStorageExceptNintendoAccount Cookie と LocalStorage の初期化を行う場合は true を指定してください。
     *
     * @details 二度呼び出した場合、値は上書きされます。
     *          有効にした場合、SetClearCookieAndLocalStorage() の動作よりも優先して動作します。
     */
    void SetClearCookieAndLocalStorageExceptNintendoAccount(const bool clearCookieAndLocalStorageExceptNintendoAccount) NN_NOEXCEPT;

    /**
     * @brief 左スティックモードをポインターに切り替える事を許可するかどうかを設定します。
     *
     * @param[in]   pointerEnabled  左スティックモードをポインターに切り替える事を許可する場合は true を指定してください。
     *
     * @details デフォルト値は true です。
     *          二度呼び出した場合、値は上書きされます。
     */
    void SetPointerEnabled(const bool pointerEnabled) NN_NOEXCEPT;

    /**
     * @brief 右スティック入力による空間ナビ操作を有効にするかどうかを設定します。
     *
     * @param[in] enabled   有効にしたい場合に true を指定してください。
     *
     * @details 有効にした場合、拡大縮小ができないページにおいて右スティックにより空間ナビ操作を行えるようになります。
     *          拡大縮小が可能なページでは、右スティックは常に拡大縮小操作になります。
     */
    void SetRightStickSpatialNavigationEnabled(const bool enabled) NN_NOEXCEPT;

    /**
     * @internal
     *
     * @brief ログインページを開く際に必要なデータを一つのストレージにまとめ、ライブラリアプレットの入力チャンネルにプッシュします。
     *
     * @param[in]   libraryAppletHandle 対象のライブラリアプレットのハンドルを指定します。
     */
    void PushToInChannel(nn::applet::LibraryAppletHandle handle) const NN_NOEXCEPT;

    /**
     * @internal
     *
     * @brief   対象とするフッターボタンが起動時に表示されるかどうかを設定します。
     *
     * @param[in]   buttonId     対象のフッターボタンの ID を指定します。
     * @param[in]   visible      対象のフッターボタンを起動時に表示するかどうかを指定します。
     *
     * @details  デフォルト値は A ボタン、 B ボタン、 X ボタン、L スティック押込み、 +・- ボタンが true で、それ以外は false です。
     *           同じフッターボタンに対して二回呼び出した場合、そのフッターボタンに対する設定は上書きされます。
     */
    void SetBootFooterButtonVisible(const LoginFooterButtonId buttonId, const bool visible) NN_NOEXCEPT;

protected:
    /**
     * @internal
     * @brief 情報を格納するために使用するデータのサイズ(バイト)です。
     */
    static const size_t DataSize = 8192;

    /**
     * @internal
     */
    NN_ALIGNAS(4) uint8_t m_Data[DataSize];
};

//--------------------------------------------------------------------------
/**
 * @brief ログインページを開いた結果の情報を提供するクラスです。
 */
class LoginPageReturnValue
{
public:
    /**
    * @brief NULL 終端を含む最後に開いていたページの URL の最大バイト数
    */
    static const size_t LastUrlMaxLength = 4096;

    /**
     * @brief 有効な情報を持たない状態でクラスを生成します。
     */
    LoginPageReturnValue() NN_NOEXCEPT;

    /**
     * @brief 終了理由を返します。
     *
     * @return 終了理由を表す定数(@ref LoginExitReason)
     */
    LoginExitReason GetLoginExitReason() const NN_NOEXCEPT;

    /**
     * @brief 最後に開いていたページの URL を取得します。
     *
     * @return ページの URL を返します。
     *
     * @pre
     *     GetLoginExitReason() == LoginExitReason_CallbackUrlReached
     */
    const char* GetLastUrl() const NN_NOEXCEPT;

    /**
     * @brief 最後に開いていたページの URL のバイト数を取得します。
     *
     * @return URL のバイト数を返します。
     *
     * @pre
     *     GetLoginExitReason() == LoginExitReason_CallbackUrlReached
     */
    size_t GetLastUrlSize() const NN_NOEXCEPT;

protected:
    /**
     * @internal
     */
    uint64_t m_ExitReason;
    /**
     * @internal
     */
    char m_LastUrl[LastUrlMaxLength];
    /**
     * @internal
     */
    uint64_t m_LastUrlSize;
};

}} // namespace nn::web
